<?php
namespace App;

use Mpdf\Mpdf;
use PDO;

class ReceiptPdf
{
    private PDO $pdo;

    // Fixed VAT number as requested
    private string $vatNumber = "238 7878 47";

    public function __construct(PDO $pdo) { $this->pdo = $pdo; }

    /**
     * Browser preview HTML (58mm layout)
     */
    public function getMonthlyReceiptHtml(int $reportId): string
    {
        $report = $this->getReport($reportId);
        if (!$report) { return "<p>Report not found</p>"; }

        // Main departments (exclude paypoint/lottery/scratchcard)
        $depts = $this->getDepartments($reportId);

        // Separate category section (only paypoint/lottery/scratchcard)
        $extraDepts = $this->getPaypointLotteryScratchcard($reportId);

        // VAT summary: paypoint/lottery/scratchcard are counted under Exempt
        $vatLines = $this->getVatSummary($reportId);

        $deptGross = 0.0;
        foreach ($depts as $d) { $deptGross += (float)$d['gross']; }

        $extraGross = 0.0;
        foreach ($extraDepts as $d) { $extraGross += (float)$d['gross']; }

        $totVat = 0.0;
        $totNet = 0.0;
        foreach ($vatLines as $v) {
            $totVat += (float)$v['vat'];
            $totNet += (float)$v['net'];
        }

        return $this->renderHtml(
            $report,
            $depts,
            $extraDepts,
            $vatLines,
            $deptGross,   // Departmental Total (EXCLUDES paypoint/lottery/scratchcard)
            $extraGross,  // Category subtotal (ONLY paypoint/lottery/scratchcard)
            $totVat,
            $totNet,
            true
        );
    }

    /**
     * Download / open 58mm receipt PDF
     */
    public function outputMonthlyReceipt(int $reportId): void
    {
        $report = $this->getReport($reportId);
        if (!$report) { http_response_code(404); echo "Report not found"; return; }

        $depts      = $this->getDepartments($reportId);
        $extraDepts = $this->getPaypointLotteryScratchcard($reportId);
        $vatLines   = $this->getVatSummary($reportId);

        $deptGross = 0.0;
        foreach ($depts as $d) { $deptGross += (float)$d['gross']; }

        $extraGross = 0.0;
        foreach ($extraDepts as $d) { $extraGross += (float)$d['gross']; }

        $totVat = 0.0;
        $totNet = 0.0;
        foreach ($vatLines as $v) {
            $totVat += (float)$v['vat'];
            $totNet += (float)$v['net'];
        }

        // Receipt height estimate (includes footer + small paper feed for cutter)
        $lines =
            26
            + count($depts)
            + (count($extraDepts) ? (count($extraDepts) + 4) : 0) // category + subtotal
            + max(4, count($vatLines))
            + 10; // thank you + cut line + feed

        $height = max(160, 5.1 * $lines);

        $mpdf = new Mpdf([
            'mode' => 'utf-8',
            'format' => [58, $height],
            'margin_left' => 2,
            'margin_right' => 2,
            'margin_top' => 2,
            'margin_bottom' => 1,
        ]);

        $html = $this->renderHtml(
            $report,
            $depts,
            $extraDepts,
            $vatLines,
            $deptGross,
            $extraGross,
            $totVat,
            $totNet,
            false
        );

        $mpdf->WriteHTML($html);

        $fileName = preg_replace('/[^A-Za-z0-9\-_]+/', '_', $report['store_name'])
            . '_' . sprintf('%04d-%02d', (int)$report['year'], (int)$report['month'])
            . '_58mm.pdf';

        header('Content-Type: application/pdf');
        header('Content-Disposition: inline; filename="'.$fileName.'"');
        $mpdf->Output($fileName, 'I');
    }

    private function money(float $amount): string
    {
        return '£' . number_format($amount, 2);
    }

    private function rateLabel($rate): string
    {
        if ($rate === null) return 'Exempt';
        if (is_string($rate) && strtolower(trim($rate)) === 'exempt') return 'Exempt';

        $r = (float)$rate;
        if ($r < 0) return 'Exempt';
        if ($r >= 1.0) return rtrim(rtrim(number_format($r, 2), '0'), '.') . '%';
        return rtrim(rtrim(number_format($r * 100, 2), '0'), '.') . '%';
    }

    private function renderHtml(
        array $report,
        array $depts,
        array $extraDepts,
        array $vatLines,
        float $deptGross,
        float $extraGross,
        float $totVat,
        float $totNet,
        bool $forBrowser
    ): string {
        $monthName = date('M', mktime(0,0,0,(int)$report['month'],1,(int)$report['year']));
        $title   = htmlspecialchars($report['store_name']);
        $address = trim((string)($report['store_address'] ?? ''));
        $period  = htmlspecialchars($monthName.' '.$report['year']);

        $year  = (int)$report['year'];
        $month = (int)$report['month'];
        $first = new \DateTimeImmutable(sprintf('%04d-%02d-01', $year, $month));
        $last  = $first->modify('last day of this month');

        $firstTxn = $first->format('d-m-Y') . ' 00:00:00';
        $lastTxn  = $last->format('d-m-Y') . ' 23:59:59';

        ob_start(); ?>
<!doctype html>
<html>
<head>
<meta charset="utf-8">
<style>
  body { font-family: monospace; font-size: 9px; line-height: 1.18; margin: 0; }
  .c { text-align: center; }
  .r { text-align: right; }
  .row { display: flex; justify-content: space-between; gap: 6px; }
  .sep { border-top: 1px dashed #000; margin: 6px 0; }
  .muted { color: #333; }
  table { width: 100%; border-collapse: collapse; }
  td { padding: 1px 0; vertical-align: top; }
  .w70 { width: 70%; }
  .w30 { width: 30%; }
  .w33 { width: 33.33%; }

  /* Smaller VAT summary */
  .vat { font-size: 8px; line-height: 1.10; }
  .vat td { padding: 0; }

  .feed { height: 10mm; }

  <?php if ($forBrowser): ?>
  .page { width: 58mm; margin: 0 auto; padding: 2mm; background: #fff; color: #000; }
  @media print {
    body { background: #fff; }
    .no-print { display: none !important; }
    .page { width: 58mm; margin: 0; padding: 0; }
    @page { size: 58mm auto; margin: 0; }
  }
  <?php endif; ?>
</style>
</head>
<body>
<?php if ($forBrowser): ?><div class="page"><?php endif; ?>

  <div class="c">
    <div style="font-weight:bold; font-size:11px;"><?= $title ?></div>

    <?php if ($address !== ''): ?>
      <div class="muted" style="margin-top:2px; font-size:8px; line-height:1.15;">
        <?= nl2br(htmlspecialchars($address)) ?>
      </div>
    <?php endif; ?>

    <div class="muted" style="margin-top:2px; font-size:8px;">VAT No: <?= htmlspecialchars($this->vatNumber) ?></div>

    <div style="margin-top:3px;">Monthly Sales Report</div>
    <div style="margin-top:1px;" class="muted"><?= $period ?></div>
  </div>

  <div class="sep"></div>

  <div class="row"><div>First Txn: <?= htmlspecialchars($firstTxn) ?></div><div></div></div>
  <div class="row"><div>Last Txn: <?= htmlspecialchars($lastTxn) ?></div><div></div></div>

  <div class="sep"></div>

  <div class="row"><div>Department Sales</div><div class="muted">Total</div></div>

  <table>
    <?php foreach ($depts as $d): ?>
      <tr>
        <td class="w70"><?= htmlspecialchars($d['department']) ?></td>
        <td class="w30 r"><?= htmlspecialchars($this->money((float)$d['gross'])) ?></td>
      </tr>
    <?php endforeach; ?>
  </table>

  <div class="sep"></div>

  <!-- Departmental total EXCLUDES PayPoint/Lottery/Scratchcard -->
  <table>
    <tr><td class="w70">Sub Total</td><td class="w30 r"><?= htmlspecialchars($this->money($deptGross)) ?></td></tr>
    <tr><td class="w70">Departmental Total</td><td class="w30 r"><?= htmlspecialchars($this->money($deptGross)) ?></td></tr>
  </table>

  <!-- Divider UNDER Departmental Total to separate the category section -->
  <?php if (!empty($extraDepts)): ?>
    <div class="sep"></div>

    <div style="font-weight:bold;">Category</div>

    <table>
      <?php foreach ($extraDepts as $d): ?>
        <tr>
          <td class="w70"><?= htmlspecialchars($d['department']) ?></td>
          <td class="w30 r"><?= htmlspecialchars($this->money((float)$d['gross'])) ?></td>
        </tr>
      <?php endforeach; ?>

      <tr>
        <td class="w70" style="font-weight:bold; padding-top:3px;">Sub Total</td>
        <td class="w30 r" style="font-weight:bold; padding-top:3px;"><?= htmlspecialchars($this->money($extraGross)) ?></td>
      </tr>
    </table>
  <?php endif; ?>

  <div class="sep"></div>

  <div class="c" style="font-weight:bold;">Vat Summary</div>

  <!-- Removed GROSS column as requested -->
  <table class="vat">
    <tr>
      <td class="w33">Rate%</td>
      <td class="w33 r">Net</td>
      <td class="w33 r">Vat</td>
    </tr>
    <?php foreach ($vatLines as $v): ?>
      <tr>
        <td><?= htmlspecialchars($this->rateLabel($v['vat_rate'])) ?></td>
        <td class="r"><?= htmlspecialchars($this->money((float)$v['net'])) ?></td>
        <td class="r"><?= htmlspecialchars($this->money((float)$v['vat'])) ?></td>
      </tr>
    <?php endforeach; ?>
      <tr>
        <td style="font-weight:bold;">Total</td>
        <td class="r" style="font-weight:bold;"><?= htmlspecialchars($this->money($totNet)) ?></td>
        <td class="r" style="font-weight:bold;"><?= htmlspecialchars($this->money($totVat)) ?></td>
      </tr>
  </table>

  <div class="sep"></div>

  <div class="c" style="font-weight:bold; font-size:10px;">Thank you!</div>
  <div class="c muted" style="margin-top:2px;">-- Cut Here --</div>

  <div class="feed"></div>

<?php if ($forBrowser): ?></div><?php endif; ?>
</body>
</html>
<?php
        return ob_get_clean();
    }

    private function getReport(int $id): ?array
    {
        $stmt = $this->pdo->prepare(
            "SELECT mr.id,
                    mr.year,
                    mr.month,
                    mr.walkins,
                    s.name AS store_name,
                    s.store_address AS store_address
             FROM monthly_reports mr
             JOIN stores s ON s.id = mr.store_id
             WHERE mr.id=?"
        );
        $stmt->execute([$id]);
        $r = $stmt->fetch();
        return $r ?: null;
    }

    /**
     * Main department list (exclude paypoint/lottery/scratchcard)
     */
    private function getDepartments(int $reportId): array
    {
        $stmt = $this->pdo->prepare(
            "SELECT department,
                    COALESCE(gross,0) AS gross,
                    COALESCE(vat,0)   AS vat,
                    COALESCE(net,0)   AS net
             FROM department_sales
             WHERE report_id = ?
               AND LOWER(department) NOT IN ('paypoint','lottery','scratchcard')
             ORDER BY id ASC"
        );
        $stmt->execute([$reportId]);
        return $stmt->fetchAll();
    }

    /**
     * Category section rows (paypoint, lottery, scratchcard)
     */
    private function getPaypointLotteryScratchcard(int $reportId): array
    {
        $stmt = $this->pdo->prepare(
            "SELECT department,
                    COALESCE(gross,0) AS gross,
                    COALESCE(vat,0)   AS vat,
                    COALESCE(net,0)   AS net
             FROM department_sales
             WHERE report_id = ?
               AND LOWER(department) IN ('paypoint','lottery','scratchcard')
             ORDER BY
               CASE LOWER(department)
                 WHEN 'paypoint' THEN 1
                 WHEN 'lottery' THEN 2
                 WHEN 'scratchcard' THEN 3
                 ELSE 4
               END"
        );
        $stmt->execute([$reportId]);
        return $stmt->fetchAll();
    }

    /**
     * VAT Summary:
     * - paypoint/lottery/scratchcard must come under Exempt (vat_rate = -1)
     * - VAT missing -> 0
     */
    private function getVatSummary(int $reportId): array
    {
        $stmt = $this->pdo->prepare(
            "SELECT
                CASE
                  WHEN LOWER(department) IN ('paypoint','lottery','scratchcard') THEN -1
                  ELSE CAST(COALESCE(vat_rate, -1) AS DECIMAL(10,4))
                END AS vat_rate,

                SUM(
                  CASE
                    WHEN LOWER(department) IN ('paypoint','lottery','scratchcard')
                      THEN COALESCE(gross,0)
                    ELSE COALESCE(net,0)
                  END
                ) AS net,

                SUM(
                  CASE
                    WHEN LOWER(department) IN ('paypoint','lottery','scratchcard')
                      THEN 0
                    ELSE COALESCE(vat,0)
                  END
                ) AS vat

             FROM department_sales
             WHERE report_id = ?
             GROUP BY
                CASE
                  WHEN LOWER(department) IN ('paypoint','lottery','scratchcard') THEN -1
                  ELSE CAST(COALESCE(vat_rate, -1) AS DECIMAL(10,4))
                END
             ORDER BY vat_rate DESC"
        );

        $stmt->execute([$reportId]);
        return $stmt->fetchAll();
    }
}
