<?php
namespace App;

use PDO;
use PhpOffice\PhpSpreadsheet\IOFactory;

class ExcelImporter
{
    private PDO $pdo;

    private const MONTHS = [
        'Jan'=>1,'Feb'=>2,'Mar'=>3,'Apr'=>4,'May'=>5,'Jun'=>6,
        'Jul'=>7,'Aug'=>8,'Sep'=>9,'Oct'=>10,'Nov'=>11,'Dec'=>12,
    ];

    public function __construct(PDO $pdo)
    {
        $this->pdo = $pdo;
    }

    /**
     * @return array<int, array{store:string, year:int, month:int, report_id:int}>
     */
    public function import(string $xlsxPath, ?string $forcedStoreName=null, ?float $walkins=null): array
    {
        $spreadsheet = IOFactory::load($xlsxPath);
        $results = [];

        foreach ($spreadsheet->getWorksheetIterator() as $sheet) {
            $rows = $sheet->toArray(null, true, true, true); // [rowNum => [A=>..., B=>...]]
            if (count($rows) < 3) continue;

            $headerRow = $rows[1] ?? [];
            $labelRow  = $rows[2] ?? [];

            // Detect month start columns from row 1 values (your file style)
            $monthStarts = []; // colLetter => header text "Store 1 - Tonypandy - Jan 2021"
            foreach ($headerRow as $col => $val) {
                if (!is_string($val)) continue;
                if (preg_match('/\b(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+(\d{4})\b/', $val)) {
                    $monthStarts[$col] = trim($val);
                }
            }

            // If no month columns detected, skip (you can add a second importer for a daily Z template later).
            if (!$monthStarts) continue;

            $monthCols = array_keys($monthStarts);
            // Ensure sorted by column letter
            usort($monthCols, fn($a,$b) => $this->colToIndex($a) <=> $this->colToIndex($b));

            for ($i=0; $i<count($monthCols); $i++) {
                $startCol = $monthCols[$i];
                $endCol = ($i+1 < count($monthCols)) ? $monthCols[$i+1] : $this->maxColLetter($headerRow);

                $meta = $this->parseStoreMonth($monthStarts[$startCol]);
                $storeName = $forcedStoreName ?: $meta['store'];
                $year = $meta['year'];
                $month = $meta['month'];

                $storeId = $this->upsertStore($storeName);
                $reportId = $this->upsertMonthlyReport($storeId, $year, $month, basename($xlsxPath), $walkins);

                // Map metric columns inside the group using label row (row 2)
                $metricCols = $this->metricColumnMap($startCol, $endCol, $labelRow);

                // Wipe existing data for idempotent re-import
                $this->pdo->prepare("DELETE FROM department_sales WHERE report_id=?")->execute([$reportId]);
                $this->pdo->prepare("DELETE FROM vat_summaries WHERE report_id=?")->execute([$reportId]);

                // Data rows start at row 3
                $deptRows = [];
                for ($r=3; $r<=count($rows); $r++) {
                    $row = $rows[$r] ?? null;
                    if (!$row) continue;

                    $dept = trim((string)($row['B'] ?? ''));
                    if ($dept === '' || strtolower($dept) === 'department') continue;

                    // Parse numbers safely:
                    // - If VAT cell is missing/blank, treat VAT as 0.00 (your rule).
                    // - If NET is missing/blank, derive NET = Gross - VAT.
                    $gross = (float)($this->toNumber($row[$metricCols['gross']] ?? 0) ?? 0.0);

                    $vat = 0.0;
                    if ($metricCols['vat']) {
                        $vatParsed = $this->toNumber($row[$metricCols['vat']] ?? null);
                        $vat = ($vatParsed === null) ? 0.0 : (float)$vatParsed;
                    }

                    $net = null;
                    if ($metricCols['net']) {
                        $net = $this->toNumber($row[$metricCols['net']] ?? null);
                    }
                    $net = ($net === null) ? max(0.0, $gross - $vat) : (float)$net;

                    $vatRate = $metricCols['vat_rate'] ? $this->toNumber($row[$metricCols['vat_rate']] ?? null) : null;
                    $catShare = $metricCols['cat_share'] ? $this->toNumber($row[$metricCols['cat_share']] ?? null) : null;

                    // Stop when we reach totals / footer area in some sheets
                    if (preg_match('/^(total|grand|vat summary|category sales)/i', $dept)) break;

                    $deptRows[] = [$dept, $gross, $vat, $net, $vatRate, $catShare];
                }

                $ins = $this->pdo->prepare(
                    "INSERT INTO department_sales (report_id, department, gross, vat, net, vat_rate, category_share)
                     VALUES (?, ?, ?, ?, ?, ?, ?)"
                );
                foreach ($deptRows as [$dept,$gross,$vat,$net,$vatRate,$catShare]) {
                    $ins->execute([$reportId, $dept, $gross, $vat, $net, $vatRate, $catShare]);
                }

                // Build VAT summary grouped by vat_rate
                $vatAgg = [];
                foreach ($deptRows as [$dept,$gross,$vat,$net,$vatRate,$catShare]) {
                    $key = ($vatRate === null || $vatRate === '') ? '0' : (string)$vatRate;
                    if (!isset($vatAgg[$key])) $vatAgg[$key] = ['rate'=>(float)$vatRate, 'net'=>0.0, 'vat'=>0.0, 'total'=>0.0];
                    $vatAgg[$key]['net'] += (float)$net;
                    $vatAgg[$key]['vat'] += (float)$vat;
                    $vatAgg[$key]['total'] += (float)$gross;
                }

                $insVat = $this->pdo->prepare(
                    "INSERT INTO vat_summaries (report_id, vat_rate, net, vat, total) VALUES (?, ?, ?, ?, ?)"
                );
                foreach ($vatAgg as $k => $v) {
                    $rate = $v['rate'] ?? 0.0;
                    $insVat->execute([$reportId, $rate, $v['net'], $v['vat'], $v['total']]);
                }

                $results[] = ['store'=>$storeName,'year'=>$year,'month'=>$month,'report_id'=>$reportId];
            }
        }

        return $results;
    }

    private function parseStoreMonth(string $header): array
    {
        // "Store 1 - Tonypandy - Jan 2021"
        $parts = array_map('trim', explode('-', $header));
        if (count($parts) < 3) {
            return ['store'=>$header, 'month'=>1, 'year'=>(int)date('Y')];
        }
        $monthYear = trim($parts[count($parts)-2]) . ' ' . trim($parts[count($parts)-1]);
        if (preg_match('/\b(Jan|Feb|Mar|Apr|May|Jun|Jul|Aug|Sep|Oct|Nov|Dec)\s+(\d{4})\b/', $monthYear, $m)) {
            $month = self::MONTHS[$m[1]] ?? 1;
            $year = (int)$m[2];
            $store = trim(implode(' - ', array_slice($parts, 0, count($parts)-2)));
            return ['store'=>$store, 'month'=>$month, 'year'=>$year];
        }
        return ['store'=>trim(implode(' - ', array_slice($parts, 0, count($parts)-1))), 'month'=>1, 'year'=>(int)date('Y')];
    }

    /** @return array{gross:string, vat:?string, net:?string, vat_rate:?string, cat_share:?string} */
    private function metricColumnMap(string $startCol, string $endColExclusive, array $labelRow): array
    {
        $cols = [];
        $startIdx = $this->colToIndex($startCol);
        $endIdx = $this->colToIndex($endColExclusive);

        for ($i=$startIdx; $i<$endIdx; $i++) {
            $col = $this->indexToCol($i);
            $label = trim((string)($labelRow[$col] ?? ''));
            $norm = strtolower(preg_replace('/[^a-z0-9% ]/i', '', $label));

            if (str_contains($norm, 'total sale') || str_contains($norm, 'gross')) $cols['gross'] = $col;
            if ($norm === 'vat') $cols['vat'] = $col;
            if (str_contains($norm, 'net')) $cols['net'] = $col;
            if (str_contains($norm, 'vat percentages') || str_contains($norm, 'vat percentage')) $cols['vat_rate'] = $col;
            if (str_contains($norm, 'category sales') || str_contains($norm, 'category sales %')) $cols['cat_share'] = $col;
        }

        // Fallback by position: [gross, vat, net, vat_rate, cat_share]
        $fallback = [];
        for ($i=$startIdx, $j=0; $i<$endIdx; $i++,$j++) $fallback[$j] = $this->indexToCol($i);

        $cols['gross'] ??= $fallback[0] ?? $startCol;
        $cols['vat'] ??= $fallback[1] ?? null;
        $cols['net'] ??= $fallback[2] ?? null;
        $cols['vat_rate'] ??= $fallback[3] ?? null;
        $cols['cat_share'] ??= $fallback[4] ?? null;

        return $cols;
    }

    private function upsertStore(string $name): int
    {
        $name = trim($name);
        $this->pdo->prepare("INSERT IGNORE INTO stores (name) VALUES (?)")->execute([$name]);
        $stmt = $this->pdo->prepare("SELECT id FROM stores WHERE name=?");
        $stmt->execute([$name]);
        return (int)$stmt->fetchColumn();
    }

    private function upsertMonthlyReport(int $storeId, int $year, int $month, ?string $filename, ?float $walkins): int
    {
        $stmt = $this->pdo->prepare(
            "INSERT INTO monthly_reports (store_id, year, month, walkins, source_filename)
             VALUES (?, ?, ?, ?, ?)
             ON DUPLICATE KEY UPDATE walkins=VALUES(walkins), source_filename=VALUES(source_filename), imported_at=CURRENT_TIMESTAMP"
        );
        $stmt->execute([$storeId, $year, $month, (float)($walkins ?? 0), $filename]);

        $stmt2 = $this->pdo->prepare("SELECT id FROM monthly_reports WHERE store_id=? AND year=? AND month=?");
        $stmt2->execute([$storeId, $year, $month]);
        return (int)$stmt2->fetchColumn();
    }

    private function toNumber($val): ?float
    {
        if ($val === null || $val === '') return null;
        if (is_numeric($val)) return (float)$val;
        $s = str_replace([',','£'], '', (string)$val);
        $s = trim($s);
        if ($s === '') return null;
        return is_numeric($s) ? (float)$s : null;
    }

    private function colToIndex(string $col): int
    {
        // A -> 0, B -> 1, Z -> 25, AA -> 26
        $col = strtoupper($col);
        $n = 0;
        for ($i=0; $i<strlen($col); $i++) {
            $n = $n * 26 + (ord($col[$i]) - 64);
        }
        return $n - 1;
    }

    private function indexToCol(int $index): string
    {
        $index += 1;
        $col = '';
        while ($index > 0) {
            $mod = ($index - 1) % 26;
            $col = chr(65 + $mod) . $col;
            $index = intdiv($index - $mod - 1, 26);
        }
        return $col;
    }

    private function maxColLetter(array $row): string
    {
        $keys = array_keys($row);
        usort($keys, fn($a,$b)=>$this->colToIndex($a)<=>$this->colToIndex($b));
        return end($keys) ?: 'Z';
    }
}
