<?php
// partials/navbar.php — auth + base path + profile image (+ RBAC preload)

// ------- Robust session start -------
$headers_already_sent = headers_sent();
$sn = session_name();
if (session_status() !== PHP_SESSION_ACTIVE) {
  if (!$headers_already_sent) { @session_start(); }
  else { @ini_set('session.cache_limiter',''); @session_cache_limiter(''); @session_start(); }
}
$has_session_cookie = isset($_COOKIE[$sn]);
$is_session_active  = (session_status() === PHP_SESSION_ACTIVE);
$is_authed = $is_session_active
  ? (isset($_SESSION['login_type']) && (isset($_SESSION['admin_id']) || isset($_SESSION['user_id'])))
  : $has_session_cookie;

// ------- App base path (supports subfolder installs) -------
function app_base_path(): string {
  $script = str_replace('\\','/', $_SERVER['SCRIPT_NAME'] ?? '/'); // e.g. /ezylend-admin/pages/lend/x.php
  $parts  = array_values(array_filter(explode('/', $script)));
  // if app is in a subfolder like /ezylend-admin, return '/ezylend-admin', else '' (root)
  return (count($parts) >= 2) ? '/'.$parts[0] : '';
}
$APP_BASE  = app_base_path();
$LOGIN_URL = ($APP_BASE ? $APP_BASE : '') . '/pages/login/login.php';

// ------- Enforce auth -------
if (!$is_authed) {
  if (!$headers_already_sent) { header('Location: '.$LOGIN_URL); exit; }
  echo '<script>window.location.href='.json_encode($LOGIN_URL).';</script>';
  echo '<noscript><meta http-equiv="refresh" content="0;url='.$LOGIN_URL.'"></noscript>';
  exit;
}

// ------- RBAC preload (ensures $_SESSION["perms"] ready) -------
require_once dirname(__DIR__) . '../api/rbac.php'; // uses your api/_bootstrap.php & api/_actor.php inside
rbac_ensure_loaded();                        // cheap: loads if missing/identity changed
// echo_perms_js_variable();                 // optional: expose to JS if you want

// ------- Profile image -------
$default_image = ($APP_BASE ?: '') . '/images/faces/face28.jpg';
$img = !empty($_SESSION['staffimage']) ? (string)$_SESSION['staffimage'] : $default_image;
// prefix relative paths with base
if ($img && !preg_match('~^(?:https?://|/)~i', $img)) {
  $img = ($APP_BASE ?: '') . '/' . ltrim($img, '/');
}
$image_to_display = $img;
?>

<style>
/* badge count */
.navbar .count-indicator{ position:relative;  }
.navbar .count-indicator .count{
  min-width:25px;height:50px;line-height:12px;padding:0 5px;font-size:11px;text-align:center;
  border-radius:999px;background:#ef4444;color:#fff;display:none;position:absolute;top:-6px;right:-6px;
}
.navbar .count-indicator .count.show{ display:inline-block; }

/* dropdown list */
.dropdown-menu.navbar-dropdown.preview-list{ width: 480px; max-width:92vw; }
#notifUnifiedList{ max-height: 420px; overflow-y:auto; padding-top:4px; }
#notifUnifiedList::-webkit-scrollbar{ width:10px; }
#notifUnifiedList::-webkit-scrollbar-thumb{ background:#d1d5db;border-radius:8px; }
.preview-item{ cursor:pointer; transition: transform .18s ease, background .18s ease; }
.preview-item:hover{ background:#f8fafc; transform: translateX(2px); }
.preview-thumbnail{ margin-right:10px; }
.preview-icon{
  width:40px;height:40px;border-radius:8px;overflow:hidden;display:grid;place-items:center;background:#eef2f7;
}
.preview-icon img{ width:100%;height:100%;object-fit:cover;display:block; }

/* small badges */
.badge-pill{
  display:inline-block;border:1px solid #e5e7eb;border-radius:999px;padding:2px 8px;font-size:11px;
  color:#374151;background:#fff;
}
.badge-blue{ border-color:#bfdbfe; background:#eff6ff; color:#1e40af; }
.badge-amber{ border-color:#fde68a; background:#fffbeb; color:#92400e; }

/* toast */
#notifToastWrap{
  position:fixed; right:16px; bottom:16px; z-index:3000; display:flex; flex-direction:column; gap:10px;
}
.toast-card{
  width:min(380px, 92vw);
  background:#fff; border:1px solid #e5e7eb; border-radius:12px; padding:12px 14px;
  box-shadow:0 18px 50px rgba(2,6,23,.18), 0 6px 16px rgba(2,6,23,.12);
  display:flex; gap:10px; align-items:flex-start; overflow:hidden;
  animation: slideIn .35s cubic-bezier(.2,.8,.2,1);
}
.toast-card.hide{ animation: slideOut .28s ease forwards; }
.toast-icon{ width:40px;height:40px;border-radius:10px;overflow:hidden;background:#eef2f7;display:grid;place-items:center;flex:0 0 auto; }
.toast-icon img{ width:100%;height:100%;object-fit:cover;display:block; }
.toast-content{ flex:1 1 auto; min-width:0; }
.toast-title{ font-weight:600; margin:0 0 2px 0; }
.toast-sub{ font-size:12px; color:#6b7280; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
.toast-close{ border:none;background:transparent;font-size:18px;line-height:1;cursor:pointer;color:#6b7280; }
@keyframes slideIn{ from{ transform:translateX(30px); opacity:0; } to{ transform:translateX(0); opacity:1; } }
@keyframes slideOut{ to{ transform:translateX(50px); opacity:0; } }

/* detail modal */
#notifDetailModal{ display:none; position:fixed; inset:0; z-index:2000; align-items:center; justify-content:center; background:rgba(15,23,42,.25); padding:16px; }
.nd-card{
  background:#fff;border:1px solid #e5e7eb;border-radius:12px;box-shadow:0 22px 60px rgba(2,6,23,.2);
  max-width:760px;width:calc(100% - 32px);padding:16px 18px;
}
.nd-body{ display:flex; gap:14px; align-items:flex-start; }
.nd-pic{ width:72px; height:72px; border-radius:12px; overflow:hidden; border:1px solid #e5e7eb; flex:0 0 auto; background:#f8fafc; display:grid; place-items:center; cursor:zoom-in; }
.nd-pic img{ width:100%; height:100%; object-fit:cover; display:block; }
.nd-content{ flex:1 1 auto; min-width:0; }

/* image lightbox (medium size) */
#imgLightbox{
  position:fixed; inset:0; z-index:3000; display:none; align-items:center; justify-content:center;
  background:rgba(0,0,0,.45); padding:20px;
}
#imgLightbox .pane{
  background:#0b1220; border-radius:12px; box-shadow:0 28px 70px rgba(0,0,0,.45);
  max-width:min(80vw, 720px); max-height:min(80vh, 520px); padding:10px; position:relative;
  display:grid; place-items:center;
  animation:lbIn .22s cubic-bezier(.2,.8,.2,1);
}
#imgLightbox img{ max-width:100%; max-height:100%; display:block; }
#imgLightbox .close{
  position:absolute; top:6px; right:6px; background:#ffffffdd; border:none; border-radius:999px;
  width:32px; height:32px; display:grid; place-items:center; cursor:pointer; font-size:18px;
}
@keyframes lbIn{ from{ transform:scale(.98); opacity:.35; } to{ transform:scale(1); opacity:1; } }

/* UK clock (glass pill) */
.ukclock-wrap{
  display:flex; align-items:center; gap:10px;
  background:linear-gradient(135deg, rgba(255,255,255,.72), rgba(255,255,255,.55));
  border:1px solid rgba(226,232,240,.8);
  box-shadow:0 8px 20px rgba(2,6,23,.06);
  border-radius:999px; padding:6px 12px;
  backdrop-filter:saturate(1.4) blur(6px);
  -webkit-backdrop-filter:saturate(1.4) blur(6px);
}
.ukclock-icon{ font-size:18px; color:#334155; opacity:.9; }
.ukclock-col{ display:flex; flex-direction:column; line-height:1.1; }
.ukclock-time{ font-weight:700; font-size:14px; color:#0f172a; }
.ukclock-date{ font-size:11px; color:#475569; }
@media (max-width: 991px){ .ukclock-date{ display:none; } }
</style>

<nav class="navbar col-lg-12 col-12 p-0 fixed-top d-flex flex-row">
  <div class="text-center navbar-brand-wrapper d-flex align-items-center justify-content-center">
    <a class="navbar-brand brand-logo mr-5" href="<?php echo htmlspecialchars($APP_BASE); ?>/index.php"><img src="<?php echo htmlspecialchars($APP_BASE); ?>/images/logo.svg" class="mr-2" alt="logo"/></a>
    <a class="navbar-brand brand-logo-mini" href="<?php echo htmlspecialchars($APP_BASE); ?>/index.php"><img src="<?php echo htmlspecialchars($APP_BASE); ?>/images/logo-mini.svg" alt="logo"/></a>
  </div>
  <div class="navbar-menu-wrapper d-flex align-items-center justify-content-end">

    <button class="navbar-toggler navbar-toggler align-self-center" type="button" data-toggle="minimize">
      <span class="icon-menu"></span>
    </button>

    <ul class="navbar-nav navbar-nav-right">

      <!-- UK Clock  -->
      <ul class="navbar-nav mr-auto d-none d-md-flex">
        <li class="nav-item">
          <div class="ukclock-wrap" id="ukClockWrap" title="UK time (Europe/London)">
            <i class="mdi mdi-clock-outline ukclock-icon"></i>
            <div class="ukclock-col">
              <div class="ukclock-time" id="ukClockTime">--:--:--</div>
              <div class="ukclock-date" id="ukClockDate">—</div>
            </div>
          </div>
        </li>
      </ul>

      <!-- Notifications -->
      <li class="nav-item dropdown" id="notifDropdownLi">
        <a class="nav-link count-indicator dropdown-toggle" id="notificationDropdown" href="#" data-toggle="dropdown">
          <i class="icon-bell mx-0"></i>
          <span class="count" id="notifCount"></span>
        </a>
        <div class="dropdown-menu dropdown-menu-right navbar-dropdown preview-list" aria-labelledby="notificationDropdown">
          <div class="d-flex align-items-center justify-content-between px-3 pt-2">
            <p class="mb-0 font-weight-bold">Notifications</p>
            <span class="text-muted small">Latest</span>
          </div>
          <div id="notifUnifiedList"><div class="px-3 pb-2 text-muted" id="notifLoading">Loading…</div></div>
        </div>
      </li>

      <!-- Speaker toggle -->
      <li class="nav-item">
        <a class="nav-link" id="soundToggleNav" href="javascript:void(0)" title="Toggle notification sound">
          <i class="mdi mdi-volume-high" id="soundIcon" style="font-size:18px;"></i>
        </a>
      </li>

      <!-- Profile -->
      <li class="nav-item nav-profile dropdown">
        <a class="nav-link dropdown-toggle" href="#" data-toggle="dropdown" id="profileDropdown">
          <img src="<?php echo htmlspecialchars($image_to_display); ?>" alt="profile"/>
        </a>
        <div class="dropdown-menu dropdown-menu-right navbar-dropdown" aria-labelledby="profileDropdown">
          <a class="dropdown-item"><i class="ti-settings text-primary"></i> Settings</a>
          <a class="dropdown-item" href="<?php echo htmlspecialchars($APP_BASE); ?>/includes/logout.php"><i class="ti-power-off text-primary"></i> Logout</a>
        </div>
      </li>

      <li class="nav-item nav-settings d-none d-lg-flex">
        <a class="nav-link" href="#"><i class="icon-ellipsis"></i></a>
      </li>
    </ul>

    <button class="navbar-toggler navbar-toggler-right d-lg-none align-self-center" type="button" data-toggle="offcanvas">
      <span class="icon-menu"></span>
    </button>
  </div>
</nav>

<!-- Detail modal -->
<div id="notifDetailModal">
  <div class="nd-card">
    <div class="d-flex align-items-center justify-content-between mb-2">
      <h5 class="mb-0" id="ndTitle">Notification</h5>
      <button type="button" id="ndClose" class="btn btn-light btn-sm">&times;</button>
    </div>
    <div class="nd-body">
      <div class="nd-pic" id="ndPic"><img id="ndImg" alt=""/></div>
      <div class="nd-content" id="ndBody"></div>
    </div>
    <div class="mt-2"><small class="text-muted" id="ndStamp"></small></div>
  </div>
</div>

<!-- Image lightbox -->
<div id="imgLightbox">
  <div class="pane">
    <button class="close" id="imgLbClose" title="Close">&times;</button>
    <img id="imgLbImg" alt="">
  </div>
</div>

<!-- Toast holder -->
<div id="notifToastWrap" aria-live="polite" aria-atomic="true"></div>

<!-- Audio: sources are set from DB only  -->
<audio id="soundNotice" preload="auto"></audio>
<audio id="soundNewRequest" preload="auto"></audio>

<script>
(function(){
  if (window.__EZY_NOTIF_INIT__) return;
  window.__EZY_NOTIF_INIT__ = true;

  const APP_BASE = <?php echo json_encode($APP_BASE, JSON_UNESCAPED_SLASHES); ?>;

  // ===== timing / perf =====
  const POLL_MS_VISIBLE = 15000;
  const POLL_MS_HIDDEN  = 45000;
  const FETCH_TIMEOUT   = 8000;
  const MAX_ITEMS       = 30;

  // ===== helpers =====
  function joinBase(path){
    if (!path) return '';
    if (/^(?:https?:)?\/\//i.test(path)) return path;
    if (path.startsWith('/')) return path;
    return APP_BASE.replace(/\/+$/,'') + '/' + path.replace(/^\/+/, '');
  }

  // ----- UK-aware time helpers -----
  function parseDbDate(value){
    if(!value) return null;
    const s = String(value).trim();

    // unix epoch
    if (/^\d+$/.test(s)) {
      const n = Number(s);
      return new Date(n < 1e12 ? n*1000 : n);
    }
    
    if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(s)) {
      const d = new Date(s.replace(' ','T'));
      return isNaN(d.getTime()) ? null : d;
    }
    
    if (/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}$/.test(s)) {
      const d = new Date(s);
      return isNaN(d.getTime()) ? null : d;
    }
    const d = new Date(s);
    return isNaN(d.getTime()) ? null : d;
  }

  const fmtUK = new Intl.DateTimeFormat('en-GB', {
    timeZone: 'Europe/London',
    weekday: 'short', day: '2-digit', month: 'short', year: 'numeric',
    hour: '2-digit', minute: '2-digit', second: '2-digit',
    hour12: false, timeZoneName: 'short'
  });

  function timeAgoUK(value){
    const d = (value instanceof Date) ? value : parseDbDate(value);
    if(!d) return '—';
    const diffSec = Math.floor((Date.now() - d.getTime())/1000);
    if(diffSec < 0) return '—';
    if(diffSec < 5)  return 'just now';
    if(diffSec < 60) return `${diffSec}s ago`;
    const m = Math.floor(diffSec/60);
    if(m < 60) return m===1 ? '1 min ago' : `${m} mins ago`;
    const h = Math.floor(m/60);
    if(h < 24) return h===1 ? '1 hr ago' : `${h} hrs ago`;
    const dsy = Math.floor(h/24);
    if(dsy < 7) return dsy===1 ? '1 day ago' : `${dsy} days ago`;
    const w = Math.floor(dsy/7);
    if(w < 5) return w===1 ? '1 wk ago' : `${w} wks ago`;
    const mo = Math.floor(dsy/30);
    if(mo < 12) return mo===1 ? '1 mo ago' : `${mo} mos ago`;
    const y = Math.floor(dsy/365);
    return y===1 ? '1 yr ago' : `${y} yrs ago`;
  }

  function gbp(n){ const v=Number(n); if(!Number.isFinite(v)) return 'N/A'; return v.toLocaleString('en-GB',{style:'currency',currency:'GBP'}); }
  function fullName(p,f,l){ return [p,f,l].filter(Boolean).join(' ').trim(); }
  function setCount(n){
    const b=document.getElementById('notifCount');
    if(n>0){ b.textContent=n; b.classList.add('show'); }
    else { b.textContent=''; b.classList.remove('show'); }
  }
  function safeImgPath(src, fallback){
    const s = (src && String(src).trim()) ? src : fallback;
    return joinBase(s);
  }

  // ===== UK Clock =====
  const clockTime = document.getElementById('ukClockTime');
  const clockDate = document.getElementById('ukClockDate');
  function pad(n){ return String(n).padStart(2,'0'); }
  function updateUkClock(){
    try{
      const now = new Date();
      const fmt = new Intl.DateTimeFormat('en-GB', {
        timeZone: 'Europe/London',
        hour: '2-digit', minute: '2-digit', second: '2-digit',
        hour12: false, timeZoneName: 'short'
      });
      const parts = fmt.formatToParts(now);
      const map = Object.fromEntries(parts.map(p=>[p.type,p.value]));
      const time = `${map.hour}:${map.minute}:${map.second} ${map.timeZoneName||''}`.trim();

      const d = new Intl.DateTimeFormat('en-GB', {
        timeZone:'Europe/London', weekday:'short', day:'2-digit', month:'short', year:'numeric'
      }).format(now);

      clockTime.textContent = time;  
      clockDate.textContent = d;     
    }catch(e){
      const n = new Date();
      clockTime.textContent = `${pad(n.getHours())}:${pad(n.getMinutes())}:${pad(n.getSeconds())}`;
      clockDate.textContent = n.toDateString();
    }
  }
  updateUkClock();
  setInterval(updateUkClock, 1000);

  // ===== WebAudio beep ) =====
  let audioCtx = null;
  let userInteracted = false;
  function ensureCtx(){
    if (!audioCtx) {
      try { audioCtx = new (window.AudioContext || window.webkitAudioContext)(); }
      catch(e){ audioCtx = null; }
    }
    return audioCtx;
  }
  async function unlockAudio(){
    if (userInteracted) return;
    userInteracted = true;
    const ctx = ensureCtx();
    if (!ctx) return;
    if (ctx.state === 'suspended') { try { await ctx.resume(); } catch(e){} }
  }
  // Global mute flag (saved in localStorage)
  let soundOn = (localStorage.getItem('notif_sound') !== '0');

  function beep(freq=880, durMs=180, gain=0.15){
    if (!soundOn) return; // 
    const ctx = ensureCtx();
    if (!ctx) return;
    const o = ctx.createOscillator();
    const g = ctx.createGain();
    o.type = 'sine'; o.frequency.value = freq;
    g.gain.value = gain;
    o.connect(g); g.connect(ctx.destination);
    const t = ctx.currentTime;
    o.start(t);
    g.gain.setValueAtTime(gain, t);
    g.gain.exponentialRampToValueAtTime(0.0001, t + durMs/1000);
    o.stop(t + durMs/1000 + 0.02);
  }
  function triplet(freq=660){
    if (!soundOn) return;
    beep(freq, 120, 0.16);
    setTimeout(()=>{ if (soundOn) beep(freq,120,0.16); }, 140);
    setTimeout(()=>{ if (soundOn) beep(freq,160,0.16); }, 320);
  }

  // audio elements (DB-loaded)
  const sndNotice     = document.getElementById('soundNotice');
  const sndNewRequest = document.getElementById('soundNewRequest');

  function applyMute(){
    const muted = !soundOn;
    document.querySelectorAll('audio').forEach(a => {
      try { a.muted = muted; } catch(e){}
    });
  }
  function stopAllAudio(){
    document.querySelectorAll('audio').forEach(a => {
      try { a.pause(); a.currentTime = 0; } catch(e){}
    });
    try { if (audioCtx && audioCtx.state !== 'suspended') audioCtx.suspend().catch(()=>{}); } catch(e){}
  }

  // ===== sound toggle UI =====
  const soundIcon = document.getElementById('soundIcon');
  const soundBtn  = document.getElementById('soundToggleNav');
  function updateSoundIcon(){ soundIcon.className = soundOn ? 'mdi mdi-volume-high' : 'mdi mdi-volume-off'; }
  updateSoundIcon();
  applyMute();

  window.addEventListener('click', unlockAudio, {once:true, passive:true});
  window.addEventListener('keydown', unlockAudio, {once:true, passive:true});
  window.addEventListener('touchstart', unlockAudio, {once:true, passive:true});

  soundBtn.addEventListener('click', async ()=>{
    soundOn = !soundOn;
    localStorage.setItem('notif_sound', soundOn ? '1' : '0');
    updateSoundIcon();
    applyMute();

    if (!soundOn) { stopAllAudio(); return; }
    await unlockAudio();
    try { if (audioCtx && audioCtx.state === 'suspended') await audioCtx.resume(); } catch(e){}
    beep(1000, 120, 0.18);
  }, {passive:true});

  async function tryPlay(el){
    try {
      el.currentTime = 0;
      const p = el.play();
      if (p && p.catch) { await p; }
      return true;
    } catch(e){ return false; }
  }
  async function playNotice(){
    if(!soundOn) return;
    if ((sndNotice.dataset.enabled|0) !== 1 || !sndNotice.src) { beep(900, 160, 0.18); return; }
    const ok = await tryPlay(sndNotice);
    if(!ok) beep(900, 160, 0.18);
  }
  async function playNewRequest(){
    if(!soundOn) return;
    if ((sndNewRequest.dataset.enabled|0) !== 1 || !sndNewRequest.src) { triplet(700); return; }
    const ok = await tryPlay(sndNewRequest);
    if(!ok) triplet(700);
  }

  // Pull active sounds 
  (async function loadSoundsFromDB(){
    try{
      const r = await fetch(joinBase('api/notification_sounds.php'), {cache:'no-store'});
      if(!r.ok) return;
      const raw = await r.json();
      const j = (raw && raw.active) ? raw.active : raw;

      // notice
      const N = j && j.notice;
      if (N){
        if (N.src) { sndNotice.src = joinBase(N.src); sndNotice.load(); }
        sndNotice.volume = Number.isFinite(+N.volume) ? Math.max(0, Math.min(1, +N.volume)) : 1.0;
        sndNotice.loop   = !!N.loop;
        sndNotice.dataset.enabled = String(N.enabled ? 1 : 0);
      } else {
        sndNotice.removeAttribute('src');
        sndNotice.dataset.enabled = '0';
      }

      // new_request
      const NR = j && j.new_request;
      if (NR){
        if (NR.src) { sndNewRequest.src = joinBase(NR.src); sndNewRequest.load(); }
        sndNewRequest.volume = Number.isFinite(+NR.volume) ? Math.max(0, Math.min(1, +NR.volume)) : 1.0;
        sndNewRequest.loop   = !!NR.loop;
        sndNewRequest.dataset.enabled = String(NR.enabled ? 1 : 0);
      } else {
        sndNewRequest.removeAttribute('src');
        sndNewRequest.dataset.enabled = '0';
      }

      applyMute();
    }catch(e){}
  })();

  // ===== detail modal + image zoom =====
  const modal   = document.getElementById('notifDetailModal');
  const ndClose = document.getElementById('ndClose');
  const ndBody  = document.getElementById('ndBody');
  const ndTitle = document.getElementById('ndTitle');
  const ndStamp = document.getElementById('ndStamp');
  const ndImg   = document.getElementById('ndImg');
  const ndPic   = document.getElementById('ndPic');

  const lb      = document.getElementById('imgLightbox');
  const lbImg   = document.getElementById('imgLbImg');
  const lbClose = document.getElementById('imgLbClose');

  function openImageLightbox(src){
    lbImg.src = src;
    lb.style.display = 'flex';
  }
  function closeImageLightbox(){
    lb.style.display = 'none';
    lbImg.removeAttribute('src');
  }
  lb.addEventListener('click', (e)=>{ if(e.target === lb) closeImageLightbox(); }, {passive:true});
  lbClose.addEventListener('click', closeImageLightbox, {passive:true});
  document.addEventListener('keydown', (e)=>{ if(e.key==='Escape' && lb.style.display==='flex') closeImageLightbox(); });

  function openDetail(item){
    ndTitle.textContent = item.title || 'Notification';

    const dt = parseDbDate(item.when);
    ndStamp.textContent = dt ? fmtUK.format(dt) : '';

    const imgSrc = safeImgPath(
      item.image_path,
      item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png'
    );
    ndImg.src = imgSrc;
    ndImg.onerror = function(){
      this.onerror = null;
      this.src = (item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png');
    };

    ndPic.onclick = ()=> openImageLightbox(imgSrc);
    ndImg.style.cursor = 'zoom-in';

    if (item.type === 'new_request') {
      ndBody.innerHTML = `
        <div class="table-responsive">
          <table class="table table-sm">
            <thead><tr><th>Client Name</th><th>Amount</th><th>Type</th><th>When</th></tr></thead>
            <tbody>
              <tr style="background:#f7faff">
                <td>${item.client || '—'}</td>
                <td>${item.amount != null ? gbp(item.amount) : '—'}</td>
                <td>${(item.loan_type||'—').toString().toUpperCase()}</td>
                <td>${timeAgoUK(dt)}</td>
              </tr>
            </tbody>
          </table>
        </div>`;
    } else {
      const body = (item.body || '').toString().replace(/\n/g,'<br>');
      ndBody.innerHTML = `
        <div class="mb-2"><span class="badge-pill badge-amber">${item.category || 'Notification'}</span></div>
        <div class="mb-2">${body || '<span class="text-muted">No content</span>'}</div>
        <div class="text-muted small">${timeAgoUK(dt)}</div>`;
    }
    modal.style.display='flex';
  }
  ndClose.addEventListener('click', ()=> modal.style.display='none', {passive:true});
  modal.addEventListener('click', (e)=>{ if(e.target===modal) modal.style.display='none'; }, {passive:true});

  // ===== toasts =====
  const toastWrap = document.getElementById('notifToastWrap');
  function showToast(item){
    const card = document.createElement('div');
    card.className = 'toast-card';
    card.dataset.key = item.key;

    const icon = document.createElement('div');
    icon.className='toast-icon';
    const img = document.createElement('img');
    img.loading = 'lazy';
    img.src = safeImgPath(
      item.image_path,
      item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png'
    );
    img.onerror = function(){
      if (this.dataset.fbk === '1') { this.onerror = null; return; }
      this.dataset.fbk = '1';
      this.src = (item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png');
    };
    icon.appendChild(img);

    const content = document.createElement('div');
    content.className='toast-content';
    const title = document.createElement('div');
    title.className='toast-title';
    title.textContent = item.title || (item.type==='new_request' ? 'New lending request' : 'Notification');
    const sub   = document.createElement('div');
    sub.className='toast-sub';
    const tdt = parseDbDate(item.when);
    sub.textContent = (item.type==='new_request')
      ? `${(item.loan_type||'Loan').toUpperCase()} · ${item.amount!=null ? gbp(item.amount) : ''}`
      : (item.category ? `${item.category} • ${timeAgoUK(tdt)}` : timeAgoUK(tdt));
    content.appendChild(title); content.appendChild(sub);

    const close = document.createElement('button');
    close.className='toast-close';
    close.innerHTML='&times;';
    close.addEventListener('click', ()=> hideToast(card), {passive:true});

    card.appendChild(icon); card.appendChild(content); card.appendChild(close);
    card.addEventListener('click', ()=> openDetail(item), {passive:true});

    // swipe-to-dismiss
    let startX=null;
    card.addEventListener('pointerdown', (e)=>{ startX=e.clientX; }, {passive:true});
    card.addEventListener('pointerup', (e)=>{
      if(startX!=null && (startX - e.clientX) > 60){ hideToast(card); }
      startX=null;
    }, {passive:true});

    toastWrap.appendChild(card);
    const timer = setTimeout(()=> hideToast(card), 5000);
    card.dataset.timer = String(timer);
  }
  function hideToast(card){
    if(!card) return;
    card.classList.add('hide');
    const t = Number(card.dataset.timer||0);
    if(t) clearTimeout(t);
    setTimeout(()=>{ if(card.parentNode) card.parentNode.removeChild(card); }, 260);
  }

  // ===== dropdown DOM =====
  const list = document.getElementById('notifUnifiedList');
  const loadingStub = document.getElementById('notifLoading');
  const state = { nodes: new Map(), items: new Map() };
  let lastSig = '';

  function hashFor(it){
    return JSON.stringify([it.title,it.body,it.image_path,it.category,it.when,it.amount,it.loan_type]);
  }
  function buildNode(item){
    const a = document.createElement('a');
    a.className='dropdown-item preview-item';
    a.dataset.key = item.key;
    a.dataset.hash = hashFor(item);

    const thumb = document.createElement('div');
    thumb.className='preview-thumbnail';
    const icon = document.createElement('div');
    icon.className='preview-icon';
    const img = document.createElement('img');
    img.loading = 'lazy';
    img.src = safeImgPath(
      item.image_path,
      item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png'
    );
    img.onerror = function(){
      if (this.dataset.fbk === '1') { this.onerror = null; return; }
      this.dataset.fbk = '1';
      this.src = (item.type==='new_request' ? APP_BASE+'/images/icons/loan.png' : APP_BASE+'/images/icons/notification.png');
    };
    icon.appendChild(img); thumb.appendChild(icon);

    const right = document.createElement('div');
    right.className='preview-item-content d-flex align-items-start justify-content-between';
    right.style.width='100%';

    const leftCol = document.createElement('div');
    leftCol.style.paddingRight='8px';

    const h6 = document.createElement('h6');
    h6.className='preview-subject font-weight-normal mb-1';
    h6.textContent = item.title || (item.type==='new_request' ? 'New lending request' : 'Notification');

    const pillWrap = document.createElement('div');
    pillWrap.className='small mb-1';
    const pill = document.createElement('span');
    pill.className = 'badge-pill ' + (item.type==='new_request' ? 'badge-blue' : 'badge-amber');
    pill.textContent = item.category || (item.type==='new_request' ? 'New Requests' : 'Notification');
    pillWrap.appendChild(pill);

    const p = document.createElement('p');
    p.className='font-weight-light small-text mb-0 text-muted';
    p.style.maxWidth='300px'; p.style.whiteSpace='nowrap'; p.style.overflow='hidden'; p.style.textOverflow='ellipsis';
    p.textContent = (item.type==='new_request')
      ? `${(item.loan_type||'Loan').toUpperCase()} · ${item.amount!=null ? gbp(item.amount) : ''}`
      : (item.body || '');

    leftCol.appendChild(h6); leftCol.appendChild(pillWrap); leftCol.appendChild(p);

    const stamp = document.createElement('span');
    stamp.className='small-text text-muted ml-2';
    const dt = parseDbDate(item.when);
    stamp.textContent = timeAgoUK(dt);

    right.appendChild(leftCol); right.appendChild(stamp);
    a.appendChild(thumb); a.appendChild(right);
    a.addEventListener('click', ()=> openDetail(item), {passive:true});

    a.style.transform='translateX(10px)'; a.style.opacity='0';
    requestAnimationFrame(()=>{
      a.style.transition='transform .18s ease, opacity .18s ease';
      a.style.transform='translateX(0)'; a.style.opacity='1';
    });

    return a;
  }
  function updateNode(a, item){
    const newHash = hashFor(item);
    if (a.dataset.hash === newHash) return;
    const fresh = buildNode(item);
    a.replaceWith(fresh);
    state.nodes.set(item.key, fresh);
  }

  // ===== fetch with single active controller + timeout + backoff =====
  let consecutiveErrors = 0;
  let activeFetchCtl = null;

  function abortActiveFetch(){
    try { if (activeFetchCtl) activeFetchCtl.abort(); } catch(e){}
    activeFetchCtl = null;
  }

  async function fetchJSON(url){
    abortActiveFetch();
    const ctl = new AbortController();
    activeFetchCtl = ctl;
    const t = setTimeout(()=> ctl.abort(), FETCH_TIMEOUT);
    try{
      const r = await fetch(url, {cache:'no-store', signal: ctl.signal, keepalive:false, credentials:'same-origin'});
      clearTimeout(t);
      if(!r.ok) throw new Error('HTTP '+r.status);
      const j = await r.json();
      consecutiveErrors = 0;
      if (activeFetchCtl === ctl) activeFetchCtl = null;
      return j;
    }catch(e){
      clearTimeout(t);
      if (activeFetchCtl === ctl) activeFetchCtl = null;
      consecutiveErrors++;
      return null;
    }
  }

  async function fetchNewRequests(){
    const arr = await fetchJSON(joinBase('api/new_lending_requests_navbar.php?limit=3')) || [];
    if (!Array.isArray(arr)) return [];
    return arr.map(x=>{
      const titleName = fullName(x.prefix, x.first_name, x.second_name) || x.client_name || 'New lending request';
      return {
        type:'new_request',
        id: Number(x.application_id || 0),
        when: (x.updated_at || x.created_at || x.createdat || null),
        title: titleName,
        client: titleName,
        amount: x.loan_amount != null ? Number(x.loan_amount) : null,
        loan_type: x.loan_type || 'Loan',
        category: 'New Requests',
        image_path: x.image_path,
        body: null
      };
    });
  }

  async function fetchOthers(){
    const data = await fetchJSON(joinBase('api/notifications.php?audience=backend&limit=50')); // API returns ACTIVE only
    const cats = (data && Array.isArray(data.categories)) ? data.categories : [];
    const out = [];
    const sevenDaysAgo = Date.now() - 7*24*3600*1000;

    cats.forEach(cat=>{
      const catName = cat.name || cat.category || 'Notification';
      const items = Array.isArray(cat.items) ? cat.items : [];
      items.forEach(it=>{
        const when = it.updated_at || it.created_at || it.createdat || null;
        const dt = parseDbDate(when);
        if (dt && dt.getTime() < sevenDaysAgo) return;
        const id = Number(it.notification_id || it.id || 0);
        out.push({
          type:'notice',
          id,
          when,
          title: it.title || catName,
          body: it.body || '',
          category: catName,
          image_path: it.image_path
        });
      });
    });
    return out;
  }

  // seen/toast dedupe
  function loadSeen(){ try{ return new Set(JSON.parse(localStorage.getItem('notif_seen_keys')||'[]')); }catch(e){ return new Set(); } }
  function saveSeen(set){ try{ localStorage.setItem('notif_seen_keys', JSON.stringify(Array.from(set).slice(-200))); }catch(e){} }
  const seenKeys = loadSeen();

  // pause polling while dropdown open
  const ddLi = document.getElementById('notifDropdownLi');
  let pollPaused = false;

  document.addEventListener('shown.bs.dropdown', (e)=>{ if (ddLi.contains(e.target)) { pollPaused = true; } }, {passive:true});
  document.addEventListener('hidden.bs.dropdown', (e)=>{ if (ddLi.contains(e.target)) { pollPaused = false; scheduleNext(500); } }, {passive:true});

  const dropdownToggle = document.getElementById('notificationDropdown');
  const obs = new MutationObserver(()=>{ pollPaused = (dropdownToggle.getAttribute('aria-expanded') === 'true'); });
  obs.observe(dropdownToggle, { attributes:true, attributeFilter:['aria-expanded'] });

  // self-scheduling loop (no overlaps)
  let pollTimer = null;
  function scheduleNext(ms){
    if (pollTimer) clearTimeout(pollTimer);
    pollTimer = setTimeout(render, ms);
  }
  window.addEventListener('pagehide', ()=>{ if(pollTimer) clearTimeout(pollTimer); obs.disconnect(); abortActiveFetch(); }, {passive:true});
  document.addEventListener('visibilitychange', ()=>{ scheduleNext(document.hidden ? POLL_MS_HIDDEN : 700); }, {passive:true});

  async function render(){
    if (pollPaused || !navigator.onLine) { return scheduleNext(POLL_MS_VISIBLE); }

    const baseDelay = document.hidden ? POLL_MS_HIDDEN : POLL_MS_VISIBLE;
    const backoff   = Math.min(consecutiveErrors, 6) * 5000;
    const delay     = baseDelay + backoff;

    try{
      const newReq = await fetchNewRequests();
      const others = await fetchOthers();

      const all = [...(newReq||[]), ...(others||[])]
        .filter(x=>x && x.when)
        .sort((a,b)=> (parseDbDate(b.when) - parseDbDate(a.when)))
        .slice(0, MAX_ITEMS);

      all.forEach(it=> it.key = `${it.type}:${it.id}`);

      const sig = JSON.stringify(all.map(it=>[it.key,it.when]));
      if (sig !== lastSig) {
        lastSig = sig;

        const list = document.getElementById('notifUnifiedList');
        const loadingStub = document.getElementById('notifLoading');
        if (loadingStub && loadingStub.parentNode===list) list.removeChild(loadingStub);

        // diff
        const wanted = new Set(all.map(i=>i.key));
        for (const item of all){
          if (!state.nodes.has(item.key)){
            const node = buildNode(item);
            state.nodes.set(item.key, node);
            state.items.set(item.key, item);
          } else {
            updateNode(state.nodes.get(item.key), item);
            state.items.set(item.key, item);
          }
        }
        // remove stale
        for (const [key,node] of Array.from(state.nodes.entries())){
          if(!wanted.has(key)){
            if(node && node.parentNode===list) list.removeChild(node);
            state.nodes.delete(key); state.items.delete(key);
          }
        }
        // reorder with fragment
        const frag = document.createDocumentFragment();
        for (const item of all){
          const node = state.nodes.get(item.key);
          if (node) frag.appendChild(node);
        }
        list.appendChild(frag);

        setCount(all.length);

        // toasts + sounds (max 3 fresh)
        let shown = 0;
        for(const item of all){
          if(shown>=3) continue;
          if(!seenKeys.has(item.key)){
            showToast(item);
            if (item.type === 'notice')       playNotice();
            else if (item.type === 'new_request') playNewRequest();
            seenKeys.add(item.key);
            shown++;
          }
        }
        if(shown>0) saveSeen(seenKeys);
      }
    } catch(e){
      // ignore; backoff handles it
    } finally {
      scheduleNext(delay);
    }
  }

  // kick off shortly after load
  scheduleNext(600);
})();
</script>
