<?php
// notification_sounds_admin.php — UI for managing notification sounds (enhanced uploader + volume meter)
$headers_already_sent = headers_sent();
$sn = session_name();
if (session_status() !== PHP_SESSION_ACTIVE) {
  if (!$headers_already_sent) { @session_start(); }
  else { @ini_set('session.cache_limiter',''); @session_cache_limiter(''); @session_start(); }
}
$authed = (isset($_SESSION['login_type']) && (isset($_SESSION['admin_id']) || isset($_SESSION['user_id'])));
if (!$authed) {
  if (!$headers_already_sent) { header('Location: login.php'); exit; }
  echo '<script>window.location.href="login.php";</script>';
  echo '<noscript><meta http-equiv="refresh" content="0;url=login.php"></noscript>';
  exit;
}

function admin_base(): string {
  $self = str_replace('\\','/', $_SERVER['SCRIPT_NAME'] ?? '/');
  if (preg_match('~^(.*/ezylend-admin)(?:/|$)~i', $self, $m)) return $m[1];
  return rtrim(dirname($self), '/');
}
$ADMIN_BASE = admin_base();
?>
<?php include '../../partials/loadcss.php'; ?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8" />
  <title>EZY LEND — Notification Sounds</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />
  <style>
    .ns-card .help { color:#6b7280; font-size:12px; }
    .badge-pill{ display:inline-block;border:1px solid #e5e7eb;border-radius:999px;padding:2px 8px;font-size:11px; color:#374151;background:#fff; }
    .badge-green { border-color:#bbf7d0; background:#ecfdf5; color:#065f46; }
    .table th, .table td{ vertical-align: middle!important; }
    .w-140{ width:140px; }

    /* ---------- Uploader tile ---------- */
    .dz{
      border:2px dashed #cbd5e1; border-radius:14px; padding:18px; background:#f8fafc;
      display:flex; gap:16px; align-items:center; cursor:pointer; transition:all .18s ease;
    }
    .dz:hover{ background:#f1f5f9; border-color:#94a3b8; }
    .dz.drag{ background:#e0f2fe; border-color:#60a5fa; }
    .dz .icon{
      width:56px; height:56px; border-radius:12px; background:linear-gradient(135deg,#6ee7b7,#60a5fa);
      display:grid; place-items:center; color:white; font-size:26px; flex:0 0 auto;
      box-shadow:0 6px 14px rgba(2,6,23,.12);
    }
    .dz .meta{ flex:1 1 auto; min-width:0; }
    .dz .meta b{ display:block; }
    .dz .meta small{ color:#6b7280; }
    .dz .btns{ display:flex; gap:8px; }
    .dz input[type=file]{ display:none; }

    /* ---------- Volume slider with bars ---------- */
    .vol-wrap{ display:flex; align-items:center; gap:14px; }
    .vol-meter{ display:flex; align-items:flex-end; gap:2px; height:16px; }
    .vol-meter span{
      width:5px; height:4px; border-radius:2px; background:#e5e7eb; transition:height .12s ease, background .12s ease, transform .12s ease;
    }
    .vol-meter span.on{
      background: linear-gradient(180deg,#7c3aed,#22d3ee);
      transform: translateY(-1px);
    }
    .vol-slider{ width:100%; -webkit-appearance:none; height:4px; background:#e5e7eb; border-radius:999px; outline:none; }
    .vol-slider::-webkit-slider-thumb{
      -webkit-appearance:none; appearance:none; width:18px; height:18px; border-radius:50%;
      background:linear-gradient(135deg,#7c3aed,#22d3ee); border:none; box-shadow:0 2px 8px rgba(0,0,0,.18);
      cursor:pointer;
    }
    .vol-slider::-moz-range-thumb{
      width:18px; height:18px; border-radius:50%; background:linear-gradient(135deg,#7c3aed,#22d3ee); border:none; cursor:pointer;
    }

    /* Flash (centered top) */
    #flashWrap{ position:relative; min-height:10px; }
    .xflash{
      position:fixed; left:50%; top:18px; transform:translateX(-50%);
      background:#fff; border:1px solid #e5e7eb; border-radius:12px; padding:10px 14px;
      box-shadow:0 18px 50px rgba(2,6,23,.18), 0 6px 16px rgba(2,6,23,.12);
      z-index:1200; animation:slideDown .25s ease;
    }
    .xflash.success{ color:#065f46; background:#ecfdf5; border-color:#bbf7d0; }
    .xflash.error{ color:#991b1b; background:#fff1f2; border-color:#fecaca; }
    @keyframes slideDown{ from{ opacity:0; transform:translate(-50%,-6px);} to{ opacity:1; transform:translate(-50%,0);} }
  </style>
</head>
<body>
  <div class="container-scroller">
    <?php include '../../partials/navbar.php'; ?>
    <div class="container-fluid page-body-wrapper">
      <?php include '../../partials/settings.php'; ?>
      <?php include '../../partials/sidebar.php'; ?>

      <div class="main-panel">
        <div class="content-wrapper">
          <div class="row">
            <div class="col-md-12 grid-margin stretch-card">
              <div class="card ns-card">
                <div class="card-body">
                  <div class="d-flex align-items-center justify-content-between">
                    <h4 class="card-title mb-0">Notification Sounds</h4>
                    <span class="help">Only <b>one</b> sound can be active per type.</span>
                  </div>

                  <div id="flashWrap"></div>

                  <!-- Upload form -->
                  <form id="uploadForm" class="mt-3" enctype="multipart/form-data">
                    <div class="form-row">
                      <div class="form-group col-md-3">
                        <label>Type</label>
                        <select class="form-control" name="type" required>
                          <option value="notice">General notifications</option>
                          <option value="new_request">New lending requests</option>
                        </select>
                      </div>

                      <div class="form-group col-md-3">
                        <label>Label</label>
                        <input type="text" class="form-control" name="label" placeholder="e.g. Soft ping" required>
                      </div>

                      <div class="form-group col-md-3">
                        <label>Volume</label>
                        <div class="vol-wrap">
                          <div class="vol-meter" id="volBars" aria-hidden="true"></div>
                          <input type="range" min="0" max="1" step="0.01" value="1" class="vol-slider" name="volume" id="volumeRange" />
                          <span id="volPct" class="text-muted" style="min-width:50px; text-align:right;">100%</span>
                        </div>
                      </div>
                      <div class="form-group col-md-1">
                            </div>
                      <div class="form-group col-md-2">
                        <label>Options</label>
                        <div class="form-check mb-2">
                          <input class="form-check-input" type="checkbox" id="loop" name="loop">
                          <label class="form-check-label" for="loop">Loop</label>
                        </div>
                        <div class="form-check">
                          <input class="form-check-input" type="checkbox" id="make_active" name="make_active">
                          <label class="form-check-label" for="make_active">Make active after upload</label>
                        </div>
                      </div>
                    </div>

                    <div class="form-row">
                      <div class="form-group col-md-4">
                        <label>Sound file (MP3 / OGG, ≤ 10 MB)</label>
                        <div class="dz" id="dropzone">
                          <div class="icon"><i class="mdi mdi-volume-high"></i></div>
                          <div class="meta">
                            <b id="dzTitle">Drag & drop audio here, or click to choose</b>
                            <small id="dzSub">Accepted: .mp3 / .ogg</small>
                          </div>
                          <div class="btns">
                            <button type="button" class="btn btn-sm btn-outline-primary" id="dzChoose">Choose</button>
                            <button type="button" class="btn btn-sm btn-light d-none" id="dzClear">Clear</button>
                          </div>
                          <input type="file" id="soundFile" name="sound_file" accept=".mp3, audio/mpeg, .ogg, audio/ogg">
                        </div>

                        <div id="previewWrap" class="mt-2" style="display:none;">
                          <audio id="previewAudio" controls preload="metadata" style="width:100%"></audio>
                          <div class="small text-muted mt-1" id="previewMeta"></div>
                        </div>
                      </div>
                        
                    </div>

                    <div class="text-right">
                      <button type="submit" class="btn btn-primary" id="btnUpload">Upload</button>
                    </div>
                  </form>

                  <hr class="mt-4 mb-3">

                  <div class="row">
                    <div class="col-md-6">
                      <h5 class="mb-2">General notifications (type: notice)</h5>
                      <div class="table-responsive">
                        <table class="table table-bordered" id="tblNotice">
                          <thead>
                            <tr>
                              <th class="w-140">Status</th>
                              <th>Label</th>
                              <th>Preview</th>
                              <th class="w-140">Actions</th>
                            </tr>
                          </thead>
                          <tbody></tbody>
                        </table>
                      </div>
                    </div>

                    <div class="col-md-6">
                      <h5 class="mb-2">New lending requests (type: new_request)</h5>
                      <div class="table-responsive">
                        <table class="table table-bordered" id="tblNewReq">
                          <thead>
                            <tr>
                              <th class="w-140">Status</th>
                              <th>Label</th>
                              <th>Preview</th>
                              <th class="w-140">Actions</th>
                            </tr>
                          </thead>
                          <tbody></tbody>
                        </table>
                      </div>
                    </div>
                  </div>

                  
                </div>
              </div>
            </div>
          </div>

          <?php include '../../partials/footer.php'; ?>
        </div>
      </div>
    </div>
  </div>

  <?php include '../../partials/loadjs.php'; ?>
  <script>
    const ADMIN_BASE = <?php echo json_encode($ADMIN_BASE, JSON_UNESCAPED_SLASHES); ?>;

    // ---------- Flash ----------
    function flash(type, text){
      const el = document.createElement('div');
      el.className = 'xflash ' + (type==='success'?'success':'error');
      el.textContent = text;
      document.body.appendChild(el);
      setTimeout(()=> el.remove(), 2600);
    }

    // ---------- Volume bars ----------
    const volBars = document.getElementById('volBars');
    const volRange= document.getElementById('volumeRange');
    const volPct  = document.getElementById('volPct');
    const BAR_COUNT = 12;

    function drawBars(){
      volBars.innerHTML = '';
      for(let i=0;i<BAR_COUNT;i++){
        const s = document.createElement('span');
        volBars.appendChild(s);
      }
      updateBars();
    }
    function updateBars(){
      const v = parseFloat(volRange.value||'0'); // 0..1
      volPct.textContent = Math.round(v*100) + '%';
      const onCount = Math.round(v * BAR_COUNT);
      [...volBars.children].forEach((bar, i)=>{
        bar.classList.toggle('on', i < onCount);
        const h = 4 + Math.round((i+1)/BAR_COUNT * 12); // 4..16px
        bar.style.height = (i < onCount ? h : 4) + 'px';
      });
    }
    volRange.addEventListener('input', updateBars);
    drawBars();

    // ---------- Uploader tile ----------
    const dz = document.getElementById('dropzone');
    const dzChoose = document.getElementById('dzChoose');
    const dzClear  = document.getElementById('dzClear');
    const dzTitle  = document.getElementById('dzTitle');
    const dzSub    = document.getElementById('dzSub');
    const fileInput= document.getElementById('soundFile');
    const previewWrap = document.getElementById('previewWrap');
    const previewAudio= document.getElementById('previewAudio');
    const previewMeta = document.getElementById('previewMeta');
    let previewURL = null;

    function fmtBytes(b){
      if (b===0) return '0 B';
      const u=['B','KB','MB','GB']; const i=Math.floor(Math.log(b)/Math.log(1024));
      return (b/Math.pow(1024,i)).toFixed(2)+' '+u[i];
    }
    function fmtTime(sec){
      sec = Math.round(sec||0);
      const m = Math.floor(sec/60), s = sec%60;
      return m + ':' + (s<10?'0':'') + s;
    }

    function setPreview(file){
      // validate
      if (!file) { clearFile(); return; }
      const ok = ['audio/mpeg','audio/mp3','audio/ogg','application/ogg','audio/x-ogg'];
      if (!ok.includes(file.type)) { flash('error','Only MP3/OGG allowed'); clearFile(); return; }
      if (file.size > 10*1024*1024) { flash('error','Max 10MB'); clearFile(); return; }

      dz.classList.remove('drag');
      dzClear.classList.remove('d-none');
      dzTitle.textContent = file.name;
      dzSub.textContent = fmtBytes(file.size);

      if (previewURL) URL.revokeObjectURL(previewURL);
      previewURL = URL.createObjectURL(file);
      previewAudio.src = previewURL;
      previewWrap.style.display = '';

      previewAudio.onloadedmetadata = ()=>{
        previewMeta.textContent = 'Duration: ' + fmtTime(previewAudio.duration || 0);
      };
    }
    function clearFile(){
      if (previewURL) { URL.revokeObjectURL(previewURL); previewURL=null; }
      fileInput.value = '';
      previewAudio.removeAttribute('src');
      previewAudio.load();
      previewWrap.style.display = 'none';
      dzClear.classList.add('d-none');
      dzTitle.textContent = 'Drag & drop audio here, or click to choose';
      dzSub.textContent = 'Accepted: .mp3 / .ogg';
    }

    dz.addEventListener('click', (e)=>{ if(e.target!==dzClear) fileInput.click(); });
    dzChoose.addEventListener('click', (e)=>{ e.stopPropagation(); fileInput.click(); });
    dzClear.addEventListener('click', (e)=>{ e.stopPropagation(); clearFile(); });

    fileInput.addEventListener('change', (e)=> setPreview(e.target.files[0]));

    dz.addEventListener('dragover', (e)=>{ e.preventDefault(); dz.classList.add('drag'); });
    dz.addEventListener('dragleave', ()=> dz.classList.remove('drag'));
    dz.addEventListener('drop', (e)=>{
      e.preventDefault(); dz.classList.remove('drag');
      const f = e.dataTransfer.files && e.dataTransfer.files[0];
      if (f) { setPreview(f); }
    });

    // ---------- Table list / actions ----------
    async function listAll(){
      const res = await fetch('api/notification_sounds_list.php', {cache:'no-store'});
      const j = await res.json();
      renderTables(j.items || []);
    }

    function rowHtml(r){
      const badge = r.enabled ? '<span class="badge-pill badge-green">Active</span>' : '<span class="badge-pill">Inactive</span>';
      const src = r.src_abs || (ADMIN_BASE + '/' + (r.src||''));
      return `
        <tr data-id="${r.id}">
          <td>
            ${badge}
            <div class="small text-muted">${r.created_at || ''}</div>
          </td>
          <td>${escapeHtml(r.label || '')}<div class="small text-muted">Vol ${r.volume} • ${Number(r.loop)?'Loop':''}</div></td>
          <td>
            <audio controls preload="none" style="width:100%">
              <source src="${src}" type="${src.toLowerCase().endsWith('.ogg') ? 'audio/ogg' : 'audio/mpeg'}">
            </audio>
          </td>
          <td>
            ${!r.enabled ? `<button class="btn btn-sm btn-success mb-1" onclick="setActive(${r.id})">Set Active</button>` :
                           `<button class="btn btn-sm btn-warning mb-1" onclick="disableSound(${r.id})">Disable</button>`}
            <button class="btn btn-sm btn-outline-danger mb-1" onclick="deleteSound(${r.id})">Delete</button>
          </td>
        </tr>
      `;
    }
    function renderTables(items){
      const noticeTbody  = document.querySelector('#tblNotice tbody');
      const newreqTbody  = document.querySelector('#tblNewReq tbody');
      noticeTbody.innerHTML = '';
      newreqTbody.innerHTML = '';
      const notice = items.filter(x=> x.type==='notice');
      const nreq   = items.filter(x=> x.type==='new_request');

      if (!notice.length) noticeTbody.innerHTML = `<tr><td colspan="4" class="text-center text-muted">No sounds</td></tr>`;
      else notice.forEach(r=> noticeTbody.insertAdjacentHTML('beforeend', rowHtml(r)));

      if (!nreq.length) newreqTbody.innerHTML = `<tr><td colspan="4" class="text-center text-muted">No sounds</td></tr>`;
      else nreq.forEach(r=> newreqTbody.insertAdjacentHTML('beforeend', rowHtml(r)));
    }
    function escapeHtml(s){ return (s||'').replace(/[&<>"]/g, c=>({ '&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;' }[c])); }

    async function setActive(id){
      const fd = new FormData(); fd.append('id', id);
      const res = await fetch('api/notification_sounds_set_active.php', {method:'POST', body: fd});
      const j = await res.json();
      if (j.success) { flash('success', 'Set active'); listAll(); } else { flash('error', j.error||'Failed'); }
    }
    async function disableSound(id){
      const fd = new FormData(); fd.append('id', id);
      const res = await fetch('api/notification_sounds_disable.php', {method:'POST', body: fd});
      const j = await res.json();
      if (j.success) { flash('success', 'Disabled'); listAll(); } else { flash('error', j.error||'Failed'); }
    }
    async function deleteSound(id){
      if (!confirm('Delete this sound?')) return;
      const fd = new FormData(); fd.append('id', id);
      const res = await fetch('api/notification_sounds_delete.php', {method:'POST', body: fd});
      const j = await res.json();
      if (j.success) { flash('success', 'Deleted'); listAll(); } else { flash('error', j.error||'Failed'); }
    }
    window.setActive = setActive;
    window.disableSound = disableSound;
    window.deleteSound = deleteSound;

    // ---------- Upload submit ----------
    document.getElementById('uploadForm').addEventListener('submit', async (e)=>{
      e.preventDefault();
      const file = fileInput.files[0];
      if (!file) { flash('error','Choose a file'); return; }

      const btn = document.getElementById('btnUpload');
      btn.disabled = true;

      try{
        const fd = new FormData(e.currentTarget);
        // force volume to current slider value (string)
        fd.set('volume', String(volRange.value));
        const res = await fetch('api/notification_sounds_upload.php', {method:'POST', body: fd});
        const j = await res.json();
        if (j && j.success){
          flash('success', 'Sound uploaded.');
          e.currentTarget.reset();
          clearFile();
          volRange.value = '1'; updateBars();
          listAll();
        }else{
          flash('error', j?.error || 'Upload failed');
        }
      }catch(_){
        flash('error','Upload failed');
      }finally{
        btn.disabled = false;
      }
    });

    // Initial
    listAll();
  </script>
</body>
</html>
