<?php include '../../partials/loadcss.php'; ?>

<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="utf-8">
  <title>EZY LEND — Dashboard</title>
  <meta name="viewport" content="width=device-width, initial-scale=1" />

  <style>
    :root{ --gap: 1rem; }

    /* ===== Shell ===== */
    .dashboard-container{ display:grid; grid-template-rows:auto auto 1fr; gap:var(--gap); }

    /* ===== Welcome ===== */
    .welcome-wrap{ display:flex; align-items:center; justify-content:space-between; gap:.75rem; flex-wrap:wrap; }

    /* ===== KPI strip ===== */
    .kpi-grid{
      display:grid; gap:var(--gap);
      grid-template-columns: repeat(4, minmax(220px, 1fr));
      align-items:stretch;
    }
    @media (max-width:1199.98px){ .kpi-grid{ grid-template-columns: repeat(2, minmax(220px, 1fr)); } }
    @media (max-width:575.98px){  .kpi-grid{ grid-template-columns: minmax(220px, 1fr); } }

    .kpi-item > .card{ height:100%; }
    .kpi-item .card-body p{ white-space:normal; }

    /* ===== Two-column main area ===== */
    .main-grid{
      display:grid; gap:var(--gap);
      grid-template-columns: 1.2fr .8fr;
      align-items:start;
    }
    @media (max-width:991.98px){ .main-grid{ grid-template-columns: 1fr; } }

    /* ===== New Lending Requests table ===== */
    .nlr-table { table-layout: fixed; width:100%; }
    .nlr-table thead th { white-space:nowrap; }
    .nlr-table td, .nlr-table th { vertical-align: middle; }
    .nlr-wrap { margin-top:.5rem; }
    .col-name{ width:46%; }
    .col-amount{ width:18%; }
    .col-type{ width:18%; }
    .col-date{ width:18%; color:#6b7280; text-align:left; }
    .truncate{ max-width:100%; white-space:nowrap; overflow:hidden; text-overflow:ellipsis; }
    .nlr-table td.col-amount,
    .nlr-table td.col-type,
    .nlr-table td.col-date{ white-space:nowrap; }

    /* Skeleton */
    .skeleton{ position:relative; overflow:hidden; background:#f3f4f6; }
    .skeleton::after{
      content:""; position:absolute; inset:0;
      background:linear-gradient(90deg, rgba(255,255,255,0) 0%, rgba(255,255,255,.45) 50%, rgba(255,255,255,0) 100%);
      transform:translateX(-100%); animation:shimmer 1.1s infinite;
    }
    @keyframes shimmer{ to{ transform:translateX(100%);} }

    .card-body .table{ margin-bottom:0; }
  </style>
</head>

<body>
  <div class="container-scroller">
    <?php include '../../partials/navbar.php'; ?>

    <div class="container-fluid page-body-wrapper">
      <?php include '../../partials/settings.php'; ?>
      <?php include '../../partials/sidebar.php'; ?>

      <div class="main-panel">
        <div class="content-wrapper">
<?php
  // ------- robust display name (fixes “Guest”) -------
  if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

  $cands = [];
  foreach ([
    'admin_firstname','admin_first_name','admin_lastname','admin_last_name','admin_name',
    'user_firstname','user_first_name','user_lastname','user_last_name','user_name',
    'firstname','first_name','lastname','last_name','name','username','staffname','display_name','email'
  ] as $k) { if (!empty($_SESSION[$k])) $cands[$k] = $_SESSION[$k]; }

  $first = $cands['admin_firstname'] ?? $cands['admin_first_name'] ?? $cands['user_firstname'] ?? $cands['user_first_name'] ?? $cands['firstname'] ?? $cands['first_name'] ?? null;
  $last  = $cands['admin_lastname']  ?? $cands['admin_last_name']  ?? $cands['user_lastname']  ?? $cands['user_last_name']  ?? $cands['lastname']  ?? $cands['last_name']  ?? null;

  if (!$first && !$last) {
    $display = $cands['admin_name'] ?? $cands['user_name'] ?? $cands['name'] ?? $cands['username'] ?? $cands['staffname'] ?? null;
    if (!$display && !empty($cands['email'])) { $display = explode('@', $cands['email'])[0]; }
  } else {
    $display = trim(($first ?? '').' '.($last ?? ''));
  }
  if (!$display) $display = 'Guest';
?>
          <div class="dashboard-container">

            <!-- Welcome -->
            <div class="welcome-wrap">
              <h3 class="font-weight-bold mb-0">Welcome, <?php echo htmlspecialchars($display, ENT_QUOTES, 'UTF-8'); ?></h3>
            </div>

            <!-- KPI strip -->
            <div class="kpi-grid">
              <div class="kpi-item">
                <div class="card card-tale">
                  <div class="card-body">
                    <p class="mb-2">Today’s Loans</p>
                    <p class="fs-30 mb-1" id="statTodayLoans">—</p>
                    <p class="mb-0 text-muted" id="statTodayLoansNote">updated</p>
                  </div>
                </div>
              </div>

              <div class="kpi-item">
                <div class="card card-dark-blue">
                  <div class="card-body">
                    <p class="mb-2">Total Instant Lends</p>
                    <p class="fs-30 mb-1" id="statInstant">—</p>
                    <p class="mb-0 text-muted" id="statInstantNote">updated</p>
                  </div>
                </div>
              </div>

              <div class="kpi-item">
                <div class="card card-light-blue">
                  <div class="card-body">
                    <p class="mb-2">Total Minutes Lends</p>
                    <p class="fs-30 mb-1" id="statMinute">—</p>
                    <p class="mb-0 text-muted" id="statMinuteNote">updated</p>
                  </div>
                </div>
              </div>

              <div class="kpi-item">
                <div class="card card-light-danger">
                  <div class="card-body">
                    <p class="mb-2">Number of Clients</p>
                    <p class="fs-30 mb-1" id="statClients">—</p>
                    <p class="mb-0 text-muted" id="statClientsNote">updated</p>
                  </div>
                </div>
              </div>
            </div>

            <!-- Main: New Lending Requests (left) + Visits cards (right) -->
            <div class="main-grid">

              <!-- New Lending Requests -->
              <div class="stretch-card">
                <div class="card">
                  <div class="card-body">
                    <div class="d-flex align-items-center justify-content-between">
                      <p class="card-title mb-0">New Lending Requests</p>
                      <button type="button" class="btn btn-sm btn-outline-secondary" id="btnRefreshNLR">Refresh</button>
                    </div>

                    <div class="table-responsive nlr-wrap">
                      <table class="table table-striped table-borderless nlr-table" id="newLendingTable" aria-live="polite">
                        <colgroup>
                          <col class="col-name" />
                          <col class="col-amount" />
                          <col class="col-type" />
                          <col class="col-date" />
                        </colgroup>
                        <thead>
                          <tr>
                            <th class="col-name">Client Name</th>
                            <th class="col-amount text-right">Amount</th>
                            <th class="col-type">Type</th>
                            <th class="col-date">Date</th>
                          </tr>
                        </thead>
                        <tbody>
                          <?php for($i=0;$i<6;$i++): ?>
                          <tr>
                            <td><div class="skeleton" style="height:14px; border-radius:6px;"></div></td>
                            <td><div class="skeleton" style="height:14px; border-radius:6px;"></div></td>
                            <td><div class="skeleton" style="height:14px; border-radius:6px;"></div></td>
                            <td><div class="skeleton" style="height:14px; border-radius:6px;"></div></td>
                          </tr>
                          <?php endfor; ?>
                        </tbody>
                      </table>
                    </div>
                  </div>
                </div>
              </div>

              <!-- Visits (Today & Month) -->
              <div class="d-flex flex-column" style="gap: var(--gap);">
                <div class="stretch-card transparent">
                  <div class="card card-tale">
                    <div class="card-body">
                      <p class="mb-2">Page Visits (Today)</p>
                      <p class="fs-30 mb-1" id="statVisitsToday">—</p>
                      <p class="mb-0 text-muted" id="statVisitsTodayNote">updated</p>
                    </div>
                  </div>
                </div>

                <div class="stretch-card transparent">
                  <div class="card card-dark-blue">
                    <div class="card-body">
                      <p class="mb-2">Page Visits (This Month)</p>
                      <p class="fs-30 mb-1" id="statVisitsMonth">—</p>
                      <p class="mb-0 text-muted" id="statVisitsMonthNote">updated</p>
                    </div>
                  </div>
                </div>
              </div>

            </div><!-- /main-grid -->

          </div><!-- /dashboard-container -->
        </div><!-- /content-wrapper -->

        <?php include '../../partials/footer.php'; ?>
      </div><!-- /main-panel -->
    </div><!-- /page-body-wrapper -->
  </div><!-- /container-scroller -->

  <?php include '../../partials/loadjs.php'; ?>

  <script>
    // Endpoints
    const ENDPOINT_NEW   = 'api/new_lending_requests.php?limit=8';
    const ENDPOINT_STATS = 'api/dashboard_stats.php';

    // Poll cadence
    const POLL_MS_NEW   = 12000;
    const POLL_MS_STATS = 20000;
    const FETCH_TIMEOUT = 8000;

    // ===== Time helpers (UK aware) =====
    // Safely parse common DB/ISO timestamps to a Date
    function parseDbDate(value){
      if(!value) return null;
      const s = String(value).trim();

      // Numeric epoch?
      if (/^\d+$/.test(s)) {
        const n = Number(s);
        return new Date(n < 1e12 ? n*1000 : n);
      }

      // MySQL DATETIME "YYYY-MM-DD HH:MM:SS"
      // IMPORTANT: Do NOT append 'Z' — treat as *local* time on the user's machine.
      if (/^\d{4}-\d{2}-\d{2} \d{2}:\d{2}:\d{2}$/.test(s)) {
        const d = new Date(s.replace(' ','T')); // local time
        return isNaN(d.getTime()) ? null : d;
      }

      // ISO without TZ "YYYY-MM-DDTHH:MM:SS" — also treat as local
      if (/^\d{4}-\d{2}-\d{2}T\d{2}:\d{2}:\d{2}$/.test(s)) {
        const d = new Date(s); // local
        return isNaN(d.getTime()) ? null : d;
      }

      // Fallback
      const d = new Date(s);
      return isNaN(d.getTime()) ? null : d;
    }

    // Tooltip formatter (Europe/London) with auto BST/GMT
    const fmtUK = new Intl.DateTimeFormat('en-GB', {
      timeZone: 'Europe/London',
      weekday: 'short', day: '2-digit', month: 'short', year: 'numeric',
      hour: '2-digit', minute: '2-digit', second: '2-digit',
      hour12: false, timeZoneName: 'short'
    });

    // Relative text (“just now”, “12 mins ago”)
    function timeAgoUK(value){
      const d = (value instanceof Date) ? value : parseDbDate(value);
      if(!d) return '—';
      const diffSec = Math.floor((Date.now() - d.getTime())/1000);
      if(diffSec < 0) return '—';
      if(diffSec < 5)  return 'just now';
      if(diffSec < 60) return `${diffSec}s ago`;
      const m = Math.floor(diffSec/60);
      if(m < 60) return m===1 ? '1 min ago' : `${m} mins ago`;
      const h = Math.floor(m/60);
      if(h < 24) return h===1 ? '1 hr ago' : `${h} hrs ago`;
      const dys = Math.floor(h/24);
      if(dys < 7) return dys===1 ? '1 day ago' : `${dys} days ago`;
      const w = Math.floor(dys/7);
      if(w < 5) return w===1 ? '1 wk ago' : `${w} wks ago`;
      const mo = Math.floor(dys/30);
      if(mo < 12) return mo===1 ? '1 mo ago' : `${mo} mos ago`;
      const y = Math.floor(dys/365);
      return y===1 ? '1 yr ago' : `${y} yrs ago`;
    }

    // General helpers
    function gbp(v){ const n=Number(v); return Number.isFinite(n)?n.toLocaleString('en-GB',{style:'currency',currency:'GBP'}):'—'; }
    function fullName(p,f,l){ return [p,f,l].filter(Boolean).join(' ').replace(/\s+/g,' ').trim(); }
    function esc(s){ return String(s ?? '').replace(/[&<>"']/g, m => ({'&':'&amp;','<':'&lt;','>':'&gt;','"':'&quot;',"'":'&#39;'}[m])); }

    function fetchJSON(url, timeoutMs=FETCH_TIMEOUT){
      const ctl=new AbortController(); const to=setTimeout(()=>ctl.abort(),timeoutMs);
      return fetch(url,{signal:ctl.signal, cache:'no-store'})
        .then(r=>{ clearTimeout(to); if(!r.ok) throw 0; return r.json(); })
        .catch(()=>{ clearTimeout(to); return null; });
    }

    // ===== New Lending Requests =====
    const tbody = document.querySelector('#newLendingTable tbody');
    const btnRefreshNLR = document.getElementById('btnRefreshNLR');
    let nlrTimer=null, nlrBusy=false, prevSig='';

    // Signature uses only created_at
    function signature(rows){ return JSON.stringify(rows.map(r=>[r.application_id, r.created_at||''])); }

    function renderNLR(rows){
      if(!Array.isArray(rows) || rows.length===0){
        tbody.innerHTML = `<tr><td colspan="4" class="text-center text-muted">No recent requests</td></tr>`;
        return;
      }
      tbody.innerHTML = rows.map(r=>{
        const nm   = fullName(r.prefix, r.first_name, r.second_name) || '—';
        const when = r.created_at || null;
        const dt   = parseDbDate(when);
        const whenText  = timeAgoUK(dt);
        const whenTitle = dt ? fmtUK.format(dt) : '';

        const typeTxt = r.loan_type ? String(r.loan_type).toUpperCase() : '—';
        return `<tr>
          <td class="col-name truncate" title="${esc(nm)}">${esc(nm)}</td>
          <td class="col-amount text-right">${gbp(r.loan_amount)}</td>
          <td class="col-type">${esc(typeTxt)}</td>
          <td class="col-date" title="${esc(whenTitle)}">${esc(whenText)}</td>
        </tr>`;
      }).join('');
    }

    async function pollNLR(force=false){
      if(nlrBusy) return;
      nlrBusy = true;
      try{
        const data = await fetchJSON(ENDPOINT_NEW);
        if(Array.isArray(data)){
          const sig = signature(data);
          if(force || sig !== prevSig){
            prevSig = sig;
            renderNLR(data);
          }
        }
      }finally{
        nlrBusy=false;
        nlrTimer=setTimeout(pollNLR, POLL_MS_NEW);
      }
    }
    btnRefreshNLR.addEventListener('click', ()=> pollNLR(true));

    // ===== Stats =====
    const elTodayLoans  = document.getElementById('statTodayLoans');
    const elInstant     = document.getElementById('statInstant');
    const elMinute      = document.getElementById('statMinute');
    const elClients     = document.getElementById('statClients');
    const elVisitsToday = document.getElementById('statVisitsToday');
    const elVisitsMonth = document.getElementById('statVisitsMonth');

    function setNum(el,v){ el.textContent = (v===null||v===undefined||v==='') ? '—' : Number(v).toLocaleString('en-GB'); }

    let statsTimer=null, statsBusy=false;
    async function pollStats(){
      if(statsBusy) return;
      statsBusy = true;
      try{
        const s = await fetchJSON(ENDPOINT_STATS);
        if(s && typeof s==='object'){
          setNum(elTodayLoans,  s.today_loans);
          setNum(elInstant,     s.total_instant_lends);
          setNum(elMinute,      s.total_minutes_lends);
          setNum(elClients,     s.num_clients);
          setNum(elVisitsToday, s.visits_today);
          setNum(elVisitsMonth, s.visits_month);
        }
      }finally{
        statsBusy=false;
        statsTimer=setTimeout(pollStats, POLL_MS_STATS);
      }
    }

    // start + cleanup
    pollNLR(true);
    pollStats();
    window.addEventListener('pagehide', ()=>{ if(nlrTimer) clearTimeout(nlrTimer); if(statsTimer) clearTimeout(statsTimer); });
  </script>
</body>
</html>
