<?php
// Include database connection
include 'db_connection.php';
session_start();

// Function to check the logged-in user and return the user type and ID
function getLoggedInUser() {
    if (isset($_SESSION['admin_id'])) {
        return ['id' => $_SESSION['admin_id'], 'type' => 'admin', 'ref_id' => $_SESSION['admin_ref_Id'] ?? null];
    } elseif (isset($_SESSION['user_id'])) {
        return ['id' => $_SESSION['user_id'], 'type' => 'user', 'ref_id' => $_SESSION['user_ref_Id'] ?? null];
    }
    return null;
}

// Check if user is logged in
$user = getLoggedInUser();
if (!$user) {
    echo "Unauthorized access.";
    exit;
}

// Assign the user ID, type, and ref_id
$updated_by = $user['ref_id'];  // Ref ID will be used for updated_by
$user_type = $user['type'];  // 'admin' or 'user'

// Receive and sanitize inputs
$chargeMode = $_POST['chargeMode'] ?? '';
$chargesType = $_POST['chargesType'] ?? '';
$amount = $_POST['amount'] ?? '';
$change_reason = trim($_POST['change_reason'] ?? '');

// Validate inputs
if (!$chargeMode || !$chargesType || !$amount || !is_numeric($amount)) {
    echo "Invalid input.";
    exit;
}

$amount = round(floatval($amount), 2);
if ($chargesType === 'Interest' && ($amount < 0 || $amount > 100)) {
    echo "Interest must be between 0 and 100.";
    exit;
}

try {
    // Start transaction
    $pdo->beginTransaction();

    // Get fee_type_id from feetype table
    $stmt = $pdo->prepare("SELECT fee_type_id FROM feetype WHERE name = ? LIMIT 1");
    $stmt->execute([$chargesType]);
    $feeType = $stmt->fetch();

    if (!$feeType) {
        echo "Invalid charges type.";
        $pdo->rollBack();
        exit;
    }
    $fee_type_id = $feeType['fee_type_id'];

    // Get loan_type_id from loantype table (Corrected table name here)
    $stmt = $pdo->prepare("SELECT loan_type_id FROM loantype WHERE name = ? LIMIT 1");
    $stmt->execute([$chargeMode]);
    $loanType = $stmt->fetch();

    if (!$loanType) {
        echo "Invalid loan type.";
        $pdo->rollBack();
        exit;
    }
    $loan_type_id = $loanType['loan_type_id'];

    // Get old amount from loanfee table
    $stmt = $pdo->prepare("SELECT amount FROM loanfee WHERE fee_type_id = ? AND loan_type_id = ? LIMIT 1");
    $stmt->execute([$fee_type_id, $loan_type_id]);
    $existing = $stmt->fetch();
    $old_amount = $existing ? $existing['amount'] : null;

    if ($existing) {
        // Update existing row in loanfee
        $stmt = $pdo->prepare("UPDATE loanfee SET amount = ?, updated_at = NOW(), updated_by = ? WHERE fee_type_id = ? AND loan_type_id = ?");
        $stmt->execute([$amount, $updated_by, $fee_type_id, $loan_type_id]);
    } else {
        // Insert new row in loanfee
        $stmt = $pdo->prepare("INSERT INTO loanfee (fee_type_id, amount, updated_at, updated_by, loan_type_id) VALUES (?, ?, NOW(), ?, ?)");
        $stmt->execute([$fee_type_id, $amount, $updated_by, $loan_type_id]);
    }

    // Track changes if old amount differs
    if ($old_amount !== null && $old_amount != $amount) {
        $stmt = $pdo->prepare("INSERT INTO track_changes (table_name, field_name, old_data, new_data, updated_by, updated_at, change_reason) VALUES (?, ?, ?, ?, ?, NOW(), ?)");
        $stmt->execute([
            'loanfee',              // Table name
            'amount',               // Field name
            $old_amount,            // Old data
            $amount,                // New data
            $updated_by,            // User who made the update
            $change_reason          // Reason for change
        ]);
    }

    // Commit the transaction
    $pdo->commit();
    echo "Charges updated successfully.";

} catch (PDOException $e) {
    // Rollback the transaction if an error occurs
    $pdo->rollBack();

    // Log detailed error information including query and input values
    error_log("Update error: " . $e->getMessage());
    error_log("SQL Query: " . $stmt->queryString);  // Log the last executed query
    error_log("Input data: chargeMode: $chargeMode, chargesType: $chargesType, amount: $amount, updated_by: $updated_by");

    // Return a generic user-friendly message
    echo "Failed to update charges. Please check the error logs for details.";
}
?>
