<?php
// update_user_credentials.php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
header('Content-Type: application/json; charset=utf-8');

// --- DB ---
require __DIR__ . '/db_connection.php';

// --- Helpers ---
function json_out($arr, int $code = 200) {
    http_response_code($code);
    echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}
function getLoggedInUser() {
    if (isset($_SESSION['admin_id'])) {
        return ['id'=>$_SESSION['admin_id'], 'type'=>'admin', 'ref_id'=>($_SESSION['admin_ref_Id'] ?? null)];
    }
    if (isset($_SESSION['user_id'])) {
        return ['id'=>$_SESSION['user_id'], 'type'=>'user', 'ref_id'=>($_SESSION['user_ref_Id'] ?? null)];
    }
    return null;
}

// --- Auth ---
$user = getLoggedInUser();
if (!$user) json_out(['error'=>'Unauthorized access.'], 401);
$updated_by = (string)($user['ref_id'] ?? 'SYSTEM'); // who changed it

// --- Inputs (form POST) ---
$user_ref_id = isset($_POST['user_id']) ? (string)$_POST['user_id'] : '';
$username    = isset($_POST['username']) ? trim((string)$_POST['username']) : '';
$password    = isset($_POST['password']) ? (string)$_POST['password'] : '';

if ($user_ref_id === '' || $username === '' || $password === '') {
    json_out(['error'=>'Missing required fields.'], 400);
}

try {
    // Fetch current user (by user_ref_id)
    $sel = $pdo->prepare("SELECT user_id, user_ref_id, username, password_hash FROM users WHERE user_ref_id = ? LIMIT 1");
    $sel->execute([$user_ref_id]);
    $current = $sel->fetch(PDO::FETCH_ASSOC);
    if (!$current) json_out(['error'=>'User not found'], 404);

    $reference_id = (string)($current['user_id'] ?? $current['user_ref_id']); // prefer numeric PK

    // Detect changes
    $changes = [];
    $new_hash = $current['password_hash'];

    if ((string)$current['username'] !== $username) {
        $changes[] = [
            'field'    => 'username',
            'old_data' => (string)$current['username'],
            'new_data' => (string)$username,
        ];
    }

    // Only mark change if the provided password (plain) does NOT verify against current hash
    if (!password_verify($password, (string)$current['password_hash'])) {
        $new_hash = password_hash($password, PASSWORD_DEFAULT);
        $changes[] = [
            'field'    => 'password_hash',
            'old_data' => (string)$current['password_hash'],
            'new_data' => (string)$new_hash,
        ];
    }

    if (!$changes) {
        json_out(['error'=>'No changes detected.'], 200);
    }

    // Apply update + track changes in a transaction
    $pdo->beginTransaction();

    $upd = $pdo->prepare("UPDATE users SET username = ?, password_hash = ? WHERE user_ref_id = ?");
    $upd->execute([$username, $new_hash, $user_ref_id]);

    $track = $pdo->prepare("
        INSERT INTO track_changes
            (table_name, reference_id, field_name, old_data, new_data, updated_by, updated_at, change_reason)
        VALUES
            ('users', ?, ?, ?, ?, ?, NOW(), ?)
    ");

    foreach ($changes as $chg) {
        $track->execute([
            $reference_id,                 // reference_id
            $chg['field'],                 // field_name
            $chg['old_data'],              // old_data
            $chg['new_data'],              // new_data
            $updated_by,                   // updated_by (ref_id from session)
            'User details update',         // change_reason
        ]);
    }

    $pdo->commit();

    json_out([
        'success' => true,
        'message' => 'User details updated successfully.',
        'changed' => array_column($changes, 'field'),
    ]);

} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    // error_log($e->getMessage());
    json_out(['error'=>'Error updating user details.'], 500);
}
