<?php
// update_staff_data.php
declare(strict_types=1);

header('Content-Type: application/json; charset=utf-8');
session_start();
require __DIR__ . '/db_connection.php'; // provides $pdo

// --- helpers ---
function me_ref(): string {
  return $_SESSION['admin_ref_Id'] ?? $_SESSION['user_ref_Id'] ?? 'SYSTEM';
}
function json_out($arr, int $code=200){ http_response_code($code); echo json_encode($arr, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function log_change(PDO $pdo, string $field, $old, $new, $refId, string $by, string $reason='Staff details update'): void {
  if ($old === $new) return;
  $stmt = $pdo->prepare("
    INSERT INTO track_changes (table_name, reference_id, field_name, old_data, new_data, updated_by, updated_at, change_reason)
    VALUES ('staff', ?, ?, ?, ?, ?, NOW(), ?)
  ");
  $stmt->execute([(string)$refId, $field, $old, $new, $by, $reason]);
}
function save_image(array $file): ?string {
  if (!isset($file['tmp_name']) || $file['error'] !== UPLOAD_ERR_OK) return null;
  $finfo = finfo_open(FILEINFO_MIME_TYPE);
  $mime  = finfo_file($finfo, $file['tmp_name']);
  finfo_close($finfo);
  $ok = ['image/jpeg'=>'jpg','image/png'=>'png'];
  if (!isset($ok[$mime])) return null;
  $ext = $ok[$mime];
  $dir = __DIR__ . '/includes/uploads/staff_images';
  if (!is_dir($dir)) mkdir($dir, 0775, true);
  $name = 'staff_' . date('Ymd_His') . '_' . bin2hex(random_bytes(4)) . '.' . $ext;
  $dest = $dir . '/' . $name;
  if (!move_uploaded_file($file['tmp_name'], $dest)) return null;
  return $name; // store file name in DB
}

// --- auth guard (optional, but recommended) ---
if (!isset($_SESSION['admin_id']) && !isset($_SESSION['user_id'])) {
  json_out(['error' => 'Unauthorized access'], 401);
}

// --- input: accept staff_id OR staff_ref_id ---
$staffKey = $_POST['staff_id'] ?? $_POST['staff_ref_id'] ?? null;
if (!$staffKey) json_out(['error' => 'Staff ID not provided'], 400);

// fetch by ref if non-numeric, else by numeric id
if (ctype_digit((string)$staffKey)) {
  $sel = $pdo->prepare("SELECT * FROM staff WHERE staff_id = ? LIMIT 1");
} else {
  $sel = $pdo->prepare("SELECT * FROM staff WHERE staff_ref_id = ? LIMIT 1");
}
$sel->execute([$staffKey]);
$before = $sel->fetch(PDO::FETCH_ASSOC);
if (!$before) json_out(['error' => 'Staff not found'], 404);

// --- collect inputs (use current values as defaults) ---
$prefix  = trim($_POST['prefix']     ?? $before['prefix']);
$first   = trim($_POST['firstName']  ?? $before['first_name']);
$last    = trim($_POST['lastName']   ?? $before['second_name']);
$gender  = trim($_POST['gender']     ?? $before['gender']);
$dobIn   = trim($_POST['dob']        ?? $before['dob']); // might be dd/mm/yyyy
$mobile  = preg_replace('/\D+/', '', $_POST['mobile'] ?? $before['phone_number']);
$email   = trim($_POST['email']      ?? $before['email']);
$apt     = trim($_POST['apt']        ?? $before['apt_no']);
$address = trim($_POST['address']    ?? $before['address']);
$post    = strtoupper(trim($_POST['postcode'] ?? $before['postcode']));
$country = trim($_POST['country']    ?? $before['country']);
$role    = trim($_POST['role']       ?? $before['role']);

// normalize DOB to Y-m-d if incoming as dd/mm/yyyy
$dob = $dobIn;
if ($dob && preg_match('~^(\d{2})/(\d{2})/(\d{4})$~', $dob, $m)) {
  $dob = "{$m[3]}-{$m[2]}-{$m[1]}";
}

// optional: image upload
$newImage = null;
if (!empty($_FILES['staff_image']) && is_array($_FILES['staff_image'])) {
  $saved = save_image($_FILES['staff_image']);
  if ($saved) $newImage = $saved; // only set if saved successfully
}

// --- perform update ---
$byCol = ctype_digit((string)$staffKey) ? 'staff_id' : 'staff_ref_id';

$sql = "
  UPDATE staff SET
    prefix=?, first_name=?, second_name=?, gender=?, dob=?, phone_number=?,
    email=?, apt_no=?, address=?, postcode=?, country=?, role=?" .
    ($newImage ? ", staff_image=?" : "") . "
  WHERE {$byCol} = ?
";
$params = [
  $prefix ?: null,
  $first  ?: null,
  $last   ?: null,
  $gender ?: null,
  $dob    ?: null,
  $mobile ?: null,
  $email  ?: null,
  $apt    ?: null,
  $address?: null,
  $post   ?: null,
  $country?: null,
  $role   ?: null
];
if ($newImage) $params[] = $newImage;
$params[] = $staffKey;

$upd = $pdo->prepare($sql);
$upd->execute($params);

// --- log changes ---
$refId = $before['staff_ref_id'] ?? $before['staff_id'] ?? (string)$staffKey; // prefer ref id
$by    = me_ref();

log_change($pdo, 'prefix',       $before['prefix'],       $prefix,   $refId, $by);
log_change($pdo, 'first_name',   $before['first_name'],   $first,    $refId, $by);
log_change($pdo, 'second_name',  $before['second_name'],  $last,     $refId, $by);
log_change($pdo, 'gender',       $before['gender'],       $gender,   $refId, $by);
log_change($pdo, 'dob',          $before['dob'],          $dob,      $refId, $by);
log_change($pdo, 'phone_number', $before['phone_number'], $mobile,   $refId, $by);
log_change($pdo, 'email',        $before['email'],        $email,    $refId, $by);
log_change($pdo, 'apt_no',       $before['apt_no'],       $apt,      $refId, $by);
log_change($pdo, 'address',      $before['address'],      $address,  $refId, $by);
log_change($pdo, 'postcode',     $before['postcode'],     $post,     $refId, $by);
log_change($pdo, 'country',      $before['country'],      $country,  $refId, $by);
log_change($pdo, 'role',         $before['role'],         $role,     $refId, $by);
if ($newImage) {
  log_change($pdo, 'staff_image', $before['staff_image'], $newImage, $refId, $by, 'Staff image update');
}

json_out(['success' => true, 'message' => 'Staff details updated.']);
