<?php
// toggle_staff_status.php
if (session_status() === PHP_SESSION_NONE) {
    session_start();
}
header('Content-Type: application/json; charset=utf-8');

/** Small JSON helper */
function json_out($arr, int $code = 200) {
    http_response_code($code);
    echo json_encode($arr, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
    exit;
}

/** Who is logged in (returns id + type + ref_id) */
function getLoggedInUser() {
    if (isset($_SESSION['admin_id'])) {
        return [
            'id'     => $_SESSION['admin_id'],
            'type'   => 'admin',
            'ref_id' => $_SESSION['admin_ref_Id'] ?? null,
        ];
    }
    if (isset($_SESSION['user_id'])) {
        return [
            'id'     => $_SESSION['user_id'],
            'type'   => 'user',
            'ref_id' => $_SESSION['user_ref_Id'] ?? null,
        ];
    }
    return null;
}

$user = getLoggedInUser();
if (!$user) {
    json_out(['error' => 'Unauthorized'], 401);
}

// DB connection (must define $pdo)
require __DIR__ . '/db_connection.php';

if (!isset($_POST['staff_id']) || $_POST['staff_id'] === '') {
    json_out(['error' => 'Staff ID is required.'], 400);
}

// Incoming POST is a staff_ref_id (string)
$staffRefId = (string)$_POST['staff_id'];

try {
    // Fetch numeric staff_id and current status
    $stmt = $pdo->prepare("SELECT staff_id, is_active FROM staff WHERE staff_ref_id = ? LIMIT 1");
    $stmt->execute([$staffRefId]);
    $row = $stmt->fetch(PDO::FETCH_ASSOC);

    if (!$row) {
        json_out(['error' => 'Staff not found.'], 404);
    }

    $staffId      = (int)$row['staff_id'];
    $currentState = (int)$row['is_active'];
    $newState     = $currentState === 1 ? 0 : 1;

    $pdo->beginTransaction();

    // Update staff active flag (by numeric PK)
    $upd = $pdo->prepare("UPDATE staff SET is_active = ? WHERE staff_id = ?");
    $upd->execute([$newState, $staffId]);

    // Track change — include change_reason = 'Active status change'
    $track = $pdo->prepare("
        INSERT INTO track_changes
            (table_name, reference_id, field_name, old_data, new_data, updated_by, updated_at, change_reason)
        VALUES
            (?,          ?,            ?,          ?,        ?,        ?,         NOW(),     ?)
    ");
    $track->execute([
        'staff',                     // table_name
        (string)$staffId,            // reference_id (numeric staff_id)
        'is_active',                 // field_name
        (string)$currentState,       // old_data
        (string)$newState,           // new_data
        (string)($user['ref_id'] ?? 'SYSTEM'), // updated_by
        'Active status change',      // change_reason
    ]);

    $pdo->commit();

    json_out([
        'success'    => true,
        'newStatus'  => $newState,
        'staff_id'   => $staffId,
        'staff_ref'  => $staffRefId,
        'message'    => 'Status toggled and change logged.'
    ]);
} catch (Throwable $e) {
    if ($pdo->inTransaction()) $pdo->rollBack();
    // error_log($e->getMessage());
    json_out(['error' => 'Server error'], 500);
}
