<?php
// Include the database connection file
require_once 'db_connection.php'; // Ensure the path to your connection file is correct

header('Content-Type: application/json'); // It's good practice to set the content type header

// SQL Query to get data (ordered by createdAt in descending order)
$sql = "SELECT CONCAT(cd.prefix, ' ', cd.first_name, ' ', cd.last_name) AS client_name,
               la.loan_amount AS amount,
               la.loan_type AS type,
               la.createdat AS date
        FROM loan_application la
        JOIN customer_details cd ON la.customer_id = cd.customer_id
        ORDER BY la.createdat DESC LIMIT 8"; // Order by createdAt in descending order

try {
    // Prepare and execute the SQL query
    $stmt = $pdo->prepare($sql);
    $stmt->execute();
    $data = [];

    // Set the correct timezone for Wales (UK)
    $timezone = new DateTimeZone('Europe/London'); // Set to Wales (UK) timezone

    
    // Fetch the results and format the time
    while ($row = $stmt->fetch()) {
        // Parse createdAt and set the timezone for both dates
        $createdAt = new DateTime($row["date"], $timezone);
        $now = new DateTime('now', $timezone);
        
        // Calculate the time difference
        $interval = $createdAt->diff($now);

        // --- NEW, IMPROVED TIME FORMATTING LOGIC ---
        // This block replaces the old logic. It checks for the largest time unit first.
        if ($interval->y > 0) {
            $timeAgo = $interval->y . " year" . ($interval->y > 1 ? "s" : "") . " ago";
        } elseif ($interval->m > 0) {
            $timeAgo = $interval->m . " month" . ($interval->m > 1 ? "s" : "") . " ago";
        } elseif ($interval->d > 0) {
            // This condition specifically handles your request for "X days ago"
            $timeAgo = $interval->d . " day" . ($interval->d > 1 ? "s" : "") . " ago";
        } elseif ($interval->h > 0) {
            $timeAgo = $interval->h . " hour" . ($interval->h > 1 ? "s" : "") . " ago";
        } elseif ($interval->i > 0) {
            $timeAgo = $interval->i . " minute" . ($interval->i > 1 ? "s" : "") . " ago";
        } else {
            $timeAgo = "just now";
        }
        // --- END OF NEW LOGIC ---

        // Concatenate the pound symbol to the amount value
        $amountWithPounds = "£" . number_format($row["amount"], 2);

        // Push formatted data to the array
        $data[] = [
            "client_name" => $row["client_name"],
            "amount" => $amountWithPounds,
            "type" => $row["type"],
            "date" => $timeAgo
        ];
    }

    // Return data as JSON response
    echo json_encode($data);

} catch (PDOException $e) {
    // Handle error if the query fails
    error_log('SQL Query Error: ' . $e->getMessage(), 3, 'errors.log');
    echo json_encode(['error' => 'Unable to fetch data']);
}
?>