<?php
header('Content-Type: application/json');
require 'db_connection.php';  // Make sure this path is correct

try {
    // Check connection variable
    if (!isset($pdo)) {
        throw new Exception("Database connection not initialized.");
    }

    // Get POST data safely
    $username = $_POST['username'] ?? '';
    $password = $_POST['password'] ?? '';
    $type = $_POST['type'] ?? 'superadmin';  // default to superadmin or set from form
    $status = $_POST['status'] ?? 'active';

    // Basic validation
    if (empty($username) || empty($password)) {
        echo json_encode(['status' => 'error', 'message' => 'Username and password are required']);
        exit;
    }

    // Check if username already exists
    $checkStmt = $pdo->prepare("SELECT COUNT(*) FROM admin WHERE username = ?");
    $checkStmt->execute([$username]);
    if ($checkStmt->fetchColumn() > 0) {
        echo json_encode(['status' => 'error', 'message' => 'Username already exists']);
        exit;
    }

    // Generate next admin_ref_Id like AD00001, AD00002, ... AD99999, AD100001 etc.
    $stmt = $pdo->query("SELECT admin_ref_Id FROM admin ORDER BY admin_id DESC LIMIT 1");
    $lastRef = $stmt->fetchColumn();

    if ($lastRef) {
        // Extract numeric part, after "AD"
        $numPart = intval(substr($lastRef, 2));
        $nextNum = $numPart + 1;
    } else {
        $nextNum = 1; // first entry
    }

    // Format admin_ref_Id
    // If less than 100000, pad with zeros up to 5 digits
    if ($nextNum < 100000) {
        $admin_ref_Id = 'AD' . str_pad($nextNum, 5, '0', STR_PAD_LEFT);
    } else {
        // If 100000 or more, just prepend AD and full number (no zero padding)
        $admin_ref_Id = 'AD' . $nextNum;
    }

    // Hash the password securely
    $passwordHash = password_hash($password, PASSWORD_DEFAULT);

    // Insert into database
    $insertStmt = $pdo->prepare("INSERT INTO admin (admin_ref_Id, username, password, type, status, created_at) VALUES (?, ?, ?, ?, ?, NOW())");
    $inserted = $insertStmt->execute([$admin_ref_Id, $username, $passwordHash, $type, $status]);

    if ($inserted) {
        echo json_encode(['status' => 'success', 'message' => 'Admin registered successfully']);
    } else {
        echo json_encode(['status' => 'error', 'message' => 'Failed to register admin']);
    }

} catch (Exception $e) {
    // Log error if you want, and send error message back
    error_log('AddAdmin Error: ' . $e->getMessage());
    echo json_encode(['status' => 'error', 'message' => 'Connection error. Please try again.']);
}
