<?php
// api/verify_documents_upload.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_upload_helpers.php'; // allowed_file, save_upload, latest_state, can_upload_row
require __DIR__.'/_audit.php';          // audit($pdo, $table, $refId, $field, $old, $new, $by, $reason)
require __DIR__.'/_actor.php';          // current_updated_by()

header('Content-Type: application/json; charset=utf-8');

/** Read numeric application_id or alphanumeric application_ref_id */
function read_app_key(): string {
  foreach ([
    $_REQUEST['application_id']     ?? null,
    $_REQUEST['application_ref_id'] ?? null,
  ] as $v) {
    if ($v !== null && $v !== '') return (string)$v;
  }
  return '';
}

/** Resolve to (application_id, customer_id) */
function resolve_app(PDO $pdo, string $appKey): array {
  if ($appKey === '') throw new RuntimeException('missing application_id');
  if (ctype_digit($appKey)) {
    $q = $pdo->prepare("SELECT application_id, customer_id FROM loan_applications WHERE application_id=? LIMIT 1");
    $q->execute([(int)$appKey]);
  } else {
    $q = $pdo->prepare("SELECT application_id, customer_id FROM loan_applications WHERE application_ref_id=? LIMIT 1");
    $q->execute([$appKey]);
  }
  $r = $q->fetch();
  if (!$r) throw new RuntimeException('application not found');
  return [(int)$r['application_id'], (int)$r['customer_id']];
}

try {
  $appKey = trim(read_app_key());
  [$aid, $cidFromDb] = resolve_app($pdo, $appKey);
  $cid = (int)($_REQUEST['customer_id'] ?? 0) ?: $cidFromDb;
  if ($cid <= 0) throw new RuntimeException('invalid customer_id');

  $actor = current_updated_by();

  $uploaded = 0;
  $had = false;

  // personal_id
  if (isset($_FILES['personal_id']) && (int)$_FILES['personal_id']['error'] !== UPLOAD_ERR_NO_FILE) {
    $had = true;

    // guard (pending/verified & not expired -> block)
    $last = latest_state($pdo, $cid, $aid, 'personal_id');
    [$ok, $why] = can_upload_row($last);
    if (!$ok) throw new RuntimeException("personal_id blocked: {$why}");

    // move + insert
    save_upload($pdo, $cid, $aid, 'personal_id', $_FILES['personal_id'], 'personal_id');

    // track
    $docId = (int)$pdo->lastInsertId();
    if ($docId > 0) {
      $sel = $pdo->prepare("SELECT file_path, original_name FROM customer_documents WHERE id=?");
      $sel->execute([$docId]);
      $row = $sel->fetch() ?: ['file_path' => null, 'original_name' => null];

      $newVal = $row['file_path'] ?: ($row['original_name'] ?: "(doc {$docId})");
      audit($pdo, 'customer_documents', $docId, 'file_path', null, $newVal, $actor, 'uploaded personal_id (verify document)');
    }

    $uploaded++;
  }

  // residence_proof
  if (isset($_FILES['residence_proof']) && (int)$_FILES['residence_proof']['error'] !== UPLOAD_ERR_NO_FILE) {
    $had = true;

    $last = latest_state($pdo, $cid, $aid, 'residence_proof');
    [$ok, $why] = can_upload_row($last);
    if (!$ok) throw new RuntimeException("residence_proof blocked: {$why}");

    save_upload($pdo, $cid, $aid, 'residence_proof', $_FILES['residence_proof'], 'residence_proof');

    $docId = (int)$pdo->lastInsertId();
    if ($docId > 0) {
      $sel = $pdo->prepare("SELECT file_path, original_name FROM customer_documents WHERE id=?");
      $sel->execute([$docId]);
      $row = $sel->fetch() ?: ['file_path' => null, 'original_name' => null];

      $newVal = $row['file_path'] ?: ($row['original_name'] ?: "(doc {$docId})");
      audit($pdo, 'customer_documents', $docId, 'file_path', null, $newVal, $actor, 'uploaded residence_proof (verify document)');
    }

    $uploaded++;
  }

  if (!$had) throw new RuntimeException('no files received');

  echo json_encode(['status' => 'success', 'message' => "Uploaded {$uploaded} file(s)"]);
} catch (Throwable $e) {
  http_response_code(400);
  echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
