<?php
declare(strict_types=1);

function allowed_file(string $name, string $type): bool {
  $ext = strtolower(pathinfo($name, PATHINFO_EXTENSION));
  if (!in_array($ext, ['jpg','jpeg','png','pdf'], true)) return false;

  $type = strtolower((string)$type);
  if ($ext === 'pdf') {
    $okPdf = ['application/pdf','application/x-pdf','application/acrobat','applications/vnd.pdf','text/pdf','text/x-pdf','application/octet-stream',''];
    return in_array($type, $okPdf, true);
  }
  $okImg = ['image/jpeg','image/png',''];
  return in_array($type, $okImg, true);
}

function normalize_mime(string $path, string $fallback=''): string {
  if (function_exists('finfo_open')) {
    $f = @finfo_open(FILEINFO_MIME_TYPE);
    if ($f) { $m = @finfo_file($f, $path); @finfo_close($f); if ($m) return $m; }
  }
  return $fallback ?: 'application/octet-stream';
}

function latest_state(PDO $pdo, int $cid, int $aid, string $docType): ?array {
  $q=$pdo->prepare("SELECT id, verified, expires_at, uploaded_at
                      FROM customer_documents
                     WHERE customer_id=? AND application_id=? AND doc_type=?
                  ORDER BY uploaded_at DESC LIMIT 1");
  $q->execute([$cid,$aid,$docType]);
  $r=$q->fetch(); return $r?:null;
}

function latest_state_subtype(PDO $pdo, int $cid, int $aid, string $docType, string $prefix): ?array {
  $pat = $prefix . '\_%';
  $q=$pdo->prepare("SELECT id, verified, expires_at, uploaded_at
                      FROM customer_documents
                     WHERE customer_id=? AND application_id=? AND doc_type=?
                       AND (file_path LIKE ? OR original_name LIKE ?)
                  ORDER BY uploaded_at DESC LIMIT 1");
  $q->execute([$cid,$aid,$docType,$pat,$pat]);
  $r=$q->fetch(); return $r?:null;
}

function can_upload_row(?array $row): array {
  if(!$row) return [true,''];
  $now = new DateTimeImmutable('now');
  $expired = !empty($row['expires_at']) && (new DateTimeImmutable($row['expires_at']) < $now);
  $v = (int)$row['verified'];
  if ($v===2 || $v===3 || $expired) return [true,''];                 // rejected/expired => allowed
  if ($v===0) return [false,'Latest is pending review. Upload disabled.'];
  if ($v===1 && !$expired) return [false,'Already verified. Re-upload after expiry only.'];
  return [false,'Upload blocked.'];
}

function _upload_err_msg(int $code): string {
  return [
    UPLOAD_ERR_OK=>'OK',UPLOAD_ERR_INI_SIZE=>'Exceeded upload_max_filesize',
    UPLOAD_ERR_FORM_SIZE=>'Exceeded MAX_FILE_SIZE',UPLOAD_ERR_PARTIAL=>'Partial upload',
    UPLOAD_ERR_NO_FILE=>'No file',UPLOAD_ERR_NO_TMP_DIR=>'Missing temp dir',
    UPLOAD_ERR_CANT_WRITE=>'Cannot write to disk',UPLOAD_ERR_EXTENSION=>'Blocked by extension',
  ][$code] ?? 'Unknown upload error';
}

function save_upload(PDO $pdo, int $cid, int $aid, string $docType, array $file, ?string $prefixForName=null): void {
  if (!isset($file['error']) || $file['error'] !== UPLOAD_ERR_OK) {
    json_out(['status'=>'error','message'=>'upload error: '._upload_err_msg((int)($file['error']??-1))], 400);
  }
  if(!allowed_file($file['name'], (string)($file['type']??''))) {
    json_out(['status'=>'error','message'=>'Invalid file type'],400);
  }

  $projectRoot = dirname(__DIR__);    // …/api/..
  $relDir = 'uploads';
  $absDir = $projectRoot.'/'.$relDir;
  ensure_dir($absDir);
  @chmod($absDir, 0775);

  $orig   = $file['name'];
  $safe   = safename($orig);
  $prefix = $prefixForName ? $prefixForName : $docType;      // land_doc1/biz_doc3/personal_id
  $rel    = $relDir."/{$aid}_{$prefix}_".time()."_".$safe;
  $abs    = $projectRoot.'/'.$rel;
  ensure_dir(dirname($abs));

  $tmp = $file['tmp_name'];
  $moved = false;
  if (is_uploaded_file($tmp)) { $moved = @move_uploaded_file($tmp, $abs); }
  if (!$moved) { $moved = @rename($tmp, $abs); }
  if (!$moved) { $moved = @copy($tmp, $abs); }
  if (!$moved) {
    json_out(['status'=>'error','message'=>'move failed (check perms on '.$absDir.')'],500);
  }

  $mimeStored = normalize_mime($abs, (string)($file['type']??''));
  @chmod($abs, 0664);

  $stmt = $pdo->prepare("INSERT INTO customer_documents
    (customer_id, application_id, doc_type, file_path, original_name, mime_type, size, verified, uploaded_at, expires_at)
    VALUES (?,?,?,?,?,?,?,0,NOW(),NULL)");
  $stmt->execute([$cid,$aid,$docType,$rel,$orig,$mimeStored,(int)$file['size']]);
}
