<?php
// api/update_release_amount.php
declare(strict_types=1);
require __DIR__.'/_bootstrap.php';
require __DIR__.'/_audit.php';
require __DIR__.'/_actor.php';

header('Content-Type: application/json; charset=utf-8');

$in = json_decode(file_get_contents('php://input'), true) ?: [];

$appId   = (int)($in['application_id'] ?? 0);
$amount  = isset($in['release_amount']) ? (float)$in['release_amount'] : null;

if ($appId <= 0) { echo json_encode(['status'=>'error','message'=>'invalid application_id']); exit; }
if ($amount === null || !is_numeric($in['release_amount']) || $amount <= 0) {
  echo json_encode(['status'=>'error','message'=>'invalid release_amount']); exit;
}

// Ensure release_amount column exists
if (!column_exists($pdo,'loan_applications','release_amount')) {
  $pdo->exec("ALTER TABLE loan_applications ADD COLUMN release_amount DECIMAL(12,2) NULL DEFAULT NULL AFTER loan_amount");
}

// Load current + requested for validation
$st = $pdo->prepare("SELECT loan_amount AS requested_amount, release_amount FROM loan_applications WHERE application_id=? LIMIT 1");
$st->execute([$appId]);
$cur = $st->fetch();
if (!$cur) { echo json_encode(['status'=>'error','message'=>'application not found']); exit; }

$requested = $cur['requested_amount'] !== null ? (float)$cur['requested_amount'] : null;
$old       = $cur['release_amount']   !== null ? (float)$cur['release_amount']   : null;

if ($requested !== null && $amount > $requested) {
  echo json_encode(['status'=>'error','message'=>'Release amount cannot exceed requested amount']); exit;
}

// Count prior changes
$cnt = $pdo->prepare("SELECT COUNT(*) FROM track_changes WHERE table_name='loan_applications' AND field_name='release_amount' AND reference_id=?");
$cnt->execute([$appId]);
$updateCount = (int)$cnt->fetchColumn();

// If no actual change, just return success
if ($old !== null && (float)$old === (float)$amount) {
  echo json_encode(['status'=>'success','message'=>'No changes','update_count'=>$updateCount]); exit;
}

// Block after 2 updates
if ($updateCount >= 2) {
  echo json_encode(['status'=>'error','message'=>'Change limit exceeded (max 2 changes).']); exit;
}

$actor = current_updated_by();
$pdo->beginTransaction();
try{
  $upd = $pdo->prepare("UPDATE loan_applications SET release_amount=? WHERE application_id=?");
  $upd->execute([$amount, $appId]);

  // Track change
  audit($pdo, 'loan_applications', $appId, 'release_amount', $old, $amount, $actor, 'set release amount');

  $pdo->commit();
  echo json_encode(['status'=>'success','message'=>'Saved','update_count'=>$updateCount+1]);
}catch(Throwable $e){
  $pdo->rollBack();
  echo json_encode(['status'=>'error','message'=>'DB error']);
}
