<?php
// api/update_customer.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_audit.php';   // audit(PDO $pdo, $table, $refId, $field, $old, $new, $updated_by='SYSTEM', $reason='')
require __DIR__.'/_actor.php';   // current_updated_by()

header('Content-Type: application/json; charset=utf-8');

// --- read JSON body ---
$raw = file_get_contents('php://input');
$in  = json_decode($raw, true);
if (!is_array($in)) {
  json_out(['status'=>'error','message'=>'Invalid JSON body'], 400);
}

$customerId = (int)($in['customer_id'] ?? 0);
if ($customerId <= 0) {
  json_out(['status'=>'error','message'=>'Missing customer_id'], 400);
}

$by = current_updated_by();

// --- fetch current row ---
$sel = $pdo->prepare("
  SELECT customer_id, customer_ref_id, prefix, first_name, second_name,
         phone_number, email, gender, martial_status, dob,
         address_line1, apt_no, postcode, created_at
    FROM customers
   WHERE customer_id = ?
   LIMIT 1
");
$sel->execute([$customerId]);
$before = $sel->fetch(PDO::FETCH_ASSOC);
if (!$before) {
  json_out(['status'=>'error','message'=>'Customer not found'], 404);
}

// --- helpers ---
function norm($v) {
  if (!isset($v)) return null;
  $v = is_string($v) ? trim($v) : $v;
  return ($v === '') ? null : $v;
}

// Allowed updatable columns (match DB exactly)
$allowed = [
  'prefix',
  'first_name',
  'second_name',
  'phone_number',
  'email',
  'gender',
  'martial_status',     // note DB spelling
  'dob',
  'address_line1',
  'apt_no',
  'postcode'
];

// Build "after" state from input
$after = $before;
foreach ($allowed as $col) {
  if (array_key_exists($col, $in)) {
    $val = norm($in[$col]);

    if ($col === 'email' && $val !== null)       $val = strtolower((string)$val);
    if ($col === 'phone_number' && $val !== null) $val = preg_replace('/\D+/', '', (string)$val);
    if ($col === 'postcode' && $val !== null)     $val = strtoupper((string)$val);

    $after[$col] = $val;
  }
}

// --- server-side validation ---
$errs = [];
if ($after['first_name']   === null || $after['first_name']   === '') $errs[] = 'First name is required.';
if ($after['second_name']  === null || $after['second_name']  === '') $errs[] = 'Second name is required.';

if ($after['email'] !== null && !filter_var($after['email'], FILTER_VALIDATE_EMAIL)) {
  $errs[] = 'Invalid email address.';
}

if ($after['phone_number'] !== null) {
  $len = strlen((string)$after['phone_number']);
  if ($len > 0 && ($len < 7 || $len > 15)) $errs[] = 'Phone number must contain 7–15 digits.';
}

if ($after['dob'] !== null) {
  $ts = strtotime((string)$after['dob']);
  if ($ts === false) {
    $errs[] = 'DOB must be a valid date (YYYY-MM-DD).';
  } else {
    $min = strtotime('-15 years');
    if ($ts > $min) $errs[] = 'Customer must be at least 15 years old.';
  }
}

$validGenders  = ['Male','Female','Other', null];
$validStatuses = ['Single','Married','Divorced','Widowed','Other', null];
if (!in_array($after['gender'], $validGenders, true)) {
  $errs[] = 'Invalid gender.';
}
if (!in_array($after['martial_status'], $validStatuses, true)) {
  $errs[] = 'Invalid marital status.';
}

if ($errs) {
  json_out(['status'=>'error','message'=>implode("\n", $errs)], 422);
}

// --- perform UPDATE ---
$upd = $pdo->prepare("
  UPDATE customers SET
    prefix         = :prefix,
    first_name     = :first_name,
    second_name    = :second_name,
    phone_number   = :phone_number,
    email          = :email,
    gender         = :gender,
    martial_status = :martial_status,
    dob            = :dob,
    address_line1  = :address_line1,
    apt_no         = :apt_no,
    postcode       = :postcode
  WHERE customer_id = :id
");
$upd->execute([
  ':prefix'         => $after['prefix'],
  ':first_name'     => $after['first_name'],
  ':second_name'    => $after['second_name'],
  ':phone_number'   => $after['phone_number'],
  ':email'          => $after['email'],
  ':gender'         => $after['gender'],
  ':martial_status' => $after['martial_status'],
  ':dob'            => $after['dob'],
  ':address_line1'  => $after['address_line1'],
  ':apt_no'         => $after['apt_no'],
  ':postcode'       => $after['postcode'],
  ':id'             => $customerId
]);

// --- track changes per field ---
$reason = 'customer update';
foreach ($allowed as $col) {
  $old = $before[$col] ?? null;
  $new = $after[$col]  ?? null;
  if ($old != $new) {
    audit($pdo, 'customers', $customerId, $col, $old, $new, $by, $reason);
  }
}

json_out(['status'=>'success','message'=>'Customer updated']);
