<?php
// api/update_application_status.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_audit.php';
require __DIR__.'/_actor.php'; // session → updated_by

$in   = json_decode(file_get_contents('php://input'), true) ?: [];
$appId  = (int)($in['application_id'] ?? 0);
$status = trim((string)($in['status'] ?? ''));
$reason = trim((string)($in['reason'] ?? ''));            // required if rejected
$by     = current_updated_by();

if ($appId <= 0) {
  json_out(['status'=>'error','message'=>'Invalid application_id'], 400);
}

$map = [
  'pending'    => 'Pending',
  'in process' => 'In Process',
  'rejected'   => 'Rejected',
  'verified'   => 'Verified',
];
$key = strtolower($status);
if (!isset($map[$key])) {
  json_out(['status'=>'error','message'=>'Invalid status'], 422);
}
$finalStatus = $map[$key];
if ($finalStatus === 'Rejected' && $reason === '') {
  json_out(['status'=>'error','message'=>'Rejection reason is required'], 422);
}

// fetch current
$st = $pdo->prepare("SELECT application_status, reason_for_rejection
                       FROM loan_applications
                      WHERE application_id=? LIMIT 1");
$st->execute([$appId]);
$old = $st->fetch();
if (!$old) json_out(['status'=>'error','message'=>'Application not found'], 404);

$pdo->beginTransaction();
try {
  $reasonSql = ($finalStatus === 'Rejected') ? $reason : null;

  // update application
  $upd = $pdo->prepare("
    UPDATE loan_applications
       SET application_status = ?,
           reason_for_rejection = ?,
           updated_by = ?,
           updated_date = NOW()
     WHERE application_id = ?
  ");
  $upd->execute([$finalStatus, $reasonSql, $by, $appId]);

  // history
  $hist = $pdo->prepare("INSERT INTO loan_application_status (application_id, status) VALUES (?, ?)");
  $hist->execute([$appId, $finalStatus]);

  $pdo->commit();

  // --- track_changes (reference_id = application_id) ---
  if ((string)$old['application_status'] !== (string)$finalStatus) {
    audit($pdo, 'loan_applications', $appId, 'application_status',
          $old['application_status'], $finalStatus, $by, 'status change');
  }
  if ((string)($old['reason_for_rejection'] ?? '') !== (string)($reasonSql ?? '')) {
    $chgReason = ($finalStatus === 'Rejected')
      ? ($reason !== '' ? $reason : 'rejected')
      : 'cleared (not rejected)';
    audit($pdo, 'loan_applications', $appId, 'reason_for_rejection',
          $old['reason_for_rejection'], $reasonSql, $by, $chgReason);
  }

  json_out(['status'=>'success','message'=>'Status updated','data'=>[
    'application_id'=>$appId,
    'status'=>$finalStatus,
    'reason'=>$reasonSql
  ]]);

} catch (Throwable $e) {
  $pdo->rollBack();
  json_out(['status'=>'error','message'=>'DB error'], 500);
}
