<?php
// api/save_bank_details.php  — UK validation + track only changed fields on UPDATE
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_audit.php';   // audit($pdo,$table,$refId,$field,$old,$new,$by,$reason)
require __DIR__.'/_actor.php';   // current_updated_by()

header('Content-Type: application/json; charset=utf-8');

$in = json_decode(file_get_contents('php://input'), true) ?: [];

$application_id       = (string)($in['application_id'] ?? '');  // might be numeric id or ref like EZY00005
$customer_id          = (int)($in['customer_id'] ?? 0);
$account_holder_name  = trim((string)($in['account_holder_name'] ?? ''));
$bank_name            = trim((string)($in['bank_name'] ?? ''));
$account_number_raw   = (string)($in['account_number'] ?? '');
$sort_code_raw        = (string)($in['sort_code'] ?? '');
$swift_bic_raw        = (string)($in['swift_bic'] ?? '');
$iban_raw             = (string)($in['iban'] ?? '');
$account_type         = trim((string)($in['account_type'] ?? ''));
$is_primary_account   = isset($in['is_primary_account']) ? (int)!!$in['is_primary_account'] : 1;

/* Resolve customer if missing (accepts numeric application_id or application_ref_id) */
if (!$customer_id && $application_id !== '') {
  if (ctype_digit($application_id)) {
    $q = $pdo->prepare("SELECT customer_id FROM loan_applications WHERE application_id = ? LIMIT 1");
    $q->execute([(int)$application_id]);
  } else {
    $q = $pdo->prepare("SELECT customer_id FROM loan_applications WHERE application_ref_id = ? LIMIT 1");
    $q->execute([$application_id]);
  }
  if ($r = $q->fetch()) $customer_id = (int)$r['customer_id'];
}
if (!$customer_id) {
  json_out(['status'=>'error','message'=>'Missing customer_id/application_id'], 400);
}

/* --- UK validation / normalization --- */
$account_number = preg_replace('/\D+/', '', $account_number_raw);
$sort_code      = preg_replace('/\D+/', '', $sort_code_raw);
$swift_bic      = strtoupper(trim($swift_bic_raw));
$iban           = strtoupper(trim($iban_raw));

$errs = [];
if ($account_holder_name === '' || mb_strlen($account_holder_name) < 2) $errs[] = 'Account holder name is required.';
if ($bank_name === '') $errs[] = 'Bank name is required.';
if (!preg_match('/^\d{6}$/', $sort_code)) $errs[] = 'Sort code must be 6 digits (e.g. 123456).';
if (!preg_match('/^\d{8}$/', $account_number)) $errs[] = 'Account number must be 8 digits.';
if ($swift_bic !== '') {
  if (!preg_match('/^[A-Z0-9]{8}([A-Z0-9]{3})?$/', $swift_bic)) $errs[] = 'SWIFT/BIC must be 8 or 11 characters.';
  if (!preg_match('/^[A-Z]{4}GB[A-Z0-9]{2}([A-Z0-9]{3})?$/', $swift_bic)) $errs[] = 'SWIFT/BIC should be UK (..GB..).';
}
if ($iban !== '') {
  // UK IBAN: GBkk + BANK(4) + SORT(6) + ACC(8) = 22 chars
  if (!preg_match('/^GB\d{2}[A-Z]{4}\d{14}$/', $iban)) $errs[] = 'Invalid UK IBAN (GBkk BANK(4) SORT(6) ACC(8)).';
}
if ($errs) json_out(['status'=>'error','message'=>implode("\n", $errs)], 422);

/* --- Prep --- */
$actor  = current_updated_by();
$reason = 'bank details update (primary only)';

/* Helper: log only if changed (normalize '' -> null) */
$logIfChanged = function(PDO $pdo, int $refId, string $field, $old, $new) use ($actor, $reason) {
  $old = ($old === '') ? null : $old;
  $new = ($new === '') ? null : $new;
  if ($old !== $new) {
    audit($pdo, 'bank_details', $refId, $field, $old, $new, $actor, $reason);
  }
};

/* Values we will persist (null for empty strings) */
$newVals = [
  'bank_name'           => $bank_name           !== '' ? $bank_name           : null,
  'account_number'      => $account_number      !== '' ? $account_number      : null,
  'sort_code'           => $sort_code           !== '' ? $sort_code           : null,
  'iban'                => $iban                !== '' ? $iban                : null,
  'swift_bic'           => $swift_bic           !== '' ? $swift_bic           : null,
  'account_type'        => $account_type        !== '' ? $account_type        : null,
  'account_holder_name' => $account_holder_name !== '' ? $account_holder_name : null,
  'is_primary_account'  => $is_primary_account ? 1 : 0,
];

$pdo->beginTransaction();
try {
  /* 1) Fetch the existing PRIMARY BEFORE changing anything */
  $cur = $pdo->prepare("SELECT * FROM bank_details WHERE customer_id = ? AND is_primary_account = 1 LIMIT 1");
  $cur->execute([$customer_id]);
  $existing = $cur->fetch();

  if ($existing) {
    /* 2) UPDATE existing primary (no demotion needed) */
    $upd = $pdo->prepare("
      UPDATE bank_details
         SET bank_name = :bank_name,
             account_number = :account_number,
             sort_code = :sort_code,
             iban = :iban,
             swift_bic = :swift_bic,
             account_type = :account_type,
             account_holder_name = :account_holder_name,
             is_primary_account = :is_primary_account
       WHERE bank_details_id = :id
       LIMIT 1
    ");
    $upd->execute([
      ':bank_name'           => $newVals['bank_name'],
      ':account_number'      => $newVals['account_number'],
      ':sort_code'           => $newVals['sort_code'],
      ':iban'                => $newVals['iban'],
      ':swift_bic'           => $newVals['swift_bic'],
      ':account_type'        => $newVals['account_type'],
      ':account_holder_name' => $newVals['account_holder_name'],
      ':is_primary_account'  => $newVals['is_primary_account'],
      ':id'                  => (int)$existing['bank_details_id'],
    ]);

    $refId = (int)$existing['bank_details_id'];

    // Track ONLY changed fields
    $logIfChanged($pdo, $refId, 'bank_name',           $existing['bank_name'] ?? null,           $newVals['bank_name']);
    $logIfChanged($pdo, $refId, 'account_number',      $existing['account_number'] ?? null,      $newVals['account_number']);
    $logIfChanged($pdo, $refId, 'sort_code',           $existing['sort_code'] ?? null,           $newVals['sort_code']);
    $logIfChanged($pdo, $refId, 'iban',                $existing['iban'] ?? null,                $newVals['iban']);
    $logIfChanged($pdo, $refId, 'swift_bic',           $existing['swift_bic'] ?? null,           $newVals['swift_bic']);
    $logIfChanged($pdo, $refId, 'account_type',        $existing['account_type'] ?? null,        $newVals['account_type']);
    $logIfChanged($pdo, $refId, 'account_holder_name', $existing['account_holder_name'] ?? null, $newVals['account_holder_name']);
    $logIfChanged($pdo, $refId, 'is_primary_account',  (int)($existing['is_primary_account'] ?? 0), $newVals['is_primary_account']);

  } else {
    /* 3) No current primary → demote any older primaries, then INSERT a new primary
          (creation isn’t “a change”, so we do not write track_changes on INSERT) */
    $pdo->prepare("UPDATE bank_details SET is_primary_account = 0 WHERE customer_id = ?")
        ->execute([$customer_id]);

    $ins = $pdo->prepare("
      INSERT INTO bank_details
        (customer_id, bank_name, account_number, sort_code, iban, swift_bic, account_type, account_holder_name, is_primary_account, created_at)
      VALUES
        (:customer_id, :bank_name, :account_number, :sort_code, :iban, :swift_bic, :account_type, :account_holder_name, :is_primary_account, NOW())
    ");
    $ins->execute([
      ':customer_id'         => $customer_id,
      ':bank_name'           => $newVals['bank_name'],
      ':account_number'      => $newVals['account_number'],
      ':sort_code'           => $newVals['sort_code'],
      ':iban'                => $newVals['iban'],
      ':swift_bic'           => $newVals['swift_bic'],
      ':account_type'        => $newVals['account_type'],
      ':account_holder_name' => $newVals['account_holder_name'],
      ':is_primary_account'  => $newVals['is_primary_account'],
    ]);
  }

  $pdo->commit();

  json_out([
    'status'=>'success',
    'message'=>'Bank information saved',
    'normalized'=>[
      'sort_code'      => $sort_code,
      'account_number' => $account_number,
      'swift_bic'      => $swift_bic,
      'iban'           => $iban
    ]
  ]);
} catch (Throwable $e) {
  $pdo->rollBack();
  json_out(['status'=>'error','message'=>'DB error'], 500);
}
