<?php
// rbac.php — drop-in RBAC wired to your api/_bootstrap.php and api/_actor.php
declare(strict_types=1);

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }

// ---- pull in PDO + actor helpers -----------------------------------------
require_once __DIR__ . '/_bootstrap.php'; // gives $pdo + helpers (json_out, column_exists, ...)
require_once __DIR__ . '/_actor.php';     // gives current_identity(), current_role_type(), ...

// ---- tiny util ------------------------------------------------------------
function rbac_table_exists(PDO $pdo, string $name): bool {
  $q = $pdo->prepare("SELECT 1 FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ? LIMIT 1");
  $q->execute([$name]);
  return (bool)$q->fetchColumn();
}

// ---- identity signature (to invalidate cached perms) ----------------------
function rbac_identity_sig(): string {
  $id = current_identity($GLOBALS['pdo'] ?? null); // ['who','id','ref','type']
  return implode('|', [
    (string)($id['who'] ?? ''),
    (string)($id['ref'] ?? ''),
    (string)($id['type'] ?? '')
  ]);
}

// ---- core: load effective permission codes into session -------------------
function rbac_load_perms(?PDO $pdo = null): void {
  $pdo = $pdo ?? ($GLOBALS['pdo'] ?? null);
  if (!$pdo instanceof PDO) { $_SESSION['perms'] = []; return; }

  $id = current_identity($pdo); // ['who','ref','type']
  $who  = (string)($id['who']  ?? '');
  $ref  = (string)($id['ref']  ?? '');
  $type = (string)($id['type'] ?? '');
  $perms = [];

  // Prefer view lookups when available (admin path first)
  if ($who === 'admin' && $ref !== '' && rbac_table_exists($pdo, 'v_admin_allowed_perm_codes')) {
    $st = $pdo->prepare("SELECT perm_code FROM v_admin_allowed_perm_codes WHERE admin_ref_id = ?");
    $st->execute([$ref]);
    $perms = array_column($st->fetchAll(PDO::FETCH_ASSOC), 'perm_code');
  }
  // Optional: user view (only if you add it later)
  elseif ($who === 'user' && $ref !== '' && rbac_table_exists($pdo, 'v_user_allowed_perm_codes')) {
    $st = $pdo->prepare("SELECT perm_code FROM v_user_allowed_perm_codes WHERE user_ref_id = ?");
    $st->execute([$ref]);
    $perms = array_column($st->fetchAll(PDO::FETCH_ASSOC), 'perm_code');
  }

  // Fallback: map type -> roles.code and collect perms via role_permissions
  if (empty($perms) && $type !== '') {
    $sql = "
      SELECT DISTINCT p.code
      FROM roles r
      JOIN role_permissions rp ON rp.role_id = r.role_id AND rp.allow = 1
      JOIN permissions p       ON p.perm_id  = rp.perm_id
      WHERE r.code = ?
    ";
    $st = $pdo->prepare($sql);
    $st->execute([$type]);
    $perms = array_column($st->fetchAll(PDO::FETCH_ASSOC), 'code');
  }

  $_SESSION['perms'] = array_values(array_unique($perms));
  $_SESSION['perms_sig'] = rbac_identity_sig();
  $_SESSION['perms_loaded_at'] = time();
}

// ---- ensure loaded (once per request / identity change) -------------------
function rbac_ensure_loaded(): void {
  $sig = rbac_identity_sig();
  if (!isset($_SESSION['perms'], $_SESSION['perms_sig']) || $_SESSION['perms_sig'] !== $sig) {
    rbac_load_perms($GLOBALS['pdo'] ?? null);
  }
}

// ---- check/guard helpers --------------------------------------------------
function can(string $code): bool {
  rbac_ensure_loaded();
  return in_array($code, $_SESSION['perms'] ?? [], true);
}
function can_any(array $codes): bool {
  rbac_ensure_loaded();
  $have = $_SESSION['perms'] ?? [];
  foreach ($codes as $c) if (in_array($c, $have, true)) return true;
  return false;
}
function guard(string $code): void {
  if (!can($code)) {
    http_response_code(403);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error'=>'forbidden','message'=>'Not enough privileges','need'=>$code]);
    exit;
  }
}

// ---- auto-guard for pages based on ui_routes ------------------------------
/*
  Looks up the current request path in ui_routes and enforces its perm_code.
  Match order: exact route, then longest suffix that matches (to support nested dirs).
*/
function autoguard(?PDO $pdo = null): void {
  $pdo = $pdo ?? ($GLOBALS['pdo'] ?? null);
  if (!$pdo instanceof PDO) return;

  if (!rbac_table_exists($pdo, 'ui_routes')) return;

  $uri  = strtok($_SERVER['REQUEST_URI'] ?? $_SERVER['PHP_SELF'] ?? '', '?');
  $path = ltrim($uri, '/');                  // e.g. pages/lend/ezycashdetails.php
  $base = basename($path);

  $sql = "SELECT perm_code
          FROM ui_routes
          WHERE route = :exact OR :path LIKE CONCAT('%', route)
          ORDER BY (route = :exact) DESC, LENGTH(route) DESC
          LIMIT 1";
  $st = $pdo->prepare($sql);
  $st->execute([':exact'=>$path, ':path'=>$path]);
  $row = $st->fetch(PDO::FETCH_ASSOC);

  if ($row && !empty($row['perm_code'])) {
    guard($row['perm_code']);
  }
}

// ---- convenience: expose current perms to JS (optional) ------------------
function echo_perms_js_variable(string $varName='APP_PERMS'): void {
  rbac_ensure_loaded();
  $perms = $_SESSION['perms'] ?? [];
  echo "<script>window.".preg_replace('/[^A-Za-z0-9_]/','_', $varName)." = ".json_encode(array_values($perms)).";</script>";
}

// ---- run once per request (safe) -----------------------------------------
rbac_ensure_loaded();
/* You can also call autoguard() at the top of each page:
   require_once __DIR__.'/rbac.php';
   autoguard();
*/
