<?php
// api/opening_screens_upload.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_actor.php';
require __DIR__.'/_audit.php';

header('Content-Type: application/json; charset=utf-8');

try {
  // 1) Label
  $label = trim($_POST['label'] ?? '');
  if ($label === '') {
    echo json_encode(['status'=>'error','message'=>'Missing label']); exit;
  }

  // 2) File: accept either "file" or "media"
  $fileArr = $_FILES['file'] ?? $_FILES['media'] ?? null;

  // If PHP upload limits were exceeded, $_FILES will be empty and/or error=UPLOAD_ERR_INI_SIZE
  if (!$fileArr || !is_array($fileArr)) {
    echo json_encode(['status'=>'error','message'=>'Missing file']); exit;
  }
  if (($fileArr['error'] ?? UPLOAD_ERR_OK) !== UPLOAD_ERR_OK) {
    $err = (int)$fileArr['error'];
    $map = [
      UPLOAD_ERR_INI_SIZE   => 'The file exceeds server limit (upload_max_filesize).',
      UPLOAD_ERR_FORM_SIZE  => 'The file exceeds form limit (MAX_FILE_SIZE).',
      UPLOAD_ERR_PARTIAL    => 'The file was only partially uploaded.',
      UPLOAD_ERR_NO_FILE    => 'No file was uploaded.',
      UPLOAD_ERR_NO_TMP_DIR => 'Missing a temporary folder on server.',
      UPLOAD_ERR_CANT_WRITE => 'Failed to write file to disk.',
      UPLOAD_ERR_EXTENSION  => 'A PHP extension stopped the file upload.',
    ];
    $msg = $map[$err] ?? 'Upload error';
    echo json_encode(['status'=>'error','message'=>$msg]); exit;
  }

  // 3) Basic size guard (keep in sync with client)
  if (($fileArr['size'] ?? 0) > 12*1024*1024) {
    echo json_encode(['status'=>'error','message'=>'File too large (max 12MB)']); exit;
  }

  // 4) Validate type (image / video / lottie)
  $origName = (string)($fileArr['name'] ?? 'file');
  $tmpPath  = (string)($fileArr['tmp_name'] ?? '');
  if ($tmpPath === '' || !is_uploaded_file($tmpPath)) {
    echo json_encode(['status'=>'error','message'=>'Temp upload not found']); exit;
  }

  $ext = strtolower(pathinfo($origName, PATHINFO_EXTENSION));
  $kind = 'image';
  if ($ext === 'json') $kind = 'lottie';
  elseif (in_array($ext, ['mp4','webm','mov','m4v','ogg','ogv','gif'], true)) {
    $kind = ($ext === 'gif') ? 'gif' : 'video';
  }

  // 5) Move file
  $safeBase = preg_replace('~[^a-z0-9\-_]+~i','_', pathinfo($origName, PATHINFO_FILENAME));
  $destDir  = __DIR__.'/../uploads/opening_screens';
  if (!is_dir($destDir)) { @mkdir($destDir, 0775, true); }
  $newName  = $safeBase.'_'.date('Ymd_His').'.'.$ext;
  $destPath = $destDir.'/'.$newName;
  if (!move_uploaded_file($tmpPath, $destPath)) {
    echo json_encode(['status'=>'error','message'=>'Failed to save uploaded file']); exit;
  }

  // 6) Store row (deactivate others if this is set active later via update API)
  $updatedBy = current_updated_by($pdo);
  $stmt = $pdo->prepare("
    INSERT INTO opening_screens (label, media_kind, file_path, is_active, is_deleted, updated_by, updated_at)
    VALUES (:label, :kind, :path, 0, 0, :by, NOW())
  ");
  $relPath = 'uploads/opening_screens/'.$newName; // path relative to web root
  $stmt->execute([
    ':label'=>$label, ':kind'=>$kind, ':path'=>$relPath, ':by'=>$updatedBy
  ]);
  $id = (int)$pdo->lastInsertId();

  // 7) Audit
  audit($pdo, 'opening_screens', $id, 'create', null, json_encode([
    'label'=>$label,'media_kind'=>$kind,'file_path'=>$relPath
  ], JSON_UNESCAPED_SLASHES), $updatedBy, 'upload');

  echo json_encode(['status'=>'success','screen_id'=>$id]);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['status'=>'error','message'=>'Server error']);
}
