<?php
// api/notifications.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
header('Content-Type: application/json; charset=utf-8');

/**
 * Query params:
 *   audience=backend|consumer   (default: backend)
 *   limit=1..100                (default: 20)
 *   days=1..90                  (default: 7)  // items older than this get is_recent=0
 *
 * Output:
 * {
 *   audience: "backend",
 *   total: <int>,
 *   total_recent: <int>,
 *   total_older: <int>,
 *   counts: { "<Category>": <int>, ... },
 *   categories: [{ name, slug, items:[...] }, ...],
 *   flat: [ ... ]  // newest first (by created_at, then id)
 * }
 */

// ---------- app base (so returned URLs are root-absolute) ----------
$script = $_SERVER['SCRIPT_NAME'] ?? '/';
$APP_BASE = rtrim(dirname(dirname($script)), '/');   // one level up from /api
if ($APP_BASE === '') $APP_BASE = '/';

// default image to fall back to (adjust file if needed)
$DEFAULT_IMG = $APP_BASE . '/assets/images/notify-default.png';

// ---------- helpers ----------
function normalize_img(?string $p, string $appBase, string $fallback): string {
  $p = trim((string)$p);
  if ($p === '') return $fallback;

  // absolute URL already
  if (preg_match('~^https?://~i', $p)) return $p;

  // already site-rooted (starts with "/")
  if (isset($p[0]) && $p[0] === '/') return $p;

  // common relative folders -> prefix with app base
  if (preg_match('~^(uploads/|assets/|images/)~i', $p) === 1) {
    return rtrim($appBase, '/') . '/' . ltrim($p, '/');
  }

  // just a filename -> assume /uploads/notifications/
  return rtrim($appBase, '/') . '/uploads/notifications/' . ltrim($p, '/');
}

function cat_label_for_row(array $r): string {
  $raw = trim((string)($r['category'] ?? ''));
  if ($raw !== '') {
    $lower = strtolower($raw);
    if (in_array($lower, ['new request','new requests','loan request','loan requests'], true)) {
      return 'New Requests';
    }
    return $raw;
  }
  // derive when missing
  $rt = strtolower((string)($r['ref_table'] ?? ''));
  if (in_array($rt, ['loan_applications','loan_application','applications'], true)) {
    return 'New Requests';
  }
  return 'General';
}
function slugify(string $s): string {
  $s = strtolower($s);
  $s = preg_replace('/[^a-z0-9]+/','-',$s);
  return trim($s,'-');
}
function age_label(string $ts): string {
  $t = strtotime($ts) ?: time();
  $diff = time() - $t;
  if ($diff < 5) return 'just now';
  if ($diff < 60) return $diff.' secs ago';
  $m = intdiv($diff, 60);
  if ($m < 60) return $m.' min'.($m===1?'':'s').' ago';
  $h = intdiv($m, 60);
  if ($h < 24) return $h.' hour'.($h===1?'':'s').' ago';
  $d = intdiv($h, 24);
  if ($d < 7) return $d.' day'.($d===1?'':'s').' ago';
  $w = intdiv($d, 7);
  return $w.' week'.($w===1?'':'s').' ago';
}

// ---------- inputs ----------
$aud = strtolower((string)($_GET['audience'] ?? 'backend'));
$aud = in_array($aud, ['backend','consumer'], true) ? $aud : 'backend';

$limit = max(1, min(100, (int)($_GET['limit'] ?? 20)));
$days  = max(1, min(90,  (int)($_GET['days']  ?? 7)));
$recentCutoff = (new DateTimeImmutable())->modify("-{$days} days");

// ---------- query (uses created_at; your table has no updated_at) ----------
$sql = "
  SELECT
    n.notification_id,
    n.category,
    n.audience,
    n.title,
    n.body,
    n.image_path,
    n.ref_table,
    n.ref_id,
    n.is_active,
    n.created_at
  FROM notifications n
  WHERE n.is_active = 1
    AND n.audience = :aud
  ORDER BY n.created_at DESC, n.notification_id DESC
  LIMIT :lim
";
$st = $pdo->prepare($sql);
$st->bindValue(':aud', $aud, PDO::PARAM_STR);
$st->bindValue(':lim', $limit, PDO::PARAM_INT);
$st->execute();
$rows = $st->fetchAll();

// ---------- build payload ----------
$grouped = [];
$counts  = [];
$flat    = [];

$total_recent = 0;
$total_older  = 0;

foreach ($rows as $r) {
  // normalize image path to root-absolute
  $r['image_path'] = normalize_img($r['image_path'] ?? '', $APP_BASE, $DEFAULT_IMG);

  // normalize category and slug
  $r['category']      = cat_label_for_row($r);
  $r['category_slug'] = slugify($r['category']);

  // age helpers
  $created = (string)($r['created_at'] ?? '');
  $r['age_seconds'] = max(0, time() - (strtotime($created) ?: time()));
  $r['age_label']   = age_label($created);

  // recent vs older
  $isRecent = true;
  try {
    $dt = new DateTimeImmutable($created);
    $isRecent = ($dt >= $recentCutoff);
  } catch (Throwable $e) {
    $isRecent = true;
  }
  $r['is_recent'] = $isRecent ? 1 : 0;
  if ($isRecent) $total_recent++; else $total_older++;

  // stream list
  $flat[] = $r;

  // group by category
  $cat = $r['category'];
  if (!isset($grouped[$cat])) {
    $grouped[$cat] = ['name' => $cat, 'slug' => $r['category_slug'], 'items' => []];
    $counts[$cat]  = 0;
  }
  $grouped[$cat]['items'][] = $r;
  $counts[$cat]++;
}

// sort categories (New Requests first)
$catNames = array_keys($grouped);
usort($catNames, function($a,$b){
  if ($a === 'New Requests' && $b !== 'New Requests') return -1;
  if ($b === 'New Requests' && $a !== 'New Requests') return 1;
  return strcmp($a,$b);
});

$out = [
  'audience'     => $aud,
  'total'        => count($flat),
  'total_recent' => $total_recent,
  'total_older'  => $total_older,
  'counts'       => $counts,
  'categories'   => [],
  'flat'         => $flat, // newest-first
];

foreach ($catNames as $name) {
  $out['categories'][] = $grouped[$name];
}

echo json_encode($out, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
