<?php
// api/notification_sounds_upload.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_actor.php';

header('Content-Type: application/json; charset=utf-8');

function out(array $o){
  echo json_encode($o, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES);
  exit;
}

// ---- Validate input ----
$type  = strtolower(trim((string)($_POST['type'] ?? '')));
$label = trim((string)($_POST['label'] ?? ''));
$vol   = (string)($_POST['volume'] ?? '1');
$loop  = isset($_POST['loop']) ? 1 : 0;
$make  = isset($_POST['make_active']) ? 1 : 0;

if(!in_array($type, ['notice','new_request'], true)) out(['success'=>false,'error'=>'Invalid type']);
if($label==='') out(['success'=>false,'error'=>'Label is required']);

$volume = max(0.0, min(1.0, floatval($vol)));

if(!isset($_FILES['sound_file'])) out(['success'=>false,'error'=>'File field "sound_file" is missing']);
$f = $_FILES['sound_file'];
if((int)$f['error'] !== UPLOAD_ERR_OK) out(['success'=>false,'error'=>'Upload error code '.$f['error']]);
if(!is_uploaded_file($f['tmp_name'])) out(['success'=>false,'error'=>'Invalid upload']);
if((int)$f['size'] > 10*1024*1024) out(['success'=>false,'error'=>'Max 10MB']);

// ---- Where to save (filesystem) ----
// We avoid using DOCUMENT_ROOT/app folder name. Instead, save relative to this file:
//   api/ -> ../assets/sounds/custom
$appAbsBase = realpath(__DIR__ . '/..'); // one level up from /api
if ($appAbsBase === false) out(['success'=>false,'error'=>'Cannot resolve app base path']);
$saveAbsDir = $appAbsBase . '/assets/sounds/custom';

if(!is_dir($saveAbsDir)){
  if(!@mkdir($saveAbsDir, 0775, true) && !is_dir($saveAbsDir)){
    out(['success'=>false,'error'=>'Cannot create folder: '.$saveAbsDir]);
  }
}

// ---- Determine extension (mp3/ogg) ----
try{
  $fi = new finfo(FILEINFO_MIME_TYPE);
  $mime = (string)$fi->file($f['tmp_name']);
}catch(Throwable $e){ $mime = ''; }

$map = [
  'audio/mpeg'       => 'mp3',
  'audio/mp3'        => 'mp3',
  'audio/x-mp3'      => 'mp3',
  'audio/ogg'        => 'ogg',
  'application/ogg'  => 'ogg',
];
$ext = $map[$mime] ?? null;
if(!$ext){
  $oext = strtolower(pathinfo((string)$f['name'], PATHINFO_EXTENSION));
  if(in_array($oext, ['mp3','ogg'], true)) $ext = $oext;
}
if(!$ext) out(['success'=>false,'error'=>'Only MP3/OGG allowed (mime='.$mime.')']);

// ---- Generate filename and move ----
$token = bin2hex(random_bytes(8));
$base  = $type.'_'.$token;
$fileAbs = $saveAbsDir.'/'.$base.'.'.$ext;

// IMPORTANT: store a **portable** web path in DB (no app-folder prefix)
// This will work after renaming /ezylend-admin to anything else.
$fileWebPortable = 'assets/sounds/custom/'.$base.'.'.$ext;

if(!@move_uploaded_file($f['tmp_name'], $fileAbs)){
  out(['success'=>false,'error'=>'Failed to save file']);
}
@chmod($fileAbs, 0644);

// ---- Who created it ----
$by = current_updated_by($pdo);

// ---- DB write (single active per type if requested) ----
try{
  $pdo->beginTransaction();

  if($make){
    $pdo->prepare('UPDATE `notification_sounds` SET `enabled`=0 WHERE `type`=:t')
        ->execute([':t'=>$type]);
  }

  $stmt = $pdo->prepare('
    INSERT INTO `notification_sounds`
      (`type`,`label`,`src`,`volume`,`loop`,`enabled`,`created_at`,`created_by`)
    VALUES
      (:type,:label,:src,:volume,:loop,:enabled,NOW(),:created_by)
  ');
  $stmt->execute([
    ':type'       => $type,
    ':label'      => $label,
    ':src'        => $fileWebPortable, // <-- store portable path
    ':volume'     => $volume,
    ':loop'       => $loop,
    ':enabled'    => ($make ? 1 : 0),
    ':created_by' => $by
  ]);

  $id = (int)$pdo->lastInsertId();
  $pdo->commit();

  out([
    'success'    => true,
    'id'         => $id,
    'src'        => $fileWebPortable, // portable path in response
    'type'       => $type,
    'label'      => $label,
    'volume'     => $volume,
    'loop'       => $loop,
    'enabled'    => $make ? 1 : 0,
    'created_by' => $by,
    'message'    => 'Uploaded'
  ]);
}catch(Throwable $e){
  if($pdo->inTransaction()) $pdo->rollBack();
  @unlink($fileAbs);
  out(['success'=>false,'error'=>'DB error: '.$e->getMessage()]);
}
