<?php
// api/notification_sounds_set_active.php — set one sound active for its type (with track changes)
declare(strict_types=1);

require __DIR__ . '/_bootstrap.php';
require __DIR__ . '/_actor.php';   // current_updated_by()
require __DIR__ . '/_audit.php';   // audit()

header('Content-Type: application/json; charset=utf-8');

// --- auth guard (same as your original) ---
if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
$authed = (isset($_SESSION['login_type']) && (isset($_SESSION['admin_id']) || isset($_SESSION['user_id'])));
if (!$authed) { http_response_code(403); echo json_encode(['error'=>'forbidden']); exit; }

$id = (int)($_POST['id'] ?? 0);
if ($id <= 0) { http_response_code(400); echo json_encode(['success'=>false,'error'=>'Missing id']); exit; }

$by = current_updated_by($pdo); // e.g., ADMIN-REF or USER-REF

try{
  // Fetch target
  $stmt = $pdo->prepare("SELECT `id`,`type`,`enabled` FROM `notification_sounds` WHERE `id`=:id LIMIT 1");
  $stmt->execute([':id'=>$id]);
  $target = $stmt->fetch(PDO::FETCH_ASSOC);
  if (!$target) { throw new RuntimeException('Not found.'); }

  $type        = (string)$target['type'];
  $wasEnabled  = (int)$target['enabled'] === 1;

  // Find any currently-active rows for same type (excluding target)
  $stPrev = $pdo->prepare("SELECT `id` FROM `notification_sounds` WHERE `type`=:t AND `enabled`=1 AND `id`<>:id");
  $stPrev->execute([':t'=>$type, ':id'=>$id]);
  $prevActives = $stPrev->fetchAll(PDO::FETCH_COLUMN, 0);

  $pdo->beginTransaction();

  // Disable all previous actives (and audit each)
  if (!empty($prevActives)) {
    $updOff = $pdo->prepare("UPDATE `notification_sounds` SET `enabled`=0 WHERE `id`=:pid");
    foreach ($prevActives as $pid) {
      $updOff->execute([':pid' => (int)$pid]);
      audit($pdo, 'notification_sounds', (int)$pid, 'enabled', '1', '0', $by, 'set-active (single-active rule)');
    }
  }

  // Enable target if not already enabled
  if (!$wasEnabled) {
    $pdo->prepare("UPDATE `notification_sounds` SET `enabled`=1 WHERE `id`=:id")->execute([':id'=>$id]);
    audit($pdo, 'notification_sounds', (int)$id, 'enabled', '0', '1', $by, 'set-active (manual)');
  } else {
    // It was already enabled — still audit the intent for traceability (optional)
    audit($pdo, 'notification_sounds', (int)$id, 'enabled', '1', '1', $by, 'set-active (no-op; already active)');
  }

  $pdo->commit();

  echo json_encode([
    'success'        => true,
    'type'           => $type,
    'activated_id'   => (int)$id,
    'deactivated'    => array_map('intval', $prevActives),
    'message'        => 'Active sound set for type: '.$type
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(400);
  echo json_encode(['success'=>false,'error'=>'DB error: '.$e->getMessage()]);
}
