<?php
// api/notification_sounds_delete.php — soft-delete a sound (+audit) and remove its file if in /assets/sounds/custom
declare(strict_types=1);

require __DIR__ . '/_bootstrap.php';
require __DIR__ . '/_actor.php';  // current_updated_by()
require __DIR__ . '/_audit.php';  // audit()

header('Content-Type: application/json; charset=utf-8');

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
$authed = (isset($_SESSION['login_type']) && (isset($_SESSION['admin_id']) || isset($_SESSION['user_id'])));
if (!$authed) {
  http_response_code(403);
  echo json_encode(['success'=>false,'error'=>'forbidden']);
  exit;
}

$id = (int)($_POST['id'] ?? 0);
if ($id <= 0) {
  http_response_code(400);
  echo json_encode(['success'=>false,'error'=>'Invalid id']);
  exit;
}

try {
  // Fetch current row
  $st = $pdo->prepare("SELECT id, type, label, src, volume, `loop`, enabled, is_deleted FROM notification_sounds WHERE id = :id LIMIT 1");
  $st->execute([':id'=>$id]);
  $row = $st->fetch(PDO::FETCH_ASSOC);
  if (!$row) {
    throw new RuntimeException('Not found.');
  }

  // If already deleted, treat as success (idempotent)
  if ((int)$row['is_deleted'] === 1) {
    echo json_encode(['success'=>true, 'already_deleted'=>true]);
    exit;
  }

  $by = current_updated_by($pdo);

  // Soft delete + disable if needed
  $pdo->beginTransaction();

  $upd = $pdo->prepare("UPDATE notification_sounds SET is_deleted = 1, enabled = 0 WHERE id = :id");
  $upd->execute([':id'=>$id]);

  // Track changes
  audit($pdo, 'notification_sounds', $id, 'is_deleted', (string)$row['is_deleted'], '1', $by, 'soft-delete');
  if ((int)$row['enabled'] === 1) {
    audit($pdo, 'notification_sounds', $id, 'enabled', '1', '0', $by, 'soft-delete');
  }

  $pdo->commit();

  // Remove file only if it's inside our custom folder
  $src = (string)$row['src'];
  $adminRoot  = realpath(dirname(__DIR__)); // .../ezylend-admin
  $customDir  = ($adminRoot !== false) ? realpath($adminRoot . '/assets/sounds/custom') : false;

  // Build absolute path from stored src:
  //  - if src is absolute like "/something", remove leading slash and prepend admin root
  //  - if src is relative like "assets/..", prepend admin root
  $absCandidate = null;
  if ($adminRoot !== false) {
    $absCandidate = realpath($adminRoot . '/' . ltrim($src, '/'));
  }

  // Delete only if file exists and is located under the customDir
  if ($customDir && $absCandidate && is_file($absCandidate) && strpos($absCandidate, $customDir) === 0) {
    @unlink($absCandidate);
  }

  echo json_encode(['success'=>true]);

} catch (Throwable $e) {
  if ($pdo->inTransaction()) $pdo->rollBack();
  http_response_code(400);
  echo json_encode(['success'=>false, 'error'=>$e->getMessage()]);
}
