<?php
// api/notification_save.php
declare(strict_types=1);
require __DIR__.'/_bootstrap.php';
header('Content-Type: application/json; charset=utf-8');

/**
 * Accepts multipart/form-data (recommended) with fields:
 *   notification_id (optional for update)
 *   category (required)
 *   audience (backend|consumer, required)
 *   title (required)
 *   body (optional)
 *   ref_table (optional)
 *   ref_id (optional, int)
 *   is_active (0|1)
 *   image_file (optional PNG/JPG, <= 3 MB)
 *
 * Saves image to /uploads/notifications and returns relative path in image_path.
 */

if ($_SERVER['REQUEST_METHOD'] !== 'POST') { json_out(['error' => 'POST required'], 405); }

// Support JSON payload too (without image)
if (stripos($_SERVER['CONTENT_TYPE'] ?? '', 'application/json') !== false) {
  $j = json_decode(file_get_contents('php://input'), true) ?: [];
  foreach ($j as $k => $v) { $_POST[$k] = $v; }
}

@session_start();
$updated_by = $_SESSION['username'] ?? $_SESSION['lastname'] ?? ($_SESSION['admin_id'] ?? $_SESSION['user_id'] ?? null);
if (is_array($updated_by)) $updated_by = json_encode($updated_by);

$id        = (int)($_POST['notification_id'] ?? 0);
$category  = trim((string)($_POST['category'] ?? ''));
$audience  = strtolower(trim((string)($_POST['audience'] ?? '')));
$title     = trim((string)($_POST['title'] ?? ''));
$body      = trim((string)($_POST['body'] ?? ''));
$ref_table = trim((string)($_POST['ref_table'] ?? ''));
$ref_id    = ($_POST['ref_id'] ?? '') === '' ? null : (int)$_POST['ref_id'];
$is_active = (int)($_POST['is_active'] ?? 0);
$is_active = ($is_active === 1) ? 1 : 0;

if ($category === '' || $title === '') { json_out(['error' => 'Category and Title are required'], 400); }
if (!in_array($audience, ['backend','consumer'], true)) { json_out(['error' => 'Audience must be backend or consumer'], 400); }

$uploadRelDir = 'uploads/notifications';
$uploadAbsDir = dirname(__DIR__) . '/'.$uploadRelDir;
ensure_dir($uploadAbsDir);

$newImagePath = null;

// Validate & move uploaded image (if present)
if (isset($_FILES['image_file']) && is_array($_FILES['image_file']) && ($_FILES['image_file']['error'] !== UPLOAD_ERR_NO_FILE)) {
  $err = $_FILES['image_file']['error'];
  if ($err !== UPLOAD_ERR_OK) {
    json_out(['error' => 'Upload error', 'detail' => (string)$err], 400);
  }
  $tmp  = $_FILES['image_file']['tmp_name'];
  $name = $_FILES['image_file']['name'];
  $type = mime_content_type($tmp) ?: '';

  $allowed = ['image/jpeg'=>'jpg', 'image/jpg'=>'jpg', 'image/png'=>'png'];
  if (!isset($allowed[$type])) { json_out(['error' => 'Only JPG/PNG allowed'], 400); }
  if (filesize($tmp) > 3*1024*1024) { json_out(['error' => 'Image must be ≤ 3MB'], 400); }

  $ext = $allowed[$type];
  $safeBase = pathinfo(safename($name), PATHINFO_FILENAME);
  $fname = $safeBase.'-'.bin2hex(random_bytes(5)).'.'.$ext;
  $destAbs = $uploadAbsDir.'/'.$fname;

  if (!move_uploaded_file($tmp, $destAbs)) {
    json_out(['error' => 'Failed to store image'], 500);
  }
  $newImagePath = $uploadRelDir.'/'.$fname; // relative path for DB
}

// If updating and no new image uploaded, keep existing path
$existing = null;
if ($id > 0) {
  $st = $pdo->prepare("SELECT image_path FROM notifications WHERE notification_id = :id");
  $st->execute([':id'=>$id]);
  $existing = $st->fetch() ?: null;
}

try {
  if ($id > 0) {
    // UPDATE
    $sql = "UPDATE notifications
            SET category = :c, audience = :a, title = :t, body = :b,
                image_path = :img, ref_table = :rt, ref_id = :rid,
                is_active = :ia, updated_by = :ub
            WHERE notification_id = :id";
    $st = $pdo->prepare($sql);
    $st->execute([
      ':c'=>$category,
      ':a'=>$audience,
      ':t'=>$title,
      ':b'=>$body,
      ':img'=>($newImagePath ?? ($existing['image_path'] ?? null)),
      ':rt'=>($ref_table !== '' ? $ref_table : null),
      ':rid'=>$ref_id,
      ':ia'=>$is_active,
      ':ub'=>$updated_by,
      ':id'=>$id
    ]);
    json_out(['success'=>true, 'mode'=>'update', 'notification_id'=>$id, 'image_path'=>($newImagePath ?? ($existing['image_path'] ?? null))]);
  } else {
    // INSERT
    $sql = "INSERT INTO notifications
              (category, audience, title, body, image_path, ref_table, ref_id, is_active, created_at, updated_by)
            VALUES
              (:c, :a, :t, :b, :img, :rt, :rid, :ia, NOW(), :ub)";
    $st = $pdo->prepare($sql);
    $st->execute([
      ':c'=>$category,
      ':a'=>$audience,
      ':t'=>$title,
      ':b'=>$body,
      ':img'=>$newImagePath,
      ':rt'=>($ref_table !== '' ? $ref_table : null),
      ':rid'=>$ref_id,
      ':ia'=>$is_active,
      ':ub'=>$updated_by
    ]);
    $newId = (int)$pdo->lastInsertId();
    json_out(['success'=>true, 'mode'=>'insert', 'notification_id'=>$newId, 'image_path'=>$newImagePath]);
  }
} catch (Throwable $e) {
  json_out(['error'=>'Save failed', 'detail'=>$e->getMessage()], 500);
}
