<?php
// api/new_lending_requests.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
header('Content-Type: application/json; charset=utf-8');

/**
 * Query params:
 *   limit     : int (1..50), default 20
 *   since     : minutes, default 7 days (10080). Set 0 to disable time filter.
 *   statuses  : optional CSV (e.g. Applied,Pending). Defaults to Applied,Pending
 */

$limit    = max(1, min(50, (int)($_GET['limit'] ?? 20)));
$sinceMin = (int)($_GET['since'] ?? (60*24*7)); // default 7 days
$params   = [];

// Status filter: treat Applied/Pending as "new requests" by default
$givenStatuses = isset($_GET['statuses'])
  ? array_filter(array_map('trim', explode(',', (string)$_GET['statuses'])))
  : [];
$statuses = $givenStatuses ?: ['Applied', 'Pending'];

$statusPlaceholders = implode(',', array_fill(0, count($statuses), '?'));
$params = array_merge($params, $statuses);

// Time filter: based on created_at ONLY
$sinceSql = '';
if ($sinceMin > 0) {
  $sinceAt = date('Y-m-d H:i:s', time() - ($sinceMin * 60));
  $sinceSql = " AND la.created_at >= ? ";
  $params[] = $sinceAt;
}

// Build SQL (LIMIT is already sanitized int)
$sql = "
  SELECT
    la.application_id,
    la.application_ref_id,
    la.customer_id,
    c.prefix,
    c.first_name,
    c.second_name,
    lt.name AS loan_type,
    la.loan_amount,
    la.application_status AS status,
    la.created_at,
    NULL AS image_path  -- keep key; UI falls back to a local icon when null
  FROM loan_applications la
  LEFT JOIN customers c ON c.customer_id = la.customer_id
  LEFT JOIN loantype  lt ON lt.loan_type_id = la.loantype
  WHERE la.application_status IN ($statusPlaceholders)
  $sinceSql
  ORDER BY la.created_at DESC, la.application_id DESC
  LIMIT {$limit}
";

try {
  $st = $pdo->prepare($sql);
  $st->execute($params);
  $rows = $st->fetchAll();

  // Normalize numeric fields for the frontend
  foreach ($rows as &$r) {
    if (isset($r['loan_amount']) && is_numeric($r['loan_amount'])) {
      $r['loan_amount'] = (float)$r['loan_amount'];
    }
  }

  echo json_encode($rows, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'Server error']);
}
