<?php
// api/fetchinstantlenddata.php
declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

$host = getenv('DB_HOST') ?: 'localhost';
$db   = getenv('DB_NAME') ?: 'ezy';
$user = getenv('DB_USER') ?: 'root';
$pass = getenv('DB_PASS') ?: '';
$dsn  = "mysql:host={$host};dbname={$db};charset=utf8mb4";

try {
    // If you already have a shared PDO in includes/db.php, require it instead.
    // require_once __DIR__ . '/../includes/db.php'; // sets $pdo
    if (!isset($pdo)) {
        $pdo = new PDO($dsn, $user, $pass, [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
    }

    // loantype=2 is INSTANT; also guard by name in case IDs change.
    $sql = "
        SELECT
            la.application_id, la.application_ref_id,
            la.customer_id,
            CONCAT(COALESCE(c.first_name, ''), ' ', COALESCE(c.second_name, '')) AS client_name,
            lt.name AS loan_type,
            la.loan_amount,
            la.application_status AS status,
            la.created_at AS createdat,
            COALESCE(la.updated_date, la.created_at) AS status_date
        FROM loan_applications la
        LEFT JOIN customers c    ON c.customer_id = la.customer_id
        LEFT JOIN loantype  lt   ON lt.loan_type_id = la.loantype
        WHERE la.loantype = 2 OR lt.name = 'INSTANT'
        ORDER BY la.created_at DESC, la.application_id DESC
    ";

    $rows = $pdo->query($sql)->fetchAll();

    foreach ($rows as &$r) {
        // numeric for frontend formatting
        if (isset($r['loan_amount']) && is_numeric($r['loan_amount'])) {
            $r['loan_amount'] = (float)$r['loan_amount'];
        } else {
            $r['loan_amount'] = null;
        }
        // fallback name
        if (!isset($r['client_name']) || trim($r['client_name']) === '') {
            $r['client_name'] = 'Unknown';
        }
    }

    echo json_encode($rows, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error', 'detail' => $e->getMessage()]);
}
