<?php
/**
 * api/fetchalllenddata.php
 * JSON feed for All Lend Details DataTable
 */

declare(strict_types=1);
header('Content-Type: application/json; charset=utf-8');

// --- DB connection (use your own include if you have one) ---
$host = getenv('DB_HOST') ?: 'localhost';
$db   = getenv('DB_NAME') ?: 'ezy';
$user = getenv('DB_USER') ?: 'root';
$pass = getenv('DB_PASS') ?: '';
$dsn  = "mysql:host={$host};dbname={$db};charset=utf8mb4";

try {
    // If you already have a config file, uncomment:
    // require_once __DIR__ . '/../includes/db.php'; // should set $pdo
    if (!isset($pdo)) {
        $pdo = new PDO($dsn, $user, $pass, [
            PDO::ATTR_ERRMODE            => PDO::ERRMODE_EXCEPTION,
            PDO::ATTR_DEFAULT_FETCH_MODE => PDO::FETCH_ASSOC,
            PDO::ATTR_EMULATE_PREPARES   => false,
        ]);
    }

    /**
     * We pull:
     * - loan_applications: id, amounts, status, created/updated
     * - customers: first/second name
     * - loantype: name
     * status_date prefers updated_date, falls back to created_at
     */
    $sql = "
        SELECT
            la.application_id,la.application_ref_id,
            la.customer_id,
            CONCAT(COALESCE(c.first_name,''),' ',COALESCE(c.second_name,'')) AS client_name,
            lt.name AS loan_type,
            la.loan_amount,
            la.application_status AS status,
            la.created_at AS createdat,
            COALESCE(la.updated_date, la.created_at) AS status_date
        FROM loan_applications la
        LEFT JOIN customers c   ON c.customer_id = la.customer_id
        LEFT JOIN loantype  lt  ON lt.loan_type_id = la.loantype
        ORDER BY la.created_at DESC, la.application_id DESC
    ";

    $stmt = $pdo->query($sql);
    $rows = $stmt->fetchAll();

    // Ensure numeric values where reasonable
    foreach ($rows as &$r) {
        if (isset($r['loan_amount'])) {
            $r['loan_amount'] = is_numeric($r['loan_amount']) ? (float)$r['loan_amount'] : null;
        }
        // Normalize empty names to 'Unknown' (optional)
        if (isset($r['client_name']) && trim($r['client_name']) === '') {
            $r['client_name'] = 'Unknown';
        }
    }

    echo json_encode($rows, JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);
} catch (Throwable $e) {
    http_response_code(500);
    echo json_encode(['error' => 'Server error', 'detail' => $e->getMessage()]);
    // You can log $e->getMessage() safely on server
}
