<?php
// api/dashboard_stats.php
declare(strict_types=1);
require __DIR__.'/_bootstrap.php';
header('Content-Type: application/json; charset=utf-8');

/**
 * Returns:
 * {
 *   "today_loans": 123,
 *   "total_instant_lends": 456,
 *   "total_minutes_lends": 789,
 *   "num_clients": 1011,
 *   "visits_today": 12,
 *   "visits_month": 345
 * }
 *
 * Assumptions:
 * - loan_applications(created_at, loantype -> loantype.loan_type_id)
 * - loantype(name)
 * - customers(...)
 * - OPTIONAL: page_visits(visited_at DATETIME). If it doesn't exist, visits_* = 0.
 */

function table_exists(PDO $pdo, string $table): bool {
  $q = $pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $q->execute([$table]);
  return (bool)$q->fetchColumn();
}

try {
  // Today’s loans
  $st = $pdo->query("SELECT COUNT(*) FROM loan_applications WHERE DATE(created_at)=CURDATE()");
  $today_loans = (int)$st->fetchColumn();

  // Totals by type (Instant / Minutes)
  $st2 = $pdo->query("
    SELECT
      SUM(CASE WHEN UPPER(COALESCE(lt.name,'')) LIKE '%INSTANT%' THEN 1 ELSE 0 END) AS total_instant,
      SUM(CASE WHEN UPPER(COALESCE(lt.name,'')) LIKE '%MINUTE%'  THEN 1 ELSE 0 END) AS total_minutes
    FROM loan_applications la
    LEFT JOIN loantype lt ON lt.loan_type_id = la.loantype
  ");
  $row2 = $st2->fetch();
  $total_instant = (int)($row2['total_instant'] ?? 0);
  $total_minutes = (int)($row2['total_minutes'] ?? 0);

  // Number of clients
  $st3 = $pdo->query("SELECT COUNT(*) FROM customers");
  $num_clients = (int)$st3->fetchColumn();

  // Page visits (optional table)
  $visits_today = 0;
  $visits_month = 0;
  if (table_exists($pdo, 'page_visits')) {
    $st4 = $pdo->query("SELECT COUNT(*) FROM page_visits WHERE DATE(visited_at)=CURDATE()");
    $visits_today = (int)$st4->fetchColumn();

    $st5 = $pdo->query("
      SELECT COUNT(*) FROM page_visits
      WHERE YEAR(visited_at)=YEAR(CURDATE()) AND MONTH(visited_at)=MONTH(CURDATE())
    ");
    $visits_month = (int)$st5->fetchColumn();
  }

  echo json_encode([
    'today_loans'          => $today_loans,
    'total_instant_lends'  => $total_instant,
    'total_minutes_lends'  => $total_minutes,
    'num_clients'          => $num_clients,
    'visits_today'         => $visits_today,
    'visits_month'         => $visits_month,
  ], JSON_UNESCAPED_UNICODE | JSON_UNESCAPED_SLASHES);

} catch (Throwable $e) {
  http_response_code(500);
  echo json_encode(['error' => 'Server error']);
}
