<?php
// api/business_documents_upload.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_upload_helpers.php'; // your existing helpers (allowed_file, save_upload, etc.)
require __DIR__.'/_audit.php';          // provides audit($pdo, $table, $refId, $field, $old, $new, $by, $reason)
require __DIR__.'/_actor.php';          // provides current_updated_by()

header('Content-Type: application/json; charset=utf-8');

/**
 * Accepts either a numeric application_id or an alphanumeric application_ref_id.
 */
function read_app_key(): string {
  foreach ([
    $_REQUEST['application_id']     ?? null,
    $_REQUEST['application_ref_id'] ?? null,
  ] as $v) {
    if ($v !== null && $v !== '') return (string)$v;
  }
  return '';
}

/**
 * Resolve app key → (application_id, customer_id)
 * Throws RuntimeException on failure.
 */
function resolve_app(PDO $pdo, string $appKey): array {
  if ($appKey === '') throw new RuntimeException('missing application_id');
  if (ctype_digit($appKey)) {
    $q = $pdo->prepare("SELECT application_id, customer_id FROM loan_applications WHERE application_id=? LIMIT 1");
    $q->execute([(int)$appKey]);
  } else {
    $q = $pdo->prepare("SELECT application_id, customer_id FROM loan_applications WHERE application_ref_id=? LIMIT 1");
    $q->execute([$appKey]);
  }
  $r = $q->fetch();
  if (!$r) throw new RuntimeException('application not found');
  return [(int)$r['application_id'], (int)$r['customer_id']];
}

try {
  $appKey = trim(read_app_key());
  [$aid, $cidFromDb] = resolve_app($pdo, $appKey);
  $cid = (int)($_REQUEST['customer_id'] ?? 0) ?: $cidFromDb;
  if ($cid <= 0) throw new RuntimeException('invalid customer_id');

  $actor = current_updated_by();

  $uploaded = 0;
  $hadFiles = false;

  // Loop expected business file fields: biz_doc1..biz_doc5
  for ($i = 1; $i <= 5; $i++) {
    $fld = "biz_doc{$i}";
    if (!isset($_FILES[$fld]) || (int)$_FILES[$fld]['error'] === UPLOAD_ERR_NO_FILE) {
      continue;
    }
    $hadFiles = true;

    // Guard against re-upload if latest is pending/verified and not expired.
    $last = latest_state_subtype($pdo, $cid, $aid, 'business', $fld);
    [$ok, $why] = can_upload_row($last);
    if (!$ok) {
      throw new RuntimeException("{$fld} blocked: {$why}");
    }

    // Save upload (customer_documents INSERT). Helper does the move + insert.
    save_upload($pdo, $cid, $aid, 'business', $_FILES[$fld], $fld);

    // Track change: reference_id = new document id
    $docId = (int)$pdo->lastInsertId();
    if ($docId > 0) {
      // Read back the row to capture values for logging
      $sel = $pdo->prepare("SELECT file_path, original_name FROM customer_documents WHERE id=?");
      $sel->execute([$docId]);
      $row = $sel->fetch() ?: ['file_path' => null, 'original_name' => null];

      // One concise audit entry per new file
      $newVal = $row['file_path'] ?: ($row['original_name'] ?: "(doc {$docId})");
      $reason = "uploaded {$fld} (business document)";
      audit($pdo, 'customer_documents', $docId, 'file_path', null, $newVal, $actor, $reason);
    }

    $uploaded++;
  }

  if (!$hadFiles) {
    throw new RuntimeException('no files received');
  }

  echo json_encode(['status' => 'success', 'message' => "Uploaded {$uploaded} file(s)"]);
} catch (Throwable $e) {
  http_response_code(400);
  echo json_encode(['status' => 'error', 'message' => $e->getMessage()]);
}
