<?php
// api/business_details_save.php
declare(strict_types=1);

require __DIR__.'/_bootstrap.php';
require __DIR__.'/_audit.php';
require __DIR__.'/_actor.php';

header('Content-Type: application/json; charset=utf-8');

$in    = json_decode(file_get_contents('php://input'), true) ?: [];
$appId = (int)($in['loan_application_id'] ?? 0);
if ($appId <= 0) json_out(['ok'=>false,'message'=>'Invalid loan_application_id'], 400);

// Resolve customer_id from application
$st = $pdo->prepare("SELECT application_id, customer_id FROM loan_applications WHERE application_id=? LIMIT 1");
$st->execute([$appId]);
$app = $st->fetch();
if (!$app) json_out(['ok'=>false,'message'=>'Application not found'], 404);
$customerId = (int)$app['customer_id'];

// Directors: expect array of {name, number}; normalize and JSON encode
$rawDirs = $in['company_directors'] ?? [];
if (is_string($rawDirs)) {
  $tmp = json_decode($rawDirs, true);
  $rawDirs = is_array($tmp) ? $tmp : [];
}
$directors = [];
if (is_array($rawDirs)) {
  foreach ($rawDirs as $d) {
    $name   = trim((string)($d['name']   ?? ''));
    $number = trim((string)($d['number'] ?? ''));
    if ($name==='' && $number==='') continue; // skip empty row
    $directors[] = ['name'=>$name,'number'=>$number];
  }
}
$directorsJson = json_encode($directors, JSON_UNESCAPED_UNICODE);

// Other fields
$payload = [
  'customer_id'         => $customerId,
  'loan_application_id' => $appId,
  'business_type'       => trim((string)($in['business_type'] ?? '')),
  'company_number'      => trim((string)($in['company_number'] ?? '')),
  'company_name'        => trim((string)($in['company_name'] ?? '')),
  'company_address'     => trim((string)($in['company_address'] ?? '')),
  'company_directors'   => $directorsJson,
  'vat_number'          => trim((string)($in['vat_number'] ?? '')),
  'industry_type'       => trim((string)($in['industry_type'] ?? '')),
];

// Basic validation
$errs = [];
if ($payload['company_name'] === '')  $errs[] = 'Company name is required.';
if ($payload['business_type'] === '') $errs[] = 'Business type is required.';
if ($errs) json_out(['ok'=>false,'message'=>implode("\n",$errs)], 422);

$actor = current_updated_by();

// Fetch existing row (by app)
$sel = $pdo->prepare("SELECT * FROM business_details WHERE loan_application_id=? LIMIT 1");
$sel->execute([$appId]);
$old = $sel->fetch();

$pdo->beginTransaction();
try {
  if ($old) {
    $fields = [
      'business_type','company_number','company_name','company_address',
      'company_directors','vat_number','industry_type'
    ];
    $changed = [];
    foreach ($fields as $f) {
      $ov = (string)($old[$f] ?? '');
      $nv = (string)($payload[$f]);
      if ($ov !== $nv) $changed[$f] = [$ov, $nv];
    }

    if (!empty($changed)) {
      $upd = $pdo->prepare("
        UPDATE business_details
           SET business_type=?, company_number=?, company_name=?, company_address=?,
               company_directors=?, vat_number=?, industry_type=?, updated_at=NOW()
         WHERE business_details_id=? LIMIT 1
      ");
      $upd->execute([
        $payload['business_type'],
        $payload['company_number'],
        $payload['company_name'],
        $payload['company_address'],
        $payload['company_directors'],
        $payload['vat_number'],
        $payload['industry_type'],
        (int)$old['business_details_id'],
      ]);

      foreach ($changed as $field => [$ov,$nv]) {
        audit($pdo, 'business_details', (int)$old['business_details_id'], $field, $ov, $nv, $actor, 'business details updated');
      }
    }
    $pdo->commit();
    json_out(['ok'=>true,'message'=> empty($changed) ? 'No changes' : 'Updated successfully']);

  } else {
    // Insert
    $ins = $pdo->prepare("
      INSERT INTO business_details
        (customer_id, loan_application_id, business_type, company_number, company_name,
         company_address, company_directors, vat_number, industry_type, created_at)
      VALUES (?,?,?,?,?,?,?,?,?,NOW())
    ");
    $ins->execute([
      $payload['customer_id'],
      $payload['loan_application_id'],
      $payload['business_type'],
      $payload['company_number'],
      $payload['company_name'],
      $payload['company_address'],
      $payload['company_directors'],
      $payload['vat_number'],
      $payload['industry_type'],
    ]);
    $newId = (int)$pdo->lastInsertId();

    // Track initial non-empty values
    foreach ([
      'business_type','company_number','company_name','company_address',
      'company_directors','vat_number','industry_type'
    ] as $f) {
      $nv = (string)$payload[$f];
      if ($nv !== '') audit($pdo, 'business_details', $newId, $f, null, $nv, $actor, 'business details created');
    }

    $pdo->commit();
    json_out(['ok'=>true,'message'=>'Created successfully','business_details_id'=>$newId]);
  }
} catch (Throwable $e) {
  $pdo->rollBack();
  json_out(['ok'=>false,'message'=>'Database error'], 500);
}
