<?php
// api/acl.php
declare(strict_types=1);

require_once __DIR__.'/_bootstrap.php';
require_once __DIR__.'/_actor.php';
require_once __DIR__.'/rbac.php';

if (session_status() !== PHP_SESSION_ACTIVE) { @session_start(); }
if (!headers_sent()) { header('Content-Type: application/json; charset=utf-8'); }

// ---- helpers ----
function read_raw_json(): array {
  static $buf = null, $json = null;
  if ($buf === null)   { $buf = file_get_contents('php://input') ?: ''; }
  if ($json === null)  { $json = $buf !== '' ? json_decode($buf, true) : []; }
  return is_array($json) ? $json : [];
}
function ok($data = []){ echo json_encode(['ok'=>true] + $data, JSON_UNESCAPED_UNICODE|JSON_UNESCAPED_SLASHES); exit; }
function bad($msg='bad_request', $code=400){ http_response_code($code); echo json_encode(['ok'=>false,'error'=>$msg]); exit; }
function tbl_exists(PDO $pdo, string $t): bool{
  $q=$pdo->prepare("SELECT COUNT(*) FROM INFORMATION_SCHEMA.TABLES WHERE TABLE_SCHEMA = DATABASE() AND TABLE_NAME = ?");
  $q->execute([$t]); return (bool)$q->fetchColumn();
}
function ints(array $a): array { return array_values(array_unique(array_map('intval',$a))); }
function clean_code(string $s): string { return preg_replace('/[^A-Za-z0-9:_-]/','', $s); }
function norm_ref($s): string { $s = trim((string)$s); $s = strtolower($s); return ($s===''||$s==='null'||$s==='undefined') ? '' : (string)$s; }
function norm_who($w): string { return strtolower($w)==='user' ? 'user' : 'admin'; }
function ref_from_id(PDO $pdo, string $who, int $id): string {
  if ($id<=0) return '';
  if ($who==='admin'){
    $st=$pdo->prepare("SELECT COALESCE(NULLIF(admin_ref_Id,''), username, CAST(admin_id AS CHAR)) FROM admin WHERE admin_id=? LIMIT 1");
    $st->execute([$id]); return (string)($st->fetchColumn() ?: '');
  } else {
    if (!tbl_exists($pdo,'users')) return '';
    $st=$pdo->prepare("SELECT COALESCE(NULLIF(user_ref_Id,''), username, CAST(user_id AS CHAR)) FROM users WHERE user_id=? LIMIT 1");
    $st->execute([$id]); return (string)($st->fetchColumn() ?: '');
  }
}

$JSON   = read_raw_json();
$action = strtolower(trim($_GET['action'] ?? ($_POST['action'] ?? ($JSON['action'] ?? ''))));

// ---------- FEATURES ----------
if ($action === 'features') {
  ok([
    'can_update' => can('settings:update'),
    'has_users'  => tbl_exists($pdo,'users'),
  ]);
}

// ---------- ROLES ----------
if ($action === 'roles_list') {
  $rows = $pdo->query("SELECT role_id, code, name FROM roles ORDER BY name")->fetchAll();
  ok(['roles'=>$rows]);
}

if ($action === 'role_create' && ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
  guard('settings:update');
  $code = clean_code($JSON['code'] ?? '');
  $name = trim((string)($JSON['name'] ?? ''));
  if ($code==='' || $name==='') bad('code_or_name_required',422);

  $st = $pdo->prepare("SELECT role_id FROM roles WHERE code=? LIMIT 1");
  $st->execute([$code]);
  if ($st->fetch()) bad('duplicate_code',409);

  $st = $pdo->prepare("INSERT INTO roles (code,name) VALUES (?,?)");
  $st->execute([$code,$name]);
  ok(['role'=>['role_id'=>(int)$pdo->lastInsertId(),'code'=>$code,'name'=>$name]]);
}

if ($action === 'perms_list') {
  $rows = $pdo->query("SELECT perm_id, code, COALESCE(label,code) AS label, COALESCE(group_code,'other') AS grp
                       FROM permissions ORDER BY grp, label")->fetchAll();
  $groups = [];
  foreach ($rows as $r) { $groups[$r['grp']][] = $r; }
  ok(['permissions'=>$rows,'groups'=>$groups]);
}

if ($action === 'role_perms_get') {
  $rid = (int)($_GET['role_id'] ?? 0);
  if ($rid<=0) bad('role_id');
  $st = $pdo->prepare("SELECT perm_id FROM role_permissions WHERE role_id=? AND allow=1");
  $st->execute([$rid]);
  ok(['perm_ids'=>array_map('intval', array_column($st->fetchAll(), 'perm_id'))]);
}

if ($action === 'role_perms_save' && ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
  guard('settings:update');
  $rid = (int)($JSON['role_id'] ?? 0);
  $perm_ids = ints($JSON['perm_ids'] ?? []);
  if ($rid<=0) bad('role_id');

  $pdo->prepare("DELETE FROM role_permissions WHERE role_id=?")->execute([$rid]);
  if ($perm_ids) {
    $ins = $pdo->prepare("INSERT IGNORE INTO role_permissions (role_id, perm_id, allow) VALUES (?, ?, 1)");
    foreach ($perm_ids as $pid) $ins->execute([$rid, $pid]);
  }
  ok();
}

// ---------- PEOPLE LIST ----------
if ($action === 'people_list') {
  $who = norm_who($_GET['who'] ?? 'admin');

  if ($who === 'admin') {
    $q = "SELECT a.admin_id AS id,
                 COALESCE(NULLIF(a.admin_ref_Id,''), a.username, CAST(a.admin_id AS CHAR)) AS ref,
                 a.username,
                 a.`type`,
                 a.`status`
          FROM admin a
          ORDER BY a.admin_id DESC
          LIMIT 500";
    $rows = $pdo->query($q)->fetchAll();
    ok(['rows'=>$rows]);
  } else {
    if (!tbl_exists($pdo,'users')) ok(['rows'=>[]]);
    $q = "SELECT u.user_id AS id,
                 COALESCE(NULLIF(u.user_ref_Id,''), u.username, CAST(u.user_id AS CHAR)) AS ref,
                 u.username,
                 u.user_type AS `type`,
                 TRIM(CONCAT(COALESCE(s.prefix,''),' ',COALESCE(s.first_name,''),' ',COALESCE(s.second_name,''))) AS staff_name
          FROM users u
          LEFT JOIN staff s
              ON s.staff_ref_id = u.user_ref_Id
             OR (s.email IS NOT NULL AND s.email <> '' AND s.email = u.username)
          ORDER BY u.user_id DESC
          LIMIT 500";
    $rows = $pdo->query($q)->fetchAll();
    ok(['rows'=>$rows]);
  }
}

// ---------- PERSON ROLES ----------
if ($action === 'person_roles_get') {
  $who = norm_who($_GET['who'] ?? 'admin');
  $ref = norm_ref($_GET['ref'] ?? '');
  $id  = (int)($_GET['id'] ?? 0);
  if ($ref==='') { $ref = ref_from_id($pdo,$who,$id); }
  if ($ref==='') bad('ref');

  if ($who === 'admin') {
    $st = $pdo->prepare("SELECT role_id FROM admin_roles WHERE admin_ref_id=?");
  } else {
    if (tbl_exists($pdo,'user_roles')) $st = $pdo->prepare("SELECT role_id FROM user_roles WHERE user_ref_id=?");
    else                               $st = $pdo->prepare("SELECT role_id FROM admin_roles WHERE admin_ref_id=?"); // fallback by ref
  }
  $st->execute([$ref]);
  ok(['role_ids'=>array_map('intval', array_column($st->fetchAll(), 'role_id'))]);
}

if ($action === 'person_roles_save' && ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
  guard('settings:update');
  $who  = norm_who($JSON['who'] ?? 'admin');
  $ref  = norm_ref($JSON['ref'] ?? '');
  $id   = (int)($JSON['id'] ?? 0);
  $role_ids = ints($JSON['role_ids'] ?? []);
  if ($ref==='') { $ref = ref_from_id($pdo,$who,$id); }
  if ($ref==='') bad('ref');

  if ($who === 'admin') {
    $pdo->prepare("DELETE FROM admin_roles WHERE admin_ref_id=?")->execute([$ref]);
    if ($role_ids) {
      $ins = $pdo->prepare("INSERT IGNORE INTO admin_roles (admin_ref_id, role_id) VALUES (?, ?)");
      foreach ($role_ids as $rid) $ins->execute([$ref,$rid]);
    }
  } else {
    if (tbl_exists($pdo,'user_roles')) {
      $pdo->prepare("DELETE FROM user_roles WHERE user_ref_id=?")->execute([$ref]);
      if ($role_ids) {
        $ins = $pdo->prepare("INSERT IGNORE INTO user_roles (user_ref_id, role_id) VALUES (?, ?)");
        foreach ($role_ids as $rid) $ins->execute([$ref,$rid]);
      }
    } else {
      // store by ref into admin_roles as a fallback
      $pdo->prepare("DELETE FROM admin_roles WHERE admin_ref_id=?")->execute([$ref]);
      if ($role_ids) {
        $ins = $pdo->prepare("INSERT IGNORE INTO admin_roles (admin_ref_id, role_id) VALUES (?, ?)");
        foreach ($role_ids as $rid) $ins->execute([$ref,$rid]);
      }
    }
  }
  ok();
}

// ---------- OVERRIDES ----------
if ($action === 'overrides_get') {
  $who = norm_who($_GET['who'] ?? 'admin');
  $ref = norm_ref($_GET['ref'] ?? '');
  $id  = (int)($_GET['id'] ?? 0);
  if ($ref==='') { $ref = ref_from_id($pdo,$who,$id); }
  if ($ref==='') bad('ref');

  if ($who === 'admin') {
    $st = $pdo->prepare("SELECT perm_id, allow FROM admin_permissions WHERE admin_ref_id=?");
  } else {
    if (tbl_exists($pdo,'user_permissions')) {
      $st = $pdo->prepare("SELECT perm_id, allow FROM user_permissions WHERE user_ref_id=?");
    } else {
      // fallback: store user overrides in admin_permissions keyed by ref
      $st = $pdo->prepare("SELECT perm_id, allow FROM admin_permissions WHERE admin_ref_id=?");
    }
  }
  $st->execute([$ref]);
  $allow=$deny=[];
  foreach($st->fetchAll() as $r){
    if ((int)$r['allow']===1) $allow[]=(int)$r['perm_id']; else $deny[]=(int)$r['perm_id'];
  }
  ok(['allow'=>$allow,'deny'=>$deny]);
}

if ($action === 'overrides_save' && ($_SERVER['REQUEST_METHOD'] ?? 'GET') === 'POST') {
  guard('settings:update');
  $who = norm_who($JSON['who'] ?? 'admin');
  $ref = norm_ref($JSON['ref'] ?? '');
  $id  = (int)($JSON['id'] ?? 0);
  $allow = ints($JSON['allow'] ?? []);
  $deny  = ints($JSON['deny']  ?? []);
  if ($ref==='') { $ref = ref_from_id($pdo,$who,$id); }
  if ($ref==='') bad('ref');

  if ($who === 'admin') {
    $pdo->prepare("DELETE FROM admin_permissions WHERE admin_ref_id=?")->execute([$ref]);
    if ($allow){ $ins=$pdo->prepare("INSERT IGNORE INTO admin_permissions (admin_ref_id, perm_id, allow) VALUES (?, ?, 1)"); foreach ($allow as $pid) $ins->execute([$ref,$pid]); }
    if ($deny) { $ins=$pdo->prepare("INSERT IGNORE INTO admin_permissions (admin_ref_id, perm_id, allow) VALUES (?, ?, 0)"); foreach ($deny  as $pid) $ins->execute([$ref,$pid]); }
  } else {
    if (tbl_exists($pdo,'user_permissions')) {
      $pdo->prepare("DELETE FROM user_permissions WHERE user_ref_id=?")->execute([$ref]);
      if ($allow){ $ins=$pdo->prepare("INSERT IGNORE INTO user_permissions (user_ref_id, perm_id, allow) VALUES (?, ?, 1)"); foreach ($allow as $pid) $ins->execute([$ref,$pid]); }
      if ($deny) { $ins=$pdo->prepare("INSERT IGNORE INTO user_permissions (user_ref_id, perm_id, allow) VALUES (?, ?, 0)"); foreach ($deny  as $pid) $ins->execute([$ref,$pid]); }
    } else {
      $pdo->prepare("DELETE FROM admin_permissions WHERE admin_ref_id=?")->execute([$ref]);
      if ($allow){ $ins=$pdo->prepare("INSERT IGNORE INTO admin_permissions (admin_ref_id, perm_id, allow) VALUES (?, ?, 1)"); foreach ($allow as $pid) $ins->execute([$ref,$pid]); }
      if ($deny) { $ins=$pdo->prepare("INSERT IGNORE INTO admin_permissions (admin_ref_id, perm_id, allow) VALUES (?, ?, 0)"); foreach ($deny  as $pid) $ins->execute([$ref,$pid]); }
    }
  }
  ok();
}

bad('unknown_action',404);
