<?php
// api/_actor.php
declare(strict_types=1);

/**
 * Helpers for "who is acting" + role gating.
 * - current_updated_by(): returns business ref (admin.admin_ref_Id or users.user_ref_Id)
 * - current_role_type():  returns admin.type or users.user_type
 * - current_identity():   full bundle (who,id,ref,type)
 *
 * Pass $pdo if you have it; otherwise tries $GLOBALS['pdo'].
 */
function _actor_pdo(?PDO $pdo): ?PDO {
  if ($pdo instanceof PDO) return $pdo;
  return (isset($GLOBALS['pdo']) && $GLOBALS['pdo'] instanceof PDO) ? $GLOBALS['pdo'] : null;
}

function current_identity(?PDO $pdo = null): array {
  if (session_status() === PHP_SESSION_NONE) { @session_start(); }
  $pdo = _actor_pdo($pdo);

  // ADMIN
  if (!empty($_SESSION['login_type']) && $_SESSION['login_type'] === 'admin' && !empty($_SESSION['admin_id'])) {
    $adminId = (int)$_SESSION['admin_id'];

    // cache hit?
    if (!empty($_SESSION['admin_ref_Id']) && isset($_SESSION['admin_type'])) {
      return [
        'who'  => 'admin',
        'id'   => $adminId,
        'ref'  => (string)$_SESSION['admin_ref_Id'],
        'type' => (string)$_SESSION['admin_type'],
      ];
    }

    // fetch once
    if ($pdo) {
      $st = $pdo->prepare("SELECT admin_ref_Id, `type` FROM `admin` WHERE admin_id = ? LIMIT 1");
      $st->execute([$adminId]);
      if ($row = $st->fetch()) {
        $_SESSION['admin_ref_Id'] = (string)($row['admin_ref_Id'] ?? '');
        $_SESSION['admin_type']   = (string)($row['type'] ?? '');
        return [
          'who'  => 'admin',
          'id'   => $adminId,
          'ref'  => (string)$_SESSION['admin_ref_Id'],
          'type' => (string)$_SESSION['admin_type'],
        ];
      }
    }
    // fallback
    return ['who'=>'admin','id'=>$adminId,'ref'=>(string)($_SESSION['admin_ref_Id'] ?? ''),'type'=>(string)($_SESSION['admin_type'] ?? '')];
  }

  // USER
  if (!empty($_SESSION['user_id'])) {
    $userId = (int)$_SESSION['user_id'];

    // cache hit?
    if (!empty($_SESSION['user_ref_Id']) && isset($_SESSION['user_type'])) {
      return [
        'who'  => 'user',
        'id'   => $userId,
        'ref'  => (string)$_SESSION['user_ref_Id'],
        'type' => (string)$_SESSION['user_type'],
      ];
    }

    // fetch once
    if ($pdo) {
      $st = $pdo->prepare("SELECT user_ref_Id, user_type FROM `users` WHERE user_id = ? LIMIT 1");
      $st->execute([$userId]);
      if ($row = $st->fetch()) {
        $_SESSION['user_ref_Id'] = (string)($row['user_ref_Id'] ?? '');
        $_SESSION['user_type']   = (string)($row['user_type'] ?? '');
        return [
          'who'  => 'user',
          'id'   => $userId,
          'ref'  => (string)$_SESSION['user_ref_Id'],
          'type' => (string)$_SESSION['user_type'],
        ];
      }
    }
    // fallback
    return ['who'=>'user','id'=>$userId,'ref'=>(string)($_SESSION['user_ref_Id'] ?? ''),'type'=>(string)($_SESSION['user_type'] ?? '')];
  }

  // SYSTEM
  return ['who'=>'system','id'=>null,'ref'=>'SYSTEM','type'=>null];
}

function current_updated_by(?PDO $pdo = null): string {
  $id = current_identity($pdo);
  return $id['ref'] !== '' ? $id['ref'] : 'SYSTEM';
}

function current_role_type(?PDO $pdo = null): ?string {
  $id = current_identity($pdo);
  return $id['type'] !== '' ? $id['type'] : null;
}

/**
 * Quick gate: allow only certain role types.
 * Example: require_role_type(['superadmin','manager']);
 */
function require_role_type(array $allowed, ?PDO $pdo = null): void {
  $type = current_role_type($pdo);
  if ($type === null || !in_array($type, $allowed, true)) {
    http_response_code(403);
    header('Content-Type: application/json; charset=utf-8');
    echo json_encode(['error'=>'forbidden','message'=>'Not enough privileges']);
    exit;
  }
}
