<?php
include 'includes/session.php';
$pageTitle = "Live Transactions";
$currentPage = "transactions_live";
require_once 'includes/header.php';

// Mock Data for Top Receipt
$receiptItems = [
    ['barcode' => '7393714518903', 'name' => 'Coca-Cola Original Taste 8 x 330ml', 'qty' => 5, 'amount' => '1700.00'],
    ['barcode' => '7393721234561', 'name' => '20x25g Walkers Classic Variety Multipack Crisps', 'qty' => 1, 'amount' => '4500.00'],
    ['barcode' => '1464795343456', 'name' => '172g Ferrero Rocher Collection', 'qty' => 2, 'amount' => '15.00'],
    ['barcode' => '2546564435545', 'name' => 'Haribo Party Box', 'qty' => 10, 'amount' => '800.00'],
    ['barcode' => '5000123456789', 'name' => 'PG Tips Original Tea Bags 80pk', 'qty' => 1, 'amount' => '250.00'],
    ['barcode' => '5012345678901', 'name' => 'McVities Digestives Original 400g', 'qty' => 3, 'amount' => '120.00'],
    ['barcode' => '5023456789012', 'name' => 'Heinz Cream of Tomato Soup 400g', 'qty' => 4, 'amount' => '100.00'],
    ['barcode' => '5034567890123', 'name' => 'Kelloggs Corn Flakes 720g', 'qty' => 1, 'amount' => '350.00'],
    ['barcode' => '5045678901234', 'name' => 'Cadbury Dairy Milk Chocolate Bar 200g', 'qty' => 5, 'amount' => '200.00'],
    ['barcode' => '5056789012345', 'name' => 'Tropicana Orange Juice Smooth 1.5L', 'qty' => 2, 'amount' => '300.00'],
    ['barcode' => '5067890123456', 'name' => 'Hovis Best of Both Bread 800g', 'qty' => 1, 'amount' => '145.00'],
    ['barcode' => '5078901234567', 'name' => 'Nescafe Gold Blend Instant Coffee 200g', 'qty' => 1, 'amount' => '600.00'],
    ['barcode' => '5089012345678', 'name' => 'Fairy Liquid Original Washing Up Liquid 433ml', 'qty' => 2, 'amount' => '150.00'],
    ['barcode' => '5090123456789', 'name' => 'Andrex Toilet Tissue White 9 Roll', 'qty' => 1, 'amount' => '550.00'],
];

?>

<style>
/* Custom Orange Scrollbar to match design */
.custom-scrollbar::-webkit-scrollbar {
    width: 6px;
    height: 6px; /* Horizontal scrollbar height */
    background-color: transparent;
}

.custom-scrollbar::-webkit-scrollbar-track {
    background-color: #f1f5f9; /* Slate-100 for visibility */
    border-radius: 10px;
}

.custom-scrollbar::-webkit-scrollbar-thumb {
    background-color: #F97316; /* Orange-500 */
    border-radius: 10px;
}

.custom-scrollbar::-webkit-scrollbar-thumb:hover {
    background-color: #EA580C; /* Orange-600 */
}
</style>

<?php
// Sidebar included at the top to match standard layout
include 'includes/sidebar.php';
?>

<main class="flex-1 flex flex-col relative z-0 overflow-y-auto bg-gray-50 dark:bg-slate-900 w-full h-screen">
    <!-- Topbar -->
    <?php include 'includes/topbar.php'; ?>

    <!-- Page Content -->
    <div id="transactionContent" class="flex-1 p-4 md:p-6 space-y-6">
        
        <!-- Header -->
        <div class="glass-panel rounded-2xl p-6 mb-6 flex flex-col xl:flex-row justify-between items-center gap-6 animate-fade-in-down relative z-50 overflow-visible bg-white dark:bg-slate-800 shadow-sm border border-gray-100 dark:border-gray-700">
            <div>
                 <h1 class="text-3xl font-bold text-gray-800 dark:text-white tracking-tight">Live Monitor</h1>
                 <p class="text-gray-500 dark:text-gray-400 mt-1">Real-time view of active tills and transactions.</p>
            </div>
            
            <!-- Controls Bar -->
            <div class="flex flex-col sm:flex-row items-center gap-4 bg-gray-50/50 dark:bg-gray-800/50 p-1.5 rounded-xl border border-gray-100 dark:border-gray-700 w-full xl:w-auto overflow-visible">
                
                 <!-- Store Filter -->
                <div class="relative z-50">
                    <button id="liveStoreFilterBtn" onclick="toggleLiveStoreDropdown()" class="flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 rounded-lg transition-colors whitespace-nowrap cursor-pointer">
                        <ion-icon name="storefront-outline" class="text-gray-400"></ion-icon>
                        <span id="liveStoreFilterText">All Stores</span>
                        <ion-icon name="chevron-down-outline" class="text-xs text-gray-400 ml-1 transition-transform duration-300"></ion-icon>
                    </button>
                    
                    <!-- Dropdown Menu -->
                    <div id="liveStoreFilterDropdown" class="absolute top-full left-0 mt-2 w-64 bg-white dark:bg-slate-800 rounded-xl shadow-xl border border-gray-100 dark:border-gray-700 z-50 p-2 space-y-1 opacity-0 invisible transform -translate-y-2 transition-all duration-200">
                        <!-- ... (Store Options) ... -->
                        <!-- All Stores -->
                        <button class="live-store-option w-full flex items-center justify-between p-2 rounded-lg bg-gray-50 dark:bg-slate-700/50 text-gray-800 dark:text-gray-200 transition-colors active" onclick="selectLiveStore('all', 'All Stores', this)">
                            <div class="flex items-center gap-3">
                                <div class="w-8 h-8 rounded-lg bg-orange-100 dark:bg-orange-500/20 flex items-center justify-center text-orange-600 dark:text-orange-400">
                                    <ion-icon name="storefront" class="text-lg"></ion-icon>
                                </div>
                                <span class="font-medium text-sm">All Stores</span>
                            </div>
                            <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-100"></ion-icon>
                        </button>
                         <!-- FNF Cardiff -->
                        <button class="live-store-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors group" onclick="selectLiveStore('cdf', 'FNF Cardiff', this)">
                            <div class="flex items-center gap-3">
                                <div class="w-8 h-8 rounded-lg bg-blue-50 dark:bg-blue-900/20 flex items-center justify-center text-blue-600 dark:text-blue-400 text-xs font-bold">
                                    CDF
                                </div>
                                <span class="font-medium text-sm group-hover:text-gray-900 dark:group-hover:text-white">FNF Cardiff</span>
                            </div>
                            <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                        </button>
                        <!-- FNF Swansea -->
                        <button class="live-store-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors group" onclick="selectLiveStore('swa', 'FNF Swansea', this)">
                            <div class="flex items-center gap-3">
                                <div class="w-8 h-8 rounded-lg bg-emerald-50 dark:bg-emerald-900/20 flex items-center justify-center text-emerald-600 dark:text-emerald-400 text-xs font-bold">
                                    SWA
                                </div>
                                <span class="font-medium text-sm group-hover:text-gray-900 dark:group-hover:text-white">FNF Swansea</span>
                            </div>
                            <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                        </button>
                    </div>
                </div>

                <div class="h-6 w-px bg-gray-300 dark:bg-gray-600 hidden sm:block"></div>

                <!-- Date Filter (Reusable Component) -->
                <?php 
                $pickerId = 'live';
                include 'includes/components/date_picker.php'; 
                ?>

                <div class="h-6 w-px bg-gray-300 dark:bg-gray-600 hidden sm:block"></div>

                 <button class="px-4 py-2 text-sm font-bold rounded-lg bg-white dark:bg-slate-700 text-gray-900 dark:text-white shadow-sm border border-gray-200 dark:border-gray-600 transition-all flex items-center gap-2">
                    <ion-icon name="desktop-outline"></ion-icon>
                    Till 1
                 </button>
                 <button class="px-4 py-2 text-sm font-medium rounded-lg text-gray-500 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 transition-all flex items-center gap-2">
                    <ion-icon name="desktop-outline"></ion-icon>
                    Till 2
                 </button>
                 <button class="px-4 py-2 text-sm font-medium rounded-lg text-gray-500 dark:text-gray-400 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 transition-all flex items-center gap-2">
                    <ion-icon name="desktop-outline"></ion-icon>
                    Till 3
                 </button>
            </div>
        </div>

        <!-- Dashboard Container -->
        <div class="space-y-6">

            <!-- Top Section: Transaction Viewer -->
            <div class="flex flex-col lg:flex-row rounded-3xl overflow-hidden shadow-xl border border-gray-200 dark:border-gray-700 h-[450px]">
                
                <!-- Left: Receipt Table Container -->
                <div class="flex-1 bg-white dark:bg-slate-800 flex flex-col relative overflow-hidden">
                    <!-- Fixed Header -->
                    <div class="px-6 pt-6 pb-2 bg-white dark:bg-slate-800 border-b border-gray-200 dark:border-gray-700 shadow-sm z-20 shrink-0">
                        <table class="w-full text-left border-collapse">
                            <thead>
                                <tr>
                                    <th class="pb-2 text-xs font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wider w-[20%]">Barcode</th>
                                    <th class="pb-2 text-xs font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wider w-[50%]">Product Name</th>
                                    <th class="pb-2 text-xs font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wider text-center w-[15%]">Quantity</th>
                                    <th class="pb-2 text-xs font-bold text-gray-500 dark:text-gray-400 uppercase tracking-wider text-right w-[15%]">Amount</th>
                                </tr>
                            </thead>
                        </table>
                    </div>

                    <!-- Scrollable Content -->
                    <div class="flex-1 overflow-y-auto px-6 pb-6 custom-scrollbar relative">
                        <table class="w-full text-left border-collapse">
                            <!-- Hidden header for alignment purposes only -->
                            <thead class="invisible h-0">
                                <tr>
                                    <th class="w-[20%] p-0 m-0 h-0 border-none"></th>
                                    <th class="w-[50%] p-0 m-0 h-0 border-none"></th>
                                    <th class="w-[15%] p-0 m-0 h-0 border-none"></th>
                                    <th class="w-[15%] p-0 m-0 h-0 border-none"></th>
                                </tr>
                            </thead>
                            <tbody id="transactionListBody" class="divide-y divide-gray-100 dark:divide-gray-700">
                                <?php foreach ($receiptItems as $item): ?>
                                <tr class="group hover:bg-gray-50 dark:hover:bg-slate-700/50">
                                    <td class="py-4 text-sm font-medium text-gray-600 dark:text-gray-300 font-mono"><?php echo $item['barcode']; ?></td>
                                    <td class="py-4 text-sm font-bold text-gray-800 dark:text-white"><?php echo $item['name']; ?></td>
                                    <td class="py-4 text-sm font-bold text-gray-800 dark:text-white text-center"><?php echo $item['qty']; ?></td>
                                    <td class="py-4 text-sm font-bold text-gray-800 dark:text-white text-right">£ <?php echo $item['amount']; ?></td>
                                </tr>
                                <?php endforeach; ?>
                            </tbody>
                        </table>
                    </div>
                </div>

                <!-- Right: Transaction Totals (Neon Mode V11 - Ultra Compact 450px) -->
                <div id="transactionTotalsPanel" class="w-full lg:w-[400px] bg-black text-white p-3 flex flex-col justify-between shrink-0 border-l border-gray-900 font-sans h-full">
                    <div class="flex-1 flex flex-col justify-center">
                        <!-- Time Header -->
                        <div class="text-center text-base font-bold font-mono mb-2 tracking-widest text-white drop-shadow-[0_0_3px_rgba(255,255,255,0.5)]">
                            TIME : <?php echo date('h:i:s A'); ?>
                        </div>

                        <!-- Txn ID & Customer Row (Simplified) -->
                        <div class="flex items-center justify-between gap-4 mb-3 border-b border-gray-800 pb-2">
                            <!-- Transaction ID -->
                            <div class="text-left">
                                 <div class="text-[10px] uppercase tracking-wider text-gray-500 font-bold mb-0.5">Txn ID</div>
                                 <div class="text-sm font-bold text-[#FF8C00] tracking-wide font-mono">#TRX-8492</div>
                            </div>
                            <!-- Customer -->
                            <div class="text-right">
                                 <div class="text-[10px] uppercase tracking-wider text-gray-500 font-bold mb-0.5">Customer</div>
                                 <div class="text-sm font-bold text-white tracking-wide truncate">Walk-in</div>
                            </div>
                        </div>
                        
                        <!-- Subtotal & Discount Row -->
                        <div class="grid grid-cols-2 gap-2 mb-2">
                            <!-- Subtotal -->
                            <div class="bg-black border border-[#FF8C00] rounded-lg p-2 text-center shadow-[0_0_6px_rgba(255,140,0,0.4)]">
                                <div class="text-white text-sm font-bold mb-0.5 drop-shadow-[0_0_2px_rgba(255,255,255,0.3)]">Subtotal</div>
                                <div class="text-base font-bold text-white tracking-wide border-b border-white/20 pb-0.5 inline-block px-1">£ 1700.00</div>
                            </div>
                            <!-- Discount -->
                            <div class="bg-black border border-[#FF8C00] rounded-lg p-2 text-center shadow-[0_0_6px_rgba(255,140,0,0.4)]">
                                <div class="text-white text-sm font-bold mb-0.5 drop-shadow-[0_0_2px_rgba(255,255,255,0.3)]">Discount</div>
                                <div class="text-base font-bold text-white tracking-wide border-b border-white/20 pb-0.5 inline-block px-1">£ 50.00</div>
                            </div>
                        </div>

                        <!-- Total Row -->
                        <div class="bg-black border border-[#FF8C00] rounded-xl p-2 text-center mb-2 shadow-[0_0_10px_rgba(255,140,0,0.5)]">
                            <div class="text-xl font-extrabold text-white tracking-wider drop-shadow-[0_0_5px_rgba(255,255,255,0.6)]">Total - £ 1650.00</div>
                        </div>

                        <!-- Cash & Card Row -->
                        <div class="grid grid-cols-2 gap-2 mb-2">
                             <!-- Cash -->
                             <div class="bg-black border border-[#FF8C00] rounded-lg p-2 text-center shadow-[0_0_6px_rgba(255,140,0,0.4)]">
                                <div class="text-white text-sm font-bold mb-0.5">Cash</div>
                                <div class="text-base font-bold text-white tracking-wide">-</div>
                            </div>
                            <!-- Card -->
                            <div class="bg-black border border-[#FF8C00] rounded-lg p-2 text-center shadow-[0_0_6px_rgba(255,140,0,0.4)]">
                                <div class="text-white text-sm font-bold mb-0.5">Card</div>
                                <div class="text-base font-bold text-white tracking-wide">£ 1650.00</div>
                            </div>
                        </div>

                        <!-- Change Row (Half Width Visual) -->
                         <div class="grid grid-cols-2 gap-2 mb-3">
                             <div class="bg-black border border-[#FF8C00] rounded-lg p-2 text-center shadow-[0_0_6px_rgba(255,140,0,0.4)]">
                                <div class="text-white text-sm font-bold mb-0.5">Change</div>
                                <div class="text-base font-bold text-white tracking-wide">-</div>
                            </div>
                            <!-- Empty spacer -->
                            <div></div>
                        </div>
                    </div>

                    <!-- Action Button -->
                    <button class="w-full bg-[#E65100] hover:bg-[#F57C00] text-white font-extrabold py-3 rounded-lg text-lg shadow-[0_0_15px_rgba(230,81,0,0.6)] transition-all uppercase tracking-wider mt-auto transform hover:scale-[1.01]">
                        SALE COMPLETED
                    </button>
                </div>
            </div>

            <!-- Bottom Section: Timeline (Modernized & Dynamic) -->
            <div class="glass-panel text-gray-800 dark:text-white rounded-3xl p-6 shadow-2xl border border-gray-200 dark:border-gray-800 bg-white dark:bg-gray-900/95 relative overflow-hidden transition-colors duration-300">
                <div class="absolute inset-0 bg-gradient-to-b from-gray-50/20 to-gray-100/50 dark:from-gray-800/20 dark:to-black/80 pointer-events-none"></div>
                
                <div class="relative z-10">
                    <div class="flex items-center justify-between mb-0">
                        <div class="flex items-center gap-6">
                             <h3 class="text-sm font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest flex items-center gap-2">
                                 <ion-icon name="time-outline" class="text-lg text-primary"></ion-icon>
                                 Live Activity Timeline
                            </h3>
                            <!-- Real-time Clock -->
                            <div class="hidden sm:block px-3 py-1 bg-gray-100 dark:bg-black/50 rounded-lg border border-gray-200 dark:border-gray-700">
                                <span id="timelineClock" class="font-mono font-bold text-sm text-gray-700 dark:text-gray-200">--:--:--</span>
                            </div>

                             <div class="relative z-40 hidden sm:block">
                                <button id="timelineTimeFilterBtn" onclick="toggleTimelineTimeDropdown()" class="flex items-center gap-2 px-3 py-1 text-sm font-medium text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 rounded-lg transition-colors whitespace-nowrap cursor-pointer">
                                    <ion-icon name="time-outline" class="text-gray-400"></ion-icon>
                                    <span id="timelineTimeFilterText">15 Mins</span>
                                    <ion-icon name="chevron-down-outline" class="text-xs text-gray-400 ml-1 transition-transform duration-300"></ion-icon>
                                </button>
                                
                                <div id="timelineTimeFilterDropdown" class="absolute top-full left-0 mt-2 w-48 bg-white dark:bg-slate-800 rounded-xl shadow-xl border border-gray-100 dark:border-gray-700 z-50 p-2 space-y-1 opacity-0 invisible transform -translate-y-2 transition-all duration-200 h-64 overflow-y-auto custom-scrollbar">
                                    <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('5', '5 Mins', this)">
                                        <span class="font-medium text-sm">5 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                    <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('10', '10 Mins', this)">
                                        <span class="font-medium text-sm">10 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                    <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg bg-gray-50 dark:bg-slate-700/50 text-gray-800 dark:text-gray-200 transition-colors active" onclick="selectTimelineTime('15', '15 Mins', this)">
                                        <span class="font-medium text-sm">15 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-100"></ion-icon>
                                    </button>
                                     <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('20', '20 Mins', this)">
                                        <span class="font-medium text-sm">20 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                     <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('25', '25 Mins', this)">
                                        <span class="font-medium text-sm">25 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                    <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('30', '30 Mins', this)">
                                        <span class="font-medium text-sm">30 Mins</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                     <button class="timeline-time-option w-full flex items-center justify-between p-2 rounded-lg hover:bg-gray-50 dark:hover:bg-slate-700/50 text-gray-600 dark:text-gray-300 transition-colors" onclick="selectTimelineTime('60', '1 Hour', this)">
                                        <span class="font-medium text-sm">1 Hour</span>
                                        <ion-icon name="checkmark-circle" class="text-orange-500 text-lg check-icon opacity-0"></ion-icon>
                                    </button>
                                </div>
                            </div>
                        </div>
                        <button id="timelineLiveToggle" class="flex items-center gap-2 group cursor-pointer hover:opacity-80 transition-all opacity-100" onclick="toggleAutoFollow()">
                            <span class="relative flex h-3 w-3">
                              <span id="liveIndicatorPing" class="animate-ping absolute inline-flex h-full w-full rounded-full bg-primary opacity-75"></span>
                              <span id="liveIndicatorDot" class="relative inline-flex rounded-full h-3 w-3 bg-primary"></span>
                            </span>
                            <span id="liveIndicatorText" class="text-xs font-bold text-gray-700 dark:text-white tracking-wider uppercase">Live</span>
                        </button>
                    </div>

                    <!-- Timeline Container -->
                    <div class="overflow-x-auto custom-scrollbar pb-2 pt-16" id="timelineScrollContainer">
                        <div class="relative select-none" id="timelineTrack" style="width: 12000px;">
                            
                            <!-- Tracks Background -->
                            <div class="relative w-full h-64 bg-gray-50 dark:bg-black/40 rounded-2xl border border-gray-200 dark:border-white/5 transition-colors">
                                
                                <!-- Vertical Grid Lines (Exact Grid) -->
                                <div class="absolute inset-0 w-full grid grid-cols-[repeat(288,1fr)] pointer-events-none z-0" id="timelineGrid">
                                    <?php 
                                    // Generate exactly 288 segments for 1440 minutes
                                    for($i=0; $i<288; $i++): 
                                    ?>
                                    <div class="h-full relative group timeline-tick border-gray-200 dark:border-white/5" data-tick-index="<?php echo $i; ?>">
                                        <div class="timeline-label absolute -top-3 left-0 -translate-x-1/2 text-[10px] font-mono font-bold text-gray-400 dark:text-gray-500 bg-white/80 dark:bg-gray-900/80 px-1 rounded opacity-50 group-hover:opacity-100 transition-opacity whitespace-nowrap z-20">00:00</div>
                                    </div>
                                    <?php endfor; ?>
                                    
                                    <!-- 24:00 label (Absolute positioned at end) -->
                                    <div class="timeline-label absolute -top-3 left-full -translate-x-full text-[10px] font-mono font-bold text-gray-400 dark:text-gray-500 bg-white/80 dark:bg-gray-900/80 px-1 rounded opacity-50 z-20">00:00</div>
                                </div>

                                <!-- Horizontal Swimlanes (Optional visual guide) -->
                                <div class="absolute inset-0 flex flex-col justify-evenly pointer-events-none opacity-50 dark:opacity-10">
                                    <div class="h-px bg-gray-200 dark:bg-white w-full"></div>
                                    <div class="h-px bg-gray-200 dark:bg-white w-full"></div>
                                    <div class="h-px bg-gray-200 dark:bg-white w-full"></div>
                                </div>

                                <!-- Events Layer -->
                                <div class="absolute inset-0 w-full z-10">
                                    
                                    <!-- Event 1: Cash (10:32 AM -> 632 mins) -->
                                    <div class="timeline-event absolute top-6 h-8 bg-gradient-to-r from-orange-500 to-orange-400 dark:from-orange-600 dark:to-orange-500 rounded-lg shadow-sm dark:shadow-lg shadow-orange-500/20 dark:shadow-orange-900/20 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                         data-time-minutes="632" data-duration="5">
                                        <!-- Tooltip -->
                                        <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-orange-600 dark:text-orange-400">Cash Sale</div>
                                            <div class="text-gray-500 dark:text-gray-400">10:32 AM • £42.50</div>
                                        </div>
                                    </div>

                                    <!-- Event 2: Line Void (10:38 AM -> 638 mins) -->
                                    <div class="timeline-event absolute top-6 h-8 bg-gradient-to-r from-yellow-400 to-yellow-300 dark:from-yellow-500 dark:to-yellow-400 rounded-lg shadow-sm dark:shadow-lg shadow-yellow-500/20 dark:shadow-yellow-900/20 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                         data-time-minutes="638" data-duration="6">
                                         <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-yellow-600 dark:text-yellow-400">Line Void</div>
                                            <div class="text-gray-500 dark:text-gray-400">10:38 AM • User: Supervisor</div>
                                        </div>
                                    </div>
                                    
                                     <!-- Event 3: Card (10:45 AM -> 645 mins) -->
                                    <div class="timeline-event absolute top-20 h-8 bg-gradient-to-r from-orange-200 to-orange-100 dark:from-orange-300 dark:to-orange-200 rounded-lg shadow-sm dark:shadow-lg shadow-orange-500/20 dark:shadow-orange-900/20 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                         data-time-minutes="645" data-duration="5">
                                        <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-orange-600 dark:text-orange-200">Card Sale</div>
                                            <div class="text-gray-500 dark:text-gray-400">10:45 AM • £12.00</div>
                                        </div>
                                    </div>

                                    <!-- Event 4: VOID (11:15 AM -> 675 mins) -->
                                    <div class="timeline-event absolute top-6 h-8 bg-gradient-to-r from-red-600 to-red-500 dark:from-red-700 dark:to-red-600 rounded-lg shadow-sm dark:shadow-lg shadow-red-500/20 dark:shadow-red-900/30 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                         data-time-minutes="675" data-duration="6">
                                        <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-red-600 dark:text-red-500">Transaction Void</div>
                                            <div class="text-gray-500 dark:text-gray-400">11:15 AM • Manager Overridden</div>
                                        </div>
                                    </div>

                                     <!-- Event 5: Payout (11:45 AM -> 705 mins) -->
                                     <div class="timeline-event absolute top-6 h-8 bg-gradient-to-r from-slate-600 to-slate-500 dark:from-slate-700 dark:to-slate-600 rounded-lg shadow-sm dark:shadow-lg shadow-slate-500/20 dark:shadow-black/40 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                          data-time-minutes="705" data-duration="6">
                                        <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-slate-600 dark:text-slate-300">Supplier Payout</div>
                                            <div class="text-gray-500 dark:text-gray-400">11:45 AM • -£20.00</div>
                                        </div>
                                     </div>

                                     <!-- Event 6: Cash Lift (12:00 PM -> 720 mins) -->
                                     <div class="timeline-event absolute top-20 h-8 bg-gradient-to-r from-slate-500 to-slate-400 dark:from-slate-600 dark:to-slate-500 rounded-lg shadow-sm dark:shadow-lg shadow-slate-500/20 dark:shadow-black/40 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                          data-time-minutes="720" data-duration="6">
                                         <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-slate-600 dark:text-slate-300">Safe Drop</div>
                                            <div class="text-gray-500 dark:text-gray-400">12:00 PM • £500.00</div>
                                        </div>
                                     </div>

                                     <!-- Event 7: Penny (1:00 PM -> 780 mins) -->
                                     <div class="timeline-event absolute top-6 h-8 bg-gradient-to-r from-gray-400 to-gray-300 dark:from-gray-500 dark:to-gray-400 rounded-lg shadow-sm dark:shadow-lg shadow-gray-500/20 dark:shadow-black/20 cursor-pointer group hover:brightness-110 transition-colors border border-white/20 dark:border-white/10"
                                          data-time-minutes="780" data-duration="6">
                                         <div class="absolute bottom-full left-1/2 -translate-x-1/2 mb-2 bg-white dark:bg-gray-900 text-gray-800 dark:text-white text-xs p-2 rounded-lg border border-gray-200 dark:border-gray-700 shadow-xl opacity-0 group-hover:opacity-100 transition-opacity whitespace-nowrap z-50 pointer-events-none">
                                            <div class="font-bold text-gray-600 dark:text-gray-300">Penny Test</div>
                                            <div class="text-gray-500 dark:text-gray-400">01:00 PM • £0.01</div>
                                        </div>
                                     </div>

                                </div>
                                
                                <!-- Playhead Indicator (Centered) -->
                                <div id="timelinePlayhead" class="absolute top-0 bottom-0 w-0.5 bg-orange-500 dark:bg-orange-500/80 shadow-[0_0_8px_rgba(255,140,0,0.5)] dark:shadow-[0_0_15px_rgba(255,140,0,0.8)] z-30 flex flex-col justify-end pointer-events-none" style="left: 0;">
                                      <div class="absolute -top-1 left-1/2 -translate-x-1/2 w-2.5 h-2.5 rounded-full bg-orange-500 shadow-md"></div>
                                      <div class="absolute bottom-4 left-1/2 -translate-x-1/2 bg-orange-500 text-white text-[9px] px-1.5 py-0.5 rounded font-bold tracking-wider shadow-lg whitespace-nowrap">NOW</div>
                                </div>
                            </div>
                        </div>
                    </div>

                    <!-- Legend (Scrollable Pills with improved styling) -->
                    <div class="flex overflow-x-auto pb-2 gap-3 no-scrollbar border-t border-gray-200 dark:border-gray-800 pt-4 mt-2">
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-gradient-to-r from-orange-600 to-orange-500 shadow-sm shadow-orange-500/50"></div>
                            <span class="text-xs font-semibold text-gray-400">CASH</span>
                        </div>
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-gradient-to-r from-orange-200 to-orange-100 dark:from-orange-300 dark:to-orange-200"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">CARD</span>
                        </div>
                         <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-gradient-to-r from-yellow-400 to-yellow-300 dark:from-yellow-500 dark:to-yellow-400"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Line Void</span>
                        </div>
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-gradient-to-r from-red-600 to-red-500 dark:from-red-700 dark:to-red-600 shadow-sm"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">VOID</span>
                        </div>
                         <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-blue-500"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Refund</span>
                        </div>
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-slate-600"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Payout</span>
                        </div>
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-slate-500"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Cash Lift</span>
                        </div>
                        <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-gray-400"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Penny</span>
                        </div>
                         <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-indigo-500"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Shift Change</span>
                        </div>
                         <div class="flex items-center gap-2 px-3 py-1.5 bg-gray-50 dark:bg-black/40 rounded-lg border border-gray-200 dark:border-gray-700/50 hover:bg-gray-100 dark:hover:bg-gray-800/50 transition-colors cursor-pointer shrink-0">
                            <div class="w-2.5 h-2.5 rounded-full bg-green-500"></div>
                            <span class="text-xs font-semibold text-gray-600 dark:text-gray-400">Discounts</span>
                        </div>
                    </div>
                </div>
            </div>

        </div>
    </div>
</main>

<style>
/* Custom Scrollbar for Timeline */
.custom-scrollbar::-webkit-scrollbar {
    height: 8px;
}
.custom-scrollbar::-webkit-scrollbar-track {
    background: transparent;
}
.custom-scrollbar::-webkit-scrollbar-thumb {
    background-color: rgba(156, 163, 175, 0.5);
    border-radius: 4px;
}
.custom-scrollbar::-webkit-scrollbar-thumb:hover {
    background-color: rgba(156, 163, 175, 0.8);
}
</style>

<script>



    // --- Global State ---
    let currentIntervalMinutes = 15; // Default
    let gridStartTime = new Date(); 
    let isAutoFollowEnabled = true;

    // --- Core Timeline Functions ---

    function updateTimelineClock() {
        const now = new Date();
        const timeString = now.toLocaleTimeString('en-US', { hour12: false });
        const clockEl = document.getElementById('timelineClock');
        if(clockEl) clockEl.textContent = timeString;
    }

    function updateTimelineLabels() {
        const track = document.getElementById('timelineTrack');
        if (!track) return;
        
        const ticks = track.querySelectorAll('.timeline-tick');
        const grid = document.getElementById('timelineGrid');
        const lastLabel = grid.querySelector('.w-0 .timeline-label');
        
        // --- FIXED 24-HOUR ARCHITECTURE (TRIPLED ZOOM) ---
        const now = new Date();
        now.setHours(0, 0, 0, 0);
        gridStartTime = new Date(now);

        // Adjust Track Width based on Interval/Zoom
        let targetWidth = 12000; // 1 Hour mode (500px per hour label)
        if (currentIntervalMinutes <= 5) targetWidth = 120000; 
        else if (currentIntervalMinutes <= 15) targetWidth = 48000;
        else if (currentIntervalMinutes <= 30) targetWidth = 24000; // 30 Min mode (500px per 30m label)
        
        track.style.width = targetWidth + 'px';
        
        ticks.forEach((tick, index) => {
            const timeInMinutes = index * 5;
            const label = tick.querySelector('.timeline-label');
            
            // Show tick/border only if it aligns with the selected interval
            if (timeInMinutes % currentIntervalMinutes === 0) {
                tick.classList.add('border-l'); // Anchor to LEFT of segment
                if (label) {
                    label.classList.remove('hidden');
                    const hours = Math.floor(timeInMinutes / 60).toString().padStart(2, '0');
                    const minutes = (timeInMinutes % 60).toString().padStart(2, '0');
                    label.textContent = `${hours}:${minutes}`;
                    
                    // Clamping: Ensure first label doesn't hang off left, others stay centered
                    if (index === 0) {
                        label.classList.remove('-translate-x-1/2');
                        label.classList.add('translate-x-0');
                    } else {
                        label.classList.add('-translate-x-1/2');
                        label.classList.remove('translate-x-0');
                    }
                }
            } else {
                tick.classList.remove('border-l');
                if (label) label.classList.add('hidden');
            }
        });

        // Handle the final 24:00 label
        if (lastLabel) {
            lastLabel.textContent = "00:00"; 
            lastLabel.classList.toggle('hidden', 1440 % currentIntervalMinutes !== 0);
            // Clamp last label to right
            lastLabel.classList.remove('-translate-x-1/2');
            lastLabel.classList.add('-translate-x-full');
        }
        
        // Final containment safety
        // Removed track.style.overflow = 'hidden' to prevent tooltip clipping

        
        updateTimelineEvents();
    }

    function updatePlayheadSmooth() {
        const container = document.getElementById('timelineScrollContainer');
        const track = document.getElementById('timelineTrack');
        const playhead = document.getElementById('timelinePlayhead');
        if (!container || !track || !playhead) return;
        
        const now = new Date();
        const startOfDay = new Date(now);
        startOfDay.setHours(0, 0, 0, 0);
        
        const msSinceMidnight = now.getTime() - startOfDay.getTime();
        const totalMsInDay = 24 * 60 * 60 * 1000;
        
        // Position on 24hr track (Percentage)
        const dayProgress = msSinceMidnight / totalMsInDay;
        const trackWidth = track.offsetWidth;
        const nowPixelPos = dayProgress * trackWidth;
        
        // Update playhead position
        playhead.style.left = nowPixelPos + 'px';
        
        // Auto-follow logic
        if (isAutoFollowEnabled) {
            const containerWidth = container.offsetWidth;
            container.scrollLeft = nowPixelPos - (containerWidth / 2);
        }
        
        requestAnimationFrame(updatePlayheadSmooth);
    }

    function updateTimelineEvents() {
        const events = document.querySelectorAll('.timeline-event');
        const track = document.getElementById('timelineTrack');
        if (!track) return;
        
        const trackWidth = track.offsetWidth;
        const totalMsInDay = 24 * 60 * 60 * 1000;
        
        events.forEach(el => {
            const timeMinutes = parseInt(el.getAttribute('data-time-minutes'));
            const duration = parseInt(el.getAttribute('data-duration') || '5');
            
            if (!isNaN(timeMinutes)) {
                const msOffset = timeMinutes * 60 * 1000;
                const leftPx = (msOffset / totalMsInDay) * trackWidth;
                const widthPx = (duration * 60 * 1000 / totalMsInDay) * trackWidth;
                
                el.style.left = leftPx + 'px';
                el.style.width = widthPx + 'px';
            }
        });
    }

    // --- Interaction Handlers ---

    function toggleLiveStoreDropdown() {
        const dropdown = document.getElementById('liveStoreFilterDropdown');
        const btn = document.getElementById('liveStoreFilterBtn');
        const icon = btn.querySelector('ion-icon[name="chevron-down-outline"]');
        if (dropdown && dropdown.classList.contains('invisible')) {
            dropdown.classList.remove('invisible', 'opacity-0', '-translate-y-2');
            if(icon) icon.classList.add('rotate-180');
        } else if (dropdown) {
            dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
            if(icon) icon.classList.remove('rotate-180');
        }
    }

    function selectLiveStore(value, label, optionBtn) {
        document.getElementById('liveStoreFilterText').textContent = label;
        document.querySelectorAll('.live-store-option').forEach(btn => {
            const check = btn.querySelector('.check-icon');
            btn.classList.remove('bg-gray-50', 'dark:bg-slate-700/50', 'active');
            if(check) check.classList.add('opacity-0');
        });
        optionBtn.classList.add('bg-gray-50', 'dark:bg-slate-700/50', 'active');
        const check = optionBtn.querySelector('.check-icon');
        if(check) check.classList.remove('opacity-0');
        toggleLiveStoreDropdown();
    }

    function toggleTimelineTimeDropdown() {
        const dropdown = document.getElementById('timelineTimeFilterDropdown');
        if (dropdown && dropdown.classList.contains('invisible')) {
            dropdown.classList.remove('invisible', 'opacity-0', '-translate-y-2');
        } else if (dropdown) {
            dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
        }
    }

    function selectTimelineTime(value, label, optionBtn) {
        document.getElementById('timelineTimeFilterText').textContent = label;
        currentIntervalMinutes = parseInt(value);
        updateTimeOptionsState('timeline-time-option', optionBtn);
        toggleTimelineTimeDropdown();
        updateTimelineLabels();
    }

    function updateTimeOptionsState(optionClass, activeBtn) {
         document.querySelectorAll('.' + optionClass).forEach(btn => {
            const check = btn.querySelector('.check-icon');
            btn.classList.remove('bg-gray-50', 'dark:bg-slate-700/50', 'active');
            if(check) check.classList.add('opacity-0');
        });
        activeBtn.classList.add('bg-gray-50', 'dark:bg-slate-700/50', 'active');
        const check = activeBtn.querySelector('.check-icon');
        if(check) check.classList.remove('opacity-0');
    }

    // --- Event Listeners & Initialization ---

    document.addEventListener('DOMContentLoaded', () => {
        const container = document.getElementById('timelineScrollContainer');
        if (container) {
            // Detect manual scroll
            container.addEventListener('mousedown', () => {
                isAutoFollowEnabled = false;
                updateLiveIndicatorUI();
            });
            
            container.addEventListener('wheel', () => {
                isAutoFollowEnabled = false;
                updateLiveIndicatorUI();
            });

            container.addEventListener('touchstart', () => {
                isAutoFollowEnabled = false;
                updateLiveIndicatorUI();
            });
        }
    });

    function toggleAutoFollow() {
        isAutoFollowEnabled = !isAutoFollowEnabled;
        updateLiveIndicatorUI();
    }

    function updateLiveIndicatorUI() {
        const ping = document.getElementById('liveIndicatorPing');
        const dot = document.getElementById('liveIndicatorDot');
        const text = document.getElementById('liveIndicatorText');
        
        if (isAutoFollowEnabled) {
            ping.classList.remove('hidden');
            dot.classList.add('bg-primary');
            dot.classList.remove('bg-gray-400');
            text.textContent = 'Live';
            text.classList.add('text-primary');
            text.classList.remove('text-gray-400');
        } else {
            ping.classList.add('hidden');
            dot.classList.remove('bg-primary');
            dot.classList.add('bg-gray-400');
            text.textContent = 'Go Live';
            text.classList.remove('text-primary');
            text.classList.add('text-gray-400');
        }
    }

    document.addEventListener('click', (e) => {
        const storeBtn = document.getElementById('liveStoreFilterBtn');
        const storeDropdown = document.getElementById('liveStoreFilterDropdown');
        if (storeBtn && storeDropdown && !storeBtn.contains(e.target) && !storeDropdown.contains(e.target)) {
            storeDropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
        }
        const tlTimeBtn = document.getElementById('timelineTimeFilterBtn');
        const tlTimeDropdown = document.getElementById('timelineTimeFilterDropdown');
        if (tlTimeBtn && tlTimeDropdown && !tlTimeBtn.contains(e.target) && !tlTimeDropdown.contains(e.target)) {
            tlTimeDropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
        }
        
        // Date Dropdown Close on Click Outside
        const dateBtn = document.getElementById('liveDateFilterBtn');
        const dateDropdown = document.getElementById('liveDateFilterDropdown');
        if (dateBtn && dateDropdown && !dateBtn.contains(e.target) && !dateDropdown.contains(e.target)) {
            dateDropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
            const icon = dateBtn.querySelector('ion-icon[name="chevron-down-outline"]');
            if(icon) icon.classList.remove('rotate-180');
        }
    });

    // --- Date Picker Toggle ---
    // --- AJAX Helper for Calendar Component Hook ---
    window.handleDateChange = async function(url) {
        // Add loading state?
        const label = document.getElementById('liveDateFilterText');
        if(label) label.classList.add('opacity-50');
        
        await fetchUpdatedContent(url);
        
        if(label) label.classList.remove('opacity-50');
    };

    async function fetchUpdatedContent(url) {
         try {
            const response = await fetch(url);
            const text = await response.text();
            const parser = new DOMParser();
            const doc = parser.parseFromString(text, 'text/html');
            
            // Update Receipt Table Body
            const newTableBody = doc.getElementById('transactionListBody');
            const currentTableBody = document.getElementById('transactionListBody');
            if (newTableBody && currentTableBody) {
                currentTableBody.innerHTML = newTableBody.innerHTML;
            }

            // Update Totals Panel
            const newTotals = doc.getElementById('transactionTotalsPanel');
            const currentTotals = document.getElementById('transactionTotalsPanel');
            if (newTotals && currentTotals) {
                currentTotals.innerHTML = newTotals.innerHTML;
            }
            
        } catch (error) {
            console.error('Error fetching updates:', error);
            // Fallback to full reload
            window.location.reload(); 
        }
    }

    // START LOOPS
    setInterval(updateTimelineClock, 1000);
    updateTimelineClock();
    
    // Delay layout calculation
    setTimeout(() => {
        requestAnimationFrame(() => {
            updateTimelineLabels();
            updatePlayheadSmooth();
        });
    }, 100);
</script>
