<?php
include 'includes/session.php';
$pageTitle = "Table Layout";
$currentPage = "table_layout";
require_once 'includes/header.php';

// Mock Zones
$zones = [
    ['id' => 1, 'name' => 'Main Dining', 'active' => true, 'tables' => 12, 'status' => 'active'],
    ['id' => 2, 'name' => 'Patio', 'active' => false, 'tables' => 8, 'status' => 'active'],
    ['id' => 3, 'name' => 'Bar Area', 'active' => false, 'tables' => 5, 'status' => 'party'], 
    ['id' => 4, 'name' => 'VIP Lounge', 'active' => false, 'tables' => 3, 'status' => 'active'],
];
?>

<!-- Sidebar -->
<?php include 'includes/sidebar.php'; ?>

<main class="flex-1 flex flex-col relative z-0 overflow-hidden w-full h-screen">
    <!-- Topbar -->
    <?php include 'includes/topbar.php'; ?>

    <div class="flex-1 flex overflow-hidden relative z-0">
        
        <!-- Left Sidebar: Zones & Tools -->
        <aside class="w-80 bg-white dark:bg-slate-900 border-r border-gray-200 dark:border-gray-800 flex flex-col z-20 shadow-xl relative">
             <div class="h-16 flex items-center px-6 border-b border-gray-200 dark:border-gray-800 bg-gray-50/50 dark:bg-slate-900/50 backdrop-blur-sm flex-shrink-0">
                <div class="flex items-center gap-3">
                    <div class="h-9 w-9 rounded-lg bg-gradient-to-br from-orange-500 to-orange-600 flex items-center justify-center shadow-lg shadow-orange-500/30 text-white">
                        <ion-icon name="grid-outline" class="text-lg"></ion-icon>
                    </div>
                    <div>
                         <h2 class="text-sm font-bold text-gray-800 dark:text-white leading-tight">Floor Plan</h2>
                         <p class="text-[10px] text-gray-500 font-medium">Manage zones & tables</p>
                    </div>
                </div>
            </div>
            
            <div class="flex-1 overflow-y-auto p-4 custom-scrollbar border-b border-gray-200 dark:border-gray-800">
                 
                 <!-- Zones Section -->
                 <div>
                    <div class="flex items-center justify-between mb-3 px-1">
                        <span class="text-xs font-bold text-gray-400 uppercase tracking-wider">Zones</span>
                        <button onclick="toggleZoneEditMode(this)" id="editZonesBtn" class="text-primary hover:text-orange-600 text-xs font-bold transition-colors">Edit All</button>
                    </div>
                    
                    <div class="space-y-2" id="zonesList">
                        <?php foreach ($zones as $zone): ?>
                        <div class="relative group/zone" id="zone-<?php echo $zone['id']; ?>" data-id="<?php echo $zone['id']; ?>" data-name="<?php echo $zone['name']; ?>" data-status="<?php echo $zone['status']; ?>" data-tables="<?php echo $zone['tables']; ?>">
                             <button class="w-full flex items-center justify-between px-4 py-3 rounded-xl text-sm font-medium transition-all duration-200 border <?php echo $zone['active'] ? 'bg-orange-50 dark:bg-primary/10 border-orange-200 dark:border-primary/30 text-primary shadow-sm' : 'bg-white dark:bg-slate-800 border-transparent hover:border-gray-200 dark:hover:border-gray-700 text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-slate-700/50'; ?>">
                                <div class="flex items-center gap-3">
                                    <span class="w-1.5 h-1.5 rounded-full <?php echo $zone['active'] ? 'bg-orange-500 animate-pulse' : 'bg-gray-300 dark:bg-gray-600 group-hover:bg-gray-400'; ?>"></span>
                                    <div class="flex flex-col items-start text-left">
                                        <span class="font-bold zone-name"><?php echo $zone['name']; ?></span>
                                        <?php if($zone['status'] === 'party'): ?>
                                            <div class="zone-badge mt-1 flex flex-col items-start">
                                                <span class="text-[10px] text-red-500 font-bold bg-red-50 dark:bg-red-900/20 px-1.5 py-0.5 rounded-md border border-red-100 dark:border-red-900/30">Reserved Party</span>
                                                <span class="text-[9px] text-red-400 font-medium ml-0.5">19:00 - 22:00</span>
                                            </div>
                                        <?php endif; ?>
                                    </div>
                                </div>
                                <span class="text-xs <?php echo $zone['active'] ? 'text-orange-400' : 'text-gray-400'; ?> tables-count"><?php echo $zone['tables']; ?></span>
                            </button>
                            
                            <!-- Edit Tools (Hidden by default) -->
                            <div class="edit-tools absolute right-2 top-1/2 -translate-y-1/2 flex items-center gap-1 opacity-0 pointer-events-none transition-opacity duration-200">
                                <button onclick="openZoneModal(<?php echo $zone['id']; ?>)" class="p-1.5 bg-white dark:bg-slate-700 rounded-lg shadow-sm border border-gray-200 dark:border-gray-600 text-gray-500 hover:text-primary hover:border-primary"><ion-icon name="create-outline"></ion-icon></button>
                                <button onclick="deleteZone(<?php echo $zone['id']; ?>)" class="p-1.5 bg-white dark:bg-slate-700 rounded-lg shadow-sm border border-gray-200 dark:border-gray-600 text-gray-500 hover:text-red-500 hover:border-red-500"><ion-icon name="trash-outline"></ion-icon></button>
                            </div>
                        </div>
                        <?php endforeach; ?>
                    </div>

                    <button id="addZoneBtn" onclick="openZoneModal()" class="w-full mt-2 py-2.5 border-2 border-dashed border-gray-300 dark:border-gray-700 rounded-xl text-gray-400 dark:text-gray-500 text-xs font-bold uppercase tracking-wider hover:border-orange-500 hover:text-orange-500 dark:hover:text-orange-400 hover:bg-orange-50 dark:hover:bg-orange-900/10 transition-all flex items-center justify-center gap-2 opacity-50 pointer-events-none cursor-default">
                        <ion-icon name="add-circle-outline" class="text-lg"></ion-icon> Add New Zone
                    </button>
                 </div>
            </div>
            
            <div class="flex-1 overflow-y-auto p-4 custom-scrollbar space-y-6">
                 
                 <!-- Tables Section -->
                 <div>
                     <div class="text-xs font-bold text-gray-400 uppercase tracking-wider mb-3 px-1">Tables & Seats</div>
                     <div class="grid grid-cols-2 gap-3">
                         <div class="draggable-tool group p-3 bg-white dark:bg-slate-700 rounded-xl border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md cursor-grab active:cursor-grabbing flex flex-col items-center gap-2 hover:border-primary/50 transition-all" data-type="round">
                             <div class="w-8 h-8 rounded-full border-2 border-gray-300 dark:border-gray-500 bg-gray-50 dark:bg-slate-800 pointer-events-none"></div>
                             <span class="text-[10px] font-bold text-gray-500 pointer-events-none">Round Table</span>
                         </div>
                         <div class="draggable-tool group p-3 bg-white dark:bg-slate-700 rounded-xl border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md cursor-grab active:cursor-grabbing flex flex-col items-center gap-2 hover:border-primary/50 transition-all" data-type="square">
                             <div class="w-8 h-8 rounded-lg border-2 border-gray-300 dark:border-gray-500 bg-gray-50 dark:bg-slate-800 pointer-events-none"></div>
                             <span class="text-[10px] font-bold text-gray-500 pointer-events-none">Square Table</span>
                         </div>
                         <div class="draggable-tool group p-3 bg-white dark:bg-slate-700 rounded-xl border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md cursor-grab active:cursor-grabbing flex flex-col items-center gap-2 hover:border-primary/50 transition-all col-span-2" data-type="rect">
                             <div class="w-16 h-8 rounded-lg border-2 border-gray-300 dark:border-gray-500 bg-gray-50 dark:bg-slate-800 pointer-events-none"></div>
                             <span class="text-[10px] font-bold text-gray-500 pointer-events-none">Rectangular Table</span>
                         </div>
                         <div class="draggable-tool group p-3 bg-white dark:bg-slate-700 rounded-xl border border-gray-200 dark:border-gray-600 shadow-sm hover:shadow-md cursor-grab active:cursor-grabbing flex flex-col items-center gap-2 hover:border-primary/50 transition-all col-span-2" data-type="booth">
                             <div class="w-16 h-10 rounded-lg border-2 border-gray-300 dark:border-gray-500 bg-gray-50 dark:bg-slate-800 flex items-center justify-between px-1 pointer-events-none">
                                <div class="h-8 w-1 bg-gray-300"></div><div class="h-8 w-1 bg-gray-300"></div>
                             </div>
                             <span class="text-[10px] font-bold text-gray-500 pointer-events-none">Booth</span>
                         </div>
                     </div>
                 </div>

                 <!-- Structure & Objects -->
                 <div>
                     <div class="text-xs font-bold text-gray-400 uppercase tracking-wider mb-3 px-1">Structure & Objects</div>
                     <div class="grid grid-cols-3 gap-2">
                         <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="wall">
                             <div class="w-1 h-6 bg-gray-400 pointer-events-none"></div>
                             <span class="text-[9px] font-bold text-gray-500">Wall</span>
                         </div>
                         <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="window">
                             <div class="w-1 h-6 bg-blue-200 pointer-events-none"></div>
                             <span class="text-[9px] font-bold text-gray-500">Window</span>
                         </div>
                         <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="door">
                             <div class="w-4 h-4 rounded-tl-full border-t border-l border-gray-500 pointer-events-none"></div>
                             <span class="text-[9px] font-bold text-gray-500">Door</span>
                         </div>
                          <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="pillar">
                             <div class="w-4 h-4 bg-gray-300 border border-gray-400 pointer-events-none"></div>
                             <span class="text-[9px] font-bold text-gray-500">Pillar</span>
                         </div>
                         <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="cashier">
                             <ion-icon name="desktop" class="text-sm"></ion-icon>
                             <span class="text-[9px] font-bold text-gray-500">POS</span>
                         </div>
                         <div class="draggable-tool group p-2 bg-white dark:bg-slate-700 rounded-lg border border-gray-200 dark:border-gray-600 hover:shadow-md cursor-grab flex flex-col items-center gap-1" data-type="plant">
                             <ion-icon name="leaf" class="text-sm text-green-500"></ion-icon>
                             <span class="text-[9px] font-bold text-gray-500">Plant</span>
                         </div>
                     </div>
                 </div>

            </div>
        </aside>

        <!-- Canvas Area -->
        <div class="flex-1 bg-gray-100 dark:bg-[#0f172a] relative overflow-hidden flex flex-col">
            <!-- Toolbar -->
            <div class="h-16 bg-white/90 dark:bg-slate-900/90 backdrop-blur-md border-b border-gray-200 dark:border-gray-800 flex items-center justify-between px-6 shadow-sm z-10 relative">
                <div class="flex items-center gap-4">
                     <div class="flex flex-col">
                        <span class="text-[10px] font-bold text-gray-400 uppercase tracking-wider">Active Canvas</span>
                        <div class="text-sm font-bold text-gray-800 dark:text-white flex items-center gap-2">
                            Main Dining Room 
                            <span class="px-2 py-0.5 rounded-md bg-green-50 dark:bg-green-900/20 text-green-600 dark:text-green-400 text-[10px] uppercase font-bold border border-green-100 dark:border-green-900/30">Live</span>
                        </div>
                    </div>
                </div>
                
                <div class="flex justify-end gap-3">
                     <button id="editLayoutBtn" onclick="toggleLayoutEditMode(true)" class="px-5 py-2 bg-primary hover:bg-orange-600 text-white rounded-xl text-xs font-bold shadow-lg shadow-orange-500/30 hover:shadow-orange-500/50 transition-all transform hover:-translate-y-0.5 flex items-center gap-2">
                        <ion-icon name="create-outline" class="text-lg"></ion-icon> Edit Layout
                     </button>
                     
                     <div id="editControls" class="hidden flex items-center gap-3">
                         <button onclick="document.getElementById('canvasGrid').innerHTML = ''" class="px-4 py-2 bg-white dark:bg-slate-800 border border-gray-200 dark:border-gray-700 text-gray-700 dark:text-gray-200 rounded-xl text-xs font-bold shadow-sm hover:shadow-md hover:bg-gray-50 dark:hover:bg-slate-700 transition-all flex items-center gap-2">
                            <ion-icon name="trash-outline" class="text-lg text-red-500"></ion-icon> Clear All
                        </button>
                        <button onclick="toggleLayoutEditMode(false)" class="px-5 py-2 bg-primary hover:bg-orange-600 text-white rounded-xl text-xs font-bold shadow-lg shadow-orange-500/30 hover:shadow-orange-500/50 transition-all transform hover:-translate-y-0.5 flex items-center gap-2">
                            <ion-icon name="save-outline" class="text-lg"></ion-icon> Save Changes
                        </button>
                     </div>
                </div>
            </div>
            
            <!-- Grid Background -->
            <div id="canvasGrid" class="flex-1 relative overflow-auto custom-scrollbar p-10 cursor-crosshair group/canvas" style="background-image: radial-gradient(#94a3b8 1px, transparent 1px); background-size: 24px 24px;">
                <!-- Tables and Objects will be injected here -->
            </div>
        </div>
    </div>
</main>

<!-- Config Modal (Table) -->
<div id="configModal" class="hidden fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm opacity-0 transition-opacity duration-300">
    <div class="bg-white dark:bg-slate-800 rounded-2xl shadow-2xl w-full max-w-sm p-6 transform scale-95 transition-transform duration-300">
        <h3 class="text-lg font-bold text-gray-800 dark:text-white mb-4">Table Configuration</h3>
        <div class="space-y-4">
            <div>
                <label class="block text-xs font-bold text-gray-500 uppercase mb-1">Table Type</label>
                <div class="flex items-center gap-2 px-3 py-2 bg-gray-100 dark:bg-slate-700 rounded-lg text-sm text-gray-700 dark:text-gray-300">
                    <span id="modalTypeIcon"></span>
                    <span id="modalTypeText" class="font-medium"></span>
                </div>
            </div>
            <div>
                 <label class="block text-xs font-bold text-gray-500 uppercase mb-1">Table Name / Number</label>
                 <input type="text" id="tableName" placeholder="e.g. T-12" class="block w-full bg-gray-50 dark:bg-slate-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-white text-sm rounded-lg p-2.5 font-bold outline-none focus:outline-none focus:border-primary focus:ring-0">
            </div>
            <div>
                 <label class="block text-xs font-bold text-gray-500 uppercase mb-1">Number of Seats</label>
                 <div class="flex items-center gap-3">
                     <button onclick="adjustSeats(-1)" class="w-10 h-10 rounded-lg border border-gray-300 dark:border-gray-600 flex items-center justify-center hover:bg-gray-50 dark:hover:bg-slate-700 font-bold text-lg outline-none focus:outline-none focus:border-primary transition-colors">-</button>
                     <input type="number" id="seatCount" value="4" class="block w-full text-center bg-white dark:bg-slate-900 border-2 border-primary dark:border-primary text-gray-900 dark:text-white text-sm rounded-lg p-2.5 font-bold outline-none focus:outline-none focus:ring-0" readonly>
                     <button onclick="adjustSeats(1)" class="w-10 h-10 rounded-lg border border-gray-300 dark:border-gray-600 flex items-center justify-center hover:bg-gray-50 dark:hover:bg-slate-700 font-bold text-lg outline-none focus:outline-none focus:border-primary transition-colors">+</button>
                 </div>
            </div>
        </div>
        <div class="flex gap-3 mt-6">
            <button onclick="closeModal(false)" class="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 rounded-xl font-bold hover:bg-gray-50 dark:hover:bg-slate-700">Cancel</button>
            <button onclick="confirmConfig()" class="flex-1 px-4 py-2 bg-primary text-white rounded-xl font-bold hover:bg-orange-600" id="confirmBtn">Confirm</button>
        </div>
    </div>
</div>

<!-- Zone Modal -->
<div id="zoneModal" class="hidden fixed inset-0 z-50 flex items-center justify-center bg-black/50 backdrop-blur-sm opacity-0 transition-opacity duration-300">
    <div class="bg-white dark:bg-slate-800 rounded-2xl shadow-2xl w-full max-w-sm p-6 transform scale-95 transition-transform duration-300">
        <h3 class="text-lg font-bold text-gray-800 dark:text-white mb-4" id="zoneModalTitle">Add New Zone</h3>
        
        <div class="space-y-4">
            <div>
                <label class="block text-xs font-bold text-gray-500 uppercase mb-1">Zone Name</label>
                <input type="text" id="zoneName" placeholder="e.g. Roof Top" class="block w-full bg-gray-50 dark:bg-slate-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-white text-sm rounded-lg p-2.5 outline-none focus:outline-none focus:border-primary focus:ring-0">
            </div>

            <div>
                 <div class="flex items-center justify-between">
                     <label class="text-xs font-bold text-gray-500 uppercase">Status</label>
                     <label class="relative inline-flex items-center cursor-pointer">
                        <input type="checkbox" id="zoneActiveToggle" class="sr-only peer" checked onchange="toggleZoneTimeInputs()">
                        <div class="w-11 h-6 bg-gray-200 peer-focus:outline-none peer-focus:ring-4 peer-focus:ring-orange-300 dark:peer-focus:ring-orange-800 rounded-full peer dark:bg-gray-700 peer-checked:after:translate-x-full peer-checked:after:border-white after:content-[''] after:absolute after:top-[2px] after:left-[2px] after:bg-white after:border-gray-300 after:border after:rounded-full after:h-5 after:w-5 after:transition-all dark:border-gray-600 peer-checked:bg-primary"></div>
                        <span class="ml-3 text-sm font-medium text-gray-900 dark:text-gray-300" id="zoneStatusLabel">Active</span>
                     </label>
                 </div>
            </div>
            
            <!-- Reserved Party Option Removed as per request -->
            <div id="reservedPartyOption" class="hidden flex items-center gap-2 mt-2">
                 <input type="checkbox" id="zonePartyCheck" class="w-4 h-4 text-primary bg-gray-100 border-gray-300 rounded focus:ring-primary dark:focus:ring-primary dark:ring-offset-gray-800 focus:ring-2 dark:bg-gray-700 dark:border-gray-600" onchange="toggleZoneTimeInputs()">
                 <label for="zonePartyCheck" class="text-sm font-medium text-gray-900 dark:text-gray-300">Reserved for Party</label>
            </div>

            <div id="partyTimeContainer" class="hidden">
                 <label class="block text-xs font-bold text-gray-500 uppercase mb-1">Party Time</label>
                 <div class="flex items-center gap-2">
                     <input type="time" id="partyStart" value="19:00" class="flex-1 bg-gray-50 dark:bg-slate-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-white text-sm rounded-lg p-2.5 outline-none focus:outline-none focus:border-primary focus:ring-0">
                     <span class="text-gray-400 font-bold">-</span>
                     <input type="time" id="partyEnd" value="22:00" class="flex-1 bg-gray-50 dark:bg-slate-900 border border-gray-300 dark:border-gray-700 text-gray-900 dark:text-white text-sm rounded-lg p-2.5 outline-none focus:outline-none focus:border-primary focus:ring-0">
                 </div>
            </div>
        </div>

        <div class="flex gap-3 mt-6">
            <button onclick="closeZoneModal()" class="flex-1 px-4 py-2 border border-gray-300 dark:border-gray-600 text-gray-700 dark:text-gray-300 rounded-xl font-bold hover:bg-gray-50 dark:hover:bg-slate-700">Cancel</button>
            <button onclick="saveZone()" class="flex-1 px-4 py-2 bg-primary text-white rounded-xl font-bold hover:bg-orange-600">Save Zone</button>
        </div>
    </div>
</div>

<style>
    /* 8-Point Resize Handles */
    .resize-handle {
        position: absolute;
        width: 8px;
        height: 8px;
        background-color: white;
        border: 1px solid #FF9909;
        border-radius: 50%;
        z-index: 50;
        display: none;
    }
    .selected .resize-handle { display: block; }
    .selected { outline: 1px dashed #FF9909; }

    /* Cursor Directions */
    .handle-nw { top: -4px; left: -4px; cursor: nw-resize; }
    .handle-n  { top: -4px; left: 50%; margin-left: -4px; cursor: n-resize; }
    .handle-ne { top: -4px; right: -4px; cursor: ne-resize; }
    .handle-w  { top: 50%; left: -4px; margin-top: -4px; cursor: w-resize; }
    .handle-e  { top: 50%; right: -4px; margin-top: -4px; cursor: e-resize; }
    .handle-sw { bottom: -4px; left: -4px; cursor: sw-resize; }
    .handle-s  { bottom: -4px; left: 50%; margin-left: -4px; cursor: s-resize; }
    .handle-se { bottom: -4px; right: -4px; cursor: se-resize; }

    /* Floating Toolbar */
    .context-toolbar {
        position: absolute;
        top: -45px;
        left: 50%;
        transform: translateX(-50%);
        background: white;
        border-radius: 8px;
        box-shadow: 0 4px 12px rgba(0,0,0,0.15);
        padding: 4px;
        display: none;
        gap: 2px;
        z-index: 100;
        white-space: nowrap;
    }
    .dark .context-toolbar { background: #1e293b; border: 1px solid #334155; }
    .selected .context-toolbar { display: flex; }

    /* Chairs */
    .chair {
        position: absolute;
        width: 16px;
        height: 16px;
        background: #e2e8f0;
        border: 1px solid #94a3b8;
        border-radius: 4px;
        z-index: 5;
    }
    .dark .chair { background: #475569; border: 1px solid #64748b; }
    .chair.round { border-radius: 50%; }

    /* Reserved Badge */
    .reserved-badge {
        position: absolute;
        top: -10px;
        right: -10px;
        background: #ef4444;
        color: white;
        font-size: 8px;
        font-weight: bold;
        padding: 2px 6px;
        border-radius: 12px;
        z-index: 60;
        box-shadow: 0 2px 4px rgba(0,0,0,0.2);
        line-height: 1;
        text-align: center;
    }
    
    /* Edit Mode Styles */
    #zonesList.edit-mode .edit-tools {
        opacity: 1;
        pointer-events: auto;
    }
    #zonesList.edit-mode .group\/zone button {
        padding-right: 5rem; 
    }
</style>

<script>
    // --- Global State ---
    let activeItem = null;
    let dragSource = null;    
    let dragAction = null;   
    let activeDragType = null;
    let ghostEl = null;
    let nextId = 1;
    let editingZoneId = null; 
    let isLayoutEditing = false; 

    // Drag State
    let initialX = 0, initialY = 0;
    let initialLeft = 0, initialTop = 0;
    let initialWidth = 0, initialHeight = 0;

    // Config State
    let pendingCreation = null; 
    let editingItem = null; 

    // --- DOM Elements (Initialized in DOMContentLoaded) ---
    let canvas = null;
    let configModal = null;
    let seatInput = null;
    let nameInput = null;
    let zoneModal = null;

    // --- initialization ---
    document.addEventListener('DOMContentLoaded', () => {
        canvas = document.getElementById('canvasGrid');
        configModal = document.getElementById('configModal');
        seatInput = document.getElementById('seatCount');
        nameInput = document.getElementById('tableName');
        zoneModal = document.getElementById('zoneModal');
        
        // Initial Mock Data
        setTimeout(() => {
             spawnElement('round', 100, 100, { seats: 4, reserved: false, load: true });
             spawnElement('rect', 300, 100, { seats: 8, reserved: false, load: true });
        }, 100);

        // Canvas Events
        if(canvas) {
            canvas.addEventListener('mousedown', (e) => {
                if (e.target === canvas) clearSelection();
            });
        }

        // Sidebar Events
        document.querySelectorAll('.draggable-tool').forEach(tool => {
            // Init Disabled State
            tool.classList.add('opacity-50', 'pointer-events-none', 'cursor-default');
            tool.classList.remove('cursor-grab', 'active:cursor-grabbing');
            
            tool.addEventListener('mousedown', (e) => startSidebarDrag(e, tool));
            tool.addEventListener('touchstart', (e) => startSidebarDrag(e, tool));
        });
    });

    // --- Global Functions ---

    window.clearSelection = function() {
        document.querySelectorAll('.draggable-item').forEach(el => {
            el.classList.remove('selected', 'z-50');
            if(el.dataset.zIndex) el.style.zIndex = el.dataset.zIndex;
        });
        activeItem = null;
    }

    window.spawnElement = function(type, x, y, config = {}) {
        // Drag Drop Prompt
        // Drag Drop Prompt (Disabled for creation to auto-increment)
        // if (!config.bypassPrompt && ['round', 'square', 'rect', 'booth'].includes(type) && !config.load) {
        //    pendingCreation = { type, x, y };
        //    openModal(type, null); // New Item
        //    return;
        // }

        const id = config.name || ((['round', 'square', 'rect', 'booth'].includes(type)) ? `T-0${nextId++}` : '');
        const el = document.createElement('div');
        el.className = 'draggable-item absolute group';
        el.dataset.type = type;
        el.dataset.id = nextId;
        el.dataset.label = id;
        el.dataset.zIndex = 10;
        el.style.zIndex = 10;
        el.style.left = x + 'px';
        el.style.top = y + 'px';

        let content = '';
        let style = '';
        
        // Default to 4 seats for tables if not specified
        let seats = config.seats;
        if (seats === undefined) {
             seats = (['round', 'square', 'rect', 'booth'].includes(type)) ? 4 : 0;
        }
        
        let reserved = config.reserved || false;
        el.dataset.seats = seats;
        el.dataset.reserved = reserved;

        // Styles
        switch(type) {
            case 'round':
                style = 'width: 96px; height: 96px; border-radius: 50%; background: white; border: 2px solid #cbd5e1;';
                content = `<div class="flex items-center justify-center w-full h-full pointer-events-none font-bold text-gray-500 text-xs select-none table-label">${id}</div>`;
                break;
            case 'square':
                style = 'width: 96px; height: 96px; border-radius: 12px; background: white; border: 2px solid #cbd5e1;';
                content = `<div class="flex items-center justify-center w-full h-full pointer-events-none font-bold text-gray-500 text-xs select-none table-label">${id}</div>`;
                break;
            case 'rect':
                style = 'width: 160px; height: 96px; border-radius: 8px; background: white; border: 2px solid #cbd5e1;';
                content = `<div class="flex items-center justify-center w-full h-full pointer-events-none font-bold text-gray-500 text-xs select-none table-label">${id}</div>`;
                break;
            case 'booth':
                style = 'width: 160px; height: 112px; border-radius: 16px; background: white; border: 2px solid #cbd5e1;';
                content = `<div class="flex flex-col items-center justify-center w-full h-full pointer-events-none font-bold text-gray-500 text-xs select-none table-label"><span>${id}</span><span class="text-[8px] text-gray-400">Booth</span></div>`;
                break;
            case 'wall': style = 'width: 16px; height: 160px; background: #94a3b8; border: 1px solid #64748b;'; break;
            case 'window': style = 'width: 16px; height: 160px; background: #bfdbfe; border-left: 1px solid #60a5fa; border-right: 1px solid #60a5fa; opacity: 0.8;'; break;
            case 'pillar': style = 'width: 48px; height: 48px; background: #cbd5e1; border: 1px solid #94a3b8;'; break;
            case 'door': style = 'width: 64px; height: 64px; border-top: 4px solid #64748b; border-left: 4px solid #64748b; border-top-left-radius: 100%; border-right: 1px dashed #cbd5e1; border-bottom: 1px dashed #cbd5e1;'; break;
            case 'cashier': 
                style = 'width: 112px; height: 80px; background: #f1f5f9; border: 2px solid #cbd5e1; border-radius: 8px;'; 
                content = '<div class="flex flex-col items-center justify-center w-full h-full pointer-events-none text-gray-500"><ion-icon name="desktop-outline" class="text-2xl"></ion-icon><span class="text-[10px] font-bold">POS</span></div>'; break;
            case 'plant': 
                style = 'width: 64px; height: 64px; border-radius: 50%; background: #ecfccb; border: 2px solid #84cc16;'; 
                content = '<div class="flex items-center justify-center w-full h-full pointer-events-none text-green-600"><ion-icon name="leaf" class="text-3xl"></ion-icon></div>'; break;
        }

        el.style.cssText += style + " left: " + x + "px; top: " + y + "px; position: absolute;";

        // HTML Structure
        const handles = ['nw','n','ne','w','e','sw','s','se'].map(d => `<div class="resize-handle handle-${d}" data-dir="${d}"></div>`).join('');
        const toolbar = `
            <div class="context-toolbar">
                <button class="p-1 hover:bg-gray-100 rounded text-gray-600" onclick="changeLayer(this, 1)" title="Front"><ion-icon name="arrow-up-outline"></ion-icon></button>
                <button class="p-1 hover:bg-gray-100 rounded text-gray-600" onclick="changeLayer(this, -1)" title="Back"><ion-icon name="arrow-down-outline"></ion-icon></button>
                <div class="w-px bg-gray-200 h-4 mx-1 self-center"></div>
                <button class="p-1 hover:bg-red-50 rounded text-red-500" onclick="deleteElement(this)" title="Delete"><ion-icon name="trash-outline"></ion-icon></button>
            </div>`;
        
        // Reservation Badge Wrapper
        const badgeWrapper = document.createElement('div');
        badgeWrapper.className = 'badge-wrapper pointer-events-none';
        if(reserved) {
            badgeWrapper.innerHTML = `<div class="reserved-badge">RESERVED<br><span class="opacity-80 font-normal">19:00-20:30</span></div>`;
        }

        el.innerHTML = handles + toolbar + content;
        el.appendChild(badgeWrapper);

        if(seats > 0) renderChairs(el, type, seats);

        if(canvas) canvas.appendChild(el);
        setupCanvasInteraction(el);
        
        clearSelection();
        el.classList.add('selected', 'z-50');
        activeItem = el;
    }

    function renderChairs(el, type, count) {
        el.querySelectorAll('.chair').forEach(c => c.remove());
        el.dataset.seats = count;

        if (count > 0 && ['round', 'square', 'rect'].includes(type)) {
                for (let i = 0; i < count; i++) {
                const angle = (2 * Math.PI / count) * i;
                const chair = document.createElement('div');
                chair.className = 'chair ' + (type === 'round' ? 'round' : '');
                
                const x = Math.cos(angle);
                const y = Math.sin(angle);
                let dist = 65; 
                if(type === 'rect') dist = 70;

                const top = 50 + y * dist; 
                const left = 50 + x * dist;

                chair.style.top = `calc(${top}% - 8px)`; 
                chair.style.left = `calc(${left}% - 8px)`;
                chair.style.transform = `rotate(${angle * (180/Math.PI) + 90}deg)`;
                
                el.appendChild(chair);
            }
        }
    }

    // --- Drag & Drop & Interaction ---
    function setupCanvasInteraction(item) {
        item.addEventListener('mousedown', (e) => startInteraction(e, item));
        item.addEventListener('touchstart', (e) => startInteraction(e, item));
        
        if (['round', 'square', 'rect', 'booth'].includes(item.dataset.type)) {
            item.addEventListener('dblclick', (e) => {
                if(!isLayoutEditing) return;
                e.stopPropagation();
                openModal(item.dataset.type, item);
            });
        }
    }

    function startInteraction(e, item) {
        if (!isLayoutEditing) return;
        if (e.target.closest('button')) return;
        
        const isHandle = e.target.classList.contains('resize-handle');
        const interactionType = isHandle ? ('resize-' + e.target.dataset.dir) : 'move';

        clearSelection();
        item.classList.add('selected', 'z-50'); 
        activeItem = item;
        dragAction = interactionType;
        dragSource = 'canvas';

        const rect = item.getBoundingClientRect();
        
        if (e.type === 'touchstart') {
            initialX = e.touches[0].clientX;
            initialY = e.touches[0].clientY;
        } else {
            initialX = e.clientX;
            initialY = e.clientY;
        }

        initialLeft = item.offsetLeft;
        initialTop = item.offsetTop;
        initialWidth = rect.width;
        initialHeight = rect.height;

        document.addEventListener('mouseup', endDrag);
        document.addEventListener('mousemove', drag);
        document.addEventListener('touchend', endDrag);
        document.addEventListener('touchmove', drag);
    }

    function drag(e) {
        if (!activeItem || !dragSource) return;
        e.preventDefault();

        const cx = e.type.includes('touch') ? e.touches[0].clientX : e.clientX;
        const cy = e.type.includes('touch') ? e.touches[0].clientY : e.clientY;
        const dx = cx - initialX;
        const dy = cy - initialY;
        const snap = 12;

        if (dragAction === 'move') {
            let nl = initialLeft + dx;
            let nt = initialTop + dy;
            // Snap
            nl = Math.round(nl / snap) * snap;
            nt = Math.round(nt / snap) * snap;
            activeItem.style.left = nl + 'px';
            activeItem.style.top = nt + 'px';
        } else {
            const dir = dragAction.split('-')[1];
            let nw = initialWidth;
            let nh = initialHeight; 
            let nl = initialLeft; 
            let nt = initialTop;

            // Simple Resize Logic
            if (dir.includes('e')) nw = initialWidth + dx;
            if (dir.includes('s')) nh = initialHeight + dy;
            if (dir.includes('w')) { 
                nw = initialWidth - dx; 
                nl = initialLeft + dx; 
            }
            if (dir.includes('n')) { 
                nh = initialHeight - dy; 
                nt = initialTop + dy; 
            }
            
            // Min Constraints
            if(nw < 32) nw = 32; if(nh < 32) nh = 32;
            
            // Snap
            nw = Math.round(nw/snap)*snap;
            nh = Math.round(nh/snap)*snap;
            if (dir.includes('w')) nl = Math.round(nl/snap)*snap;
            if (dir.includes('n')) nt = Math.round(nt/snap)*snap;

            activeItem.style.width = nw + 'px';
            activeItem.style.height = nh + 'px';
            if (dir.includes('w')) activeItem.style.left = nl + 'px';
            if (dir.includes('n')) activeItem.style.top = nt + 'px';
        }
    }

    function endDrag() {
        document.removeEventListener('mouseup', endDrag);
        document.removeEventListener('mousemove', drag);
        document.removeEventListener('touchend', endDrag);
        document.removeEventListener('touchmove', drag);
        dragSource = null;
    }

    // --- Sidebar Dragging (Global) ---
    window.startSidebarDrag = function(e, tool) {
        if (!isLayoutEditing) return;
        e.preventDefault();
        dragSource = 'sidebar';
        activeDragType = tool.dataset.type;

        ghostEl = tool.cloneNode(true);
        ghostEl.style.position = 'fixed';
        ghostEl.style.zIndex = 1000;
        ghostEl.style.opacity = '0.8';
        ghostEl.style.width = tool.offsetWidth + 'px';
        ghostEl.classList.add('bg-white', 'shadow-xl');
        document.body.appendChild(ghostEl);
        
        updateGhost(e);
        
        document.addEventListener('mouseup', endSidebarDrag);
        document.addEventListener('mousemove', updateGhost);
        document.addEventListener('touchend', endSidebarDrag);
        document.addEventListener('touchmove', updateGhost);
    }

    function updateGhost(e) {
        if (!ghostEl) return;
        const cx = e.type.includes('touch') ? e.touches[0].clientX : e.clientX;
        const cy = e.type.includes('touch') ? e.touches[0].clientY : e.clientY;
        ghostEl.style.left = (cx - ghostEl.offsetWidth/2) + 'px';
        ghostEl.style.top = (cy - ghostEl.offsetHeight/2) + 'px';
    }

    function endSidebarDrag(e) {
        document.removeEventListener('mouseup', endSidebarDrag);
        document.removeEventListener('mousemove', updateGhost);
        document.removeEventListener('touchend', endSidebarDrag);
        document.removeEventListener('touchmove', updateGhost);

        if (dragSource === 'sidebar') {
            const cx = e.type.includes('touch') ? e.changedTouches[0].clientX : e.clientX;
            const cy = e.type.includes('touch') ? e.changedTouches[0].clientY : e.clientY;
            
            // Re-fetch canvas in case it wasn't init (safety)
            if(!canvas) canvas = document.getElementById('canvasGrid');
            const rect = canvas.getBoundingClientRect();
            
            if (cx >= rect.left && cx <= rect.right && cy >= rect.top && cy <= rect.bottom) {
                spawnElement(activeDragType, cx - rect.left, cy - rect.top, { bypassPrompt: false });
            }
        }
        if (ghostEl) ghostEl.remove();
        dragSource = null;
    }

    // --- Zone Functions (Global) ---
    window.toggleZoneEditMode = function(btn) {
        const list = document.getElementById('zonesList');
        if (list.classList.contains('edit-mode')) {
            list.classList.remove('edit-mode');
            btn.innerText = 'Edit All';
            btn.classList.remove('text-green-600');
            btn.classList.add('text-primary');
        } else {
            list.classList.add('edit-mode');
            btn.innerText = 'Done';
            btn.classList.remove('text-primary');
            btn.classList.add('text-green-600');
        }
    }

    window.openZoneModal = function(id = null) {
        if(!zoneModal) zoneModal = document.getElementById('zoneModal');
        zoneModal.classList.remove('hidden');
        setTimeout(() => {
            zoneModal.classList.remove('opacity-0');
            zoneModal.querySelector('div').classList.remove('scale-95');
            zoneModal.querySelector('div').classList.add('scale-100');
        }, 10);
        
        const title = document.getElementById('zoneModalTitle');
        const nameInput = document.getElementById('zoneName');
        const activeToggle = document.getElementById('zoneActiveToggle');
        const partyCheck = document.getElementById('zonePartyCheck');
        
        editingZoneId = id;
        
        if (id) {
            title.innerText = "Edit Zone";
            const zoneEl = document.getElementById('zone-' + id);
            nameInput.value = zoneEl.dataset.name;
            const status = zoneEl.dataset.status;
            
            if (status === 'active') {
                activeToggle.checked = true;
                partyCheck.checked = false;
            } else if (status === 'inactive') {
                activeToggle.checked = false;
                partyCheck.checked = false;
            } else if (status === 'party') {
                activeToggle.checked = true;
                partyCheck.checked = true;
            }
        } else {
            title.innerText = "Add New Zone";
            nameInput.value = "";
            activeToggle.checked = true;
            partyCheck.checked = false;
        }
        
        window.toggleZoneTimeInputs();
    }

    window.closeZoneModal = function() {
        if(!zoneModal) zoneModal = document.getElementById('zoneModal');
        zoneModal.classList.add('opacity-0');
        zoneModal.querySelector('div').classList.remove('scale-100');
        zoneModal.querySelector('div').classList.add('scale-95');
        setTimeout(() => {
            zoneModal.classList.add('hidden');
        }, 300);
        editingZoneId = null;
    }

    window.saveZone = function() {
        const name = document.getElementById('zoneName').value;
        const isActive = document.getElementById('zoneActiveToggle').checked;
        const isParty = document.getElementById('zonePartyCheck').checked;
        
        let status = 'inactive';
        if (isActive) {
            status = isParty ? 'party' : 'active';
        }

        const start = document.getElementById('partyStart').value;
        const end = document.getElementById('partyEnd').value;
        
        if(!name) return;
        
        let zoneEl;
        const list = document.getElementById('zonesList');

        if (editingZoneId) {
            zoneEl = document.getElementById('zone-' + editingZoneId);
        } else {
            const newId = Date.now();
            zoneEl = document.createElement('div');
            zoneEl.className = 'relative group/zone';
            zoneEl.id = 'zone-' + newId;
            zoneEl.dataset.id = newId;
            list.appendChild(zoneEl);
        }

        zoneEl.dataset.name = name;
        zoneEl.dataset.status = status;
        
        const isZoneActive = (status === 'active' || status === 'party');
        const badgeHTML = (status === 'party') ? 
            `<div class="zone-badge mt-1 flex flex-col items-start">
                <span class="text-[10px] text-red-500 font-bold bg-red-50 dark:bg-red-900/20 px-1.5 py-0.5 rounded-md border border-red-100 dark:border-red-900/30">Reserved Party</span>
                <span class="text-[9px] text-red-400 font-medium ml-0.5">${start} - ${end}</span>
             </div>` : '';
        
        const activeClass = isZoneActive ? 'bg-orange-50 dark:bg-primary/10 border-orange-200 dark:border-primary/30 text-primary shadow-sm' : 'bg-white dark:bg-slate-800 border-transparent hover:border-gray-200 dark:hover:border-gray-700 text-gray-600 dark:text-gray-400 hover:bg-gray-50 dark:hover:bg-slate-700/50';
        const dotClass = isZoneActive ? 'bg-orange-500 animate-pulse' : 'bg-gray-300 dark:bg-gray-600 group-hover:bg-gray-400';
        const countClass = isZoneActive ? 'text-orange-400' : 'text-gray-400';
        const tableCount = zoneEl.dataset.tables || 0;

        zoneEl.innerHTML = `
            <button class="w-full flex items-center justify-between px-4 py-3 rounded-xl text-sm font-medium transition-all duration-200 border ${activeClass}">
                <div class="flex items-center gap-3">
                    <span class="w-1.5 h-1.5 rounded-full ${dotClass}"></span>
                    <div class="flex flex-col items-start text-left">
                        <span class="font-bold zone-name">${name}</span>
                        ${badgeHTML}
                    </div>
                </div>
                <span class="text-xs ${countClass} tables-count">${tableCount}</span>
            </button>
            <div class="edit-tools absolute right-2 top-1/2 -translate-y-1/2 flex items-center gap-1 opacity-0 pointer-events-none transition-opacity duration-200">
                <button onclick="openZoneModal(${editingZoneId || zoneEl.dataset.id})" class="p-1.5 bg-white dark:bg-slate-700 rounded-lg shadow-sm border border-gray-200 dark:border-gray-600 text-gray-500 hover:text-primary hover:border-primary"><ion-icon name="create-outline"></ion-icon></button>
                <button onclick="deleteZone(${editingZoneId || zoneEl.dataset.id})" class="p-1.5 bg-white dark:bg-slate-700 rounded-lg shadow-sm border border-gray-200 dark:border-gray-600 text-gray-500 hover:text-red-500 hover:border-red-500"><ion-icon name="trash-outline"></ion-icon></button>
            </div>
        `;
        
        closeZoneModal();
    }
    
    window.toggleZoneTimeInputs = function() {
        const activeToggle = document.getElementById('zoneActiveToggle');
        const partyCheck = document.getElementById('zonePartyCheck');
        const partyOptionDiv = document.getElementById('reservedPartyOption');
        const partyTimeContainer = document.getElementById('partyTimeContainer');
        const statusLabel = document.getElementById('zoneStatusLabel');

        // Update Label
        statusLabel.innerText = activeToggle.checked ? 'Active' : 'Inactive';

        // Show/Hide Party Checkbox based on Active Status
        if (activeToggle.checked) {
            partyOptionDiv.classList.remove('opacity-50', 'pointer-events-none');
        } else {
            partyOptionDiv.classList.add('opacity-50', 'pointer-events-none');
            partyCheck.checked = false; // Auto uncheck if inactive
        }

        // Show/Hide Time Inputs
        if(activeToggle.checked && partyCheck.checked) {
            partyTimeContainer.classList.remove('hidden');
        } else {
            partyTimeContainer.classList.add('hidden');
        }
    }

    // --- Helper Functions ---
    window.changeLayer = function(btn, d) {
        if (!isLayoutEditing) return;
        const el = btn.closest('.draggable-item');
        let z = parseInt(el.dataset.zIndex || 10) + d;
        if(z<1) z=1;
        el.dataset.zIndex = z;
        // Always apply the new z-index so the user sees the change immediately
        el.style.zIndex = z;
    }
    window.deleteElement = function(btn) {
        if (!isLayoutEditing) return;
        btn.closest('.draggable-item').remove();
    }
    
    window.toggleLayoutEditMode = function(edit) {
        isLayoutEditing = edit;
        const editBtn = document.getElementById('editLayoutBtn');
        const controls = document.getElementById('editControls');
        if (edit) {
            editBtn.classList.add('hidden');
            controls.classList.remove('hidden');
            
            // Enable Sidebar Tools
            document.querySelectorAll('.draggable-tool').forEach(t => {
                t.classList.remove('opacity-50', 'pointer-events-none', 'cursor-default');
                t.classList.add('cursor-grab', 'active:cursor-grabbing');
                t.style.opacity = ''; // Remove inline style if any
                t.style.cursor = '';
            });
            
            // Enable Add Zone
            if(addZoneBtn) {
                addZoneBtn.classList.remove('opacity-50', 'pointer-events-none', 'cursor-default');
            }
            
            clearSelection();
        } else {
            editBtn.classList.remove('hidden');
            controls.classList.add('hidden');
            
            // Disable Sidebar Tools
            document.querySelectorAll('.draggable-tool').forEach(t => {
                t.classList.add('opacity-50', 'pointer-events-none', 'cursor-default');
                t.classList.remove('cursor-grab', 'active:cursor-grabbing');
            });
            
            // Disable Add Zone
            if(addZoneBtn) {
                 addZoneBtn.classList.add('opacity-50', 'pointer-events-none', 'cursor-default');
            }
            
            clearSelection();
        }
    }
    
    // --- Modal Config Functions ---
    
    function openModal(type, existingItem) {
        if(!configModal) configModal = document.getElementById('configModal');
        if(!seatInput) seatInput = document.getElementById('seatCount');
        if(!nameInput) nameInput = document.getElementById('tableName');

        configModal.classList.remove('hidden');
        setTimeout(() => {
            configModal.classList.remove('opacity-0');
            configModal.querySelector('div').classList.remove('scale-95');
            configModal.querySelector('div').classList.add('scale-100');
        }, 10);
        
        document.getElementById('modalTypeText').innerText = type.charAt(0).toUpperCase() + type.slice(1) + ' Table';
        document.getElementById('modalTypeIcon').innerHTML = ''; 

        // Update Confirm Button Reference (Local scope safety)
        const confirmBtn = document.getElementById('confirmBtn');

        if (existingItem) {
            // Formatting for Edit
            editingItem = existingItem;
            confirmBtn.innerText = "Update";
            seatInput.value = existingItem.dataset.seats || 4;
            nameInput.value = existingItem.dataset.label || "Table";
        } else {
            // Formatting for New
            editingItem = null;
            confirmBtn.innerText = "Create Table";
            seatInput.value = 4;
            nameInput.value = `T-0${nextId}`;
        }
    }

    window.closeModal = function(save) {
        if(!configModal) configModal = document.getElementById('configModal');
        configModal.classList.add('opacity-0');
        configModal.querySelector('div').classList.remove('scale-100');
        configModal.querySelector('div').classList.add('scale-95');
        setTimeout(() => {
            configModal.classList.add('hidden');
        }, 300);
        
        if (!save) {
            pendingCreation = null;
            editingItem = null;
        }
    }

    window.adjustSeats = function(d) {
        if(!seatInput) seatInput = document.getElementById('seatCount');
        let v = parseInt(seatInput.value) + d;
        if (v < 1) v = 1;
        if (v > 12) v = 12;
        seatInput.value = v;
    }

    window.confirmConfig = function() {
        if(!seatInput) seatInput = document.getElementById('seatCount');
        if(!nameInput) nameInput = document.getElementById('tableName');
        
        const seats = parseInt(seatInput.value);
        const name = nameInput.value;
        const isReserved = false; 

        if (editingItem) {
            // Update Existing
            renderChairs(editingItem, editingItem.dataset.type, seats);
            editingItem.dataset.reserved = isReserved;
            editingItem.dataset.label = name;
            
            // Update Visual Label
            const labelEl = editingItem.querySelector('.table-label');
            if(labelEl) {
                // If booth, it has span structure
                if(editingItem.dataset.type === 'booth') {
                     labelEl.innerHTML = `<span>${name}</span><span class="text-[8px] text-gray-400">Booth</span>`;
                } else {
                     labelEl.innerText = name;
                }
            }
            
            const badgeWrapper = editingItem.querySelector('.badge-wrapper');
            if (badgeWrapper) badgeWrapper.innerHTML = ''; 
            
            closeModal(true);
        } else if (pendingCreation) {
            // Create New
            spawnElement(pendingCreation.type, pendingCreation.x, pendingCreation.y, { 
                seats: seats,
                name: name,
                reserved: isReserved,
                bypassPrompt: true
            });
            closeModal(true);
        }
    }
</script>
