<?php
session_start();

// --- BACKEND LOGIC ---
$RATE_LIMIT_MAX_ATTEMPTS = 5;
$RATE_LIMIT_WINDOW = 300; // 5 minutes
$LOG_FILE = __DIR__ . '/logs/audit.log';
$MOCK_USER = [
    'email' => 'admin@cashbox.com',
    'password' => 'password123' 
];

// Helper Functions
function sendJson($success, $message = '') {
    header('Content-Type: application/json');
    echo json_encode(['success' => $success, 'message' => $message]);
    exit;
}

function logEvent($event, $email) {
    global $LOG_FILE;
    if (!is_dir(dirname($LOG_FILE))) {
        mkdir(dirname($LOG_FILE), 0777, true);
    }
    $date = date('Y-m-d H:i:s');
    $maskedEmail = substr($email, 0, 3) . '***@' . explode('@', $email)[1];
    $logEntry = "[$date] [$event] User: $maskedEmail - IP: " . ($_SERVER['REMOTE_ADDR'] ?? 'UNKNOWN') . PHP_EOL;
    file_put_contents($LOG_FILE, $logEntry, FILE_APPEND);
}

// Generate CSRF Token
if (empty($_SESSION['csrf_token'])) {
    $_SESSION['csrf_token'] = bin2hex(random_bytes(32));
}

// Handle POST Request (AJAX Login)
if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    $email = $_POST['email'] ?? '';
    $password = $_POST['password'] ?? '';
    $csrf_token = $_POST['csrf_token'] ?? '';

    // Validate CSRF
    if (empty($csrf_token) || $csrf_token !== $_SESSION['csrf_token']) {
        sendJson(false, 'Invalid Session. Please refresh.');
    }

    // Rate Limiting
    if (!isset($_SESSION['login_attempts'])) {
        $_SESSION['login_attempts'] = 0;
        $_SESSION['first_attempt_time'] = time();
    }

    if ($_SESSION['login_attempts'] >= $RATE_LIMIT_MAX_ATTEMPTS) {
        $timeSinceFirst = time() - $_SESSION['first_attempt_time'];
        if ($timeSinceFirst < $RATE_LIMIT_WINDOW) {
            $wait = ceil(($RATE_LIMIT_WINDOW - $timeSinceFirst) / 60);
            sendJson(false, "Too many failed attempts. Please try again in $wait minutes.");
        } else {
            $_SESSION['login_attempts'] = 0;
            $_SESSION['first_attempt_time'] = time();
        }
    }

    // Validate Inputs
    if (!filter_var($email, FILTER_VALIDATE_EMAIL)) {
        sendJson(false, 'Invalid email format.');
    }

    // Auth Check
    if ($email === $MOCK_USER['email'] && $password === $MOCK_USER['password']) {
        $_SESSION['user_id'] = 1;
        $_SESSION['email'] = $email;
        unset($_SESSION['login_attempts']);
        unset($_SESSION['first_attempt_time']);
        logEvent('portal.login_success', $email);
        sendJson(true, 'Login successful');
    } else {
        $_SESSION['login_attempts']++;
        logEvent('portal.login_failed', $email);
        sendJson(false, 'Invalid credentials.');
    }
}

// Redirect if already logged in (GET)
if (isset($_SESSION['user_id'])) {
    header("Location: index.php");
    exit;
}
?>
<!DOCTYPE html>
<html lang="en">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Cashbox Portal - Admin Login</title>
    
    <!-- Google Fonts -->
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    
    <!-- Tailwind CSS (v3.4 CDN) -->
    <script src="https://cdn.tailwindcss.com"></script>
    <script>
        tailwind.config = {
            darkMode: 'class',
            theme: {
                extend: {
                    colors: {
                        primary: '#FF9909',
                        secondary: '#2f3542',
                        dark: '#0f172a',
                        glass: {
                            100: 'rgba(255, 255, 255, 0.1)',
                            200: 'rgba(255, 255, 255, 0.2)',
                            300: 'rgba(255, 255, 255, 0.3)',
                            dark: 'rgba(15, 23, 42, 0.6)'
                        }
                    },
                    fontFamily: {
                        sans: ['Inter', 'sans-serif'],
                    }
                }
            }
        }
    </script>
    
    <!-- Ionicons -->
    <script type="module" src="https://unpkg.com/ionicons@5.5.2/dist/ionicons/ionicons.esm.js"></script>
    <script nomodule src="https://unpkg.com/ionicons@5.5.2/dist/ionicons/ionicons.js"></script>

    <!-- Theme Initialization -->
    <script>
        if (localStorage.theme === 'dark' || (!('theme' in localStorage) && window.matchMedia('(prefers-color-scheme: dark)').matches)) {
            document.documentElement.classList.add('dark');
        } else {
            document.documentElement.classList.remove('dark');
        }
    </script>
    
    <style>
        body {
            /* Light Mode */
            background-color: #f3f4f6;
            background-image: 
                radial-gradient(at 0% 0%, rgba(255, 153, 9, 0.08) 0px, transparent 50%),
                radial-gradient(at 100% 0%, rgba(255, 153, 9, 0.05) 0px, transparent 50%);
            background-attachment: fixed;
            transition: background-color 0.3s ease;
        }

        .dark body {
            background-color: #0f172a; /* Slate 900 */
            background-image: 
                radial-gradient(at 0% 0%, rgba(255, 153, 9, 0.1) 0px, transparent 50%),
                radial-gradient(at 100% 0%, rgba(255, 153, 9, 0.05) 0px, transparent 50%);
        }

        .glass-panel {
            background: rgba(255, 255, 255, 0.8);
            backdrop-filter: blur(12px);
            -webkit-backdrop-filter: blur(12px);
            border: 1px solid rgba(255, 255, 255, 0.6);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.05);
            transition: all 0.3s ease;
        }

        .dark .glass-panel {
            background: rgba(30, 41, 59, 0.7);
            border: 1px solid rgba(255, 255, 255, 0.05);
            box-shadow: 0 4px 6px -1px rgba(0, 0, 0, 0.2);
            color: #f1f5f9;
        }

        .spinner {
            border: 2px solid rgba(255,255,255,0.3);
            border-radius: 50%;
            border-top: 2px solid #fff;
            width: 1rem;
            height: 1rem;
            animation: spin 1s linear infinite;
            display: none;
        }
        .btn-loading .spinner { display: inline-block; }
        .btn-loading .btn-text { display: none; }
        @keyframes spin { 0% { transform: rotate(0deg); } 100% { transform: rotate(360deg); } }
    </style>
</head>
<body class="antialiased h-screen flex items-center justify-center p-4">

    <!-- Theme Toggle (Floating) -->
    <button onclick="toggleTheme()" class="absolute top-6 right-6 p-2 rounded-full bg-white/50 dark:bg-slate-800/50 hover:bg-white dark:hover:bg-slate-700 transition-all text-gray-500 dark:text-gray-400 hover:text-primary backdrop-blur-sm">
        <ion-icon name="moon-outline" class="text-xl hidden dark:block"></ion-icon>
        <ion-icon name="sunny-outline" class="text-xl block dark:hidden"></ion-icon>
    </button>

    <div class="w-full max-w-md">
        
        <!-- Login Card -->
        <div class="glass-panel rounded-2xl p-8 md:p-10 shadow-xl relative overflow-hidden">
            
            <!-- Header -->
            <div class="text-center mb-8">
                <div class="inline-flex items-center justify-center w-12 h-12 rounded-xl bg-orange-100 dark:bg-orange-900/20 text-primary mb-4">
                    <ion-icon name="cube" class="text-3xl"></ion-icon>
                </div>
                <h1 class="text-2xl font-bold text-gray-900 dark:text-white tracking-tight">Welcome Back</h1>
                <p class="text-gray-500 dark:text-gray-400 text-sm mt-2">Sign in to access your admin portal</p>
            </div>

            <!-- Error Alert -->
            <div id="errorAlert" class="hidden mb-6 p-4 rounded-xl bg-red-50 dark:bg-red-900/20 border border-red-100 dark:border-red-800/30 flex items-start gap-3">
                <ion-icon name="alert-circle" class="text-red-500 text-lg mt-0.5"></ion-icon>
                <span id="errorText" class="text-sm text-red-600 dark:text-red-400 font-medium">Invalid credentials</span>
            </div>

            <form id="loginForm" class="space-y-5">
                <input type="hidden" name="csrf_token" value="<?php echo $_SESSION['csrf_token']; ?>">
                
                <!-- Email -->
                <div>
                    <label for="email" class="block text-sm font-semibold text-gray-700 dark:text-gray-300 mb-1.5">Email address</label>
                    <div class="relative group">
                        <ion-icon name="mail-outline" class="absolute left-3.5 top-3 text-gray-400 group-focus-within:text-primary transition-colors"></ion-icon>
                        <input type="email" id="email" name="email" required autofocus autocomplete="email" 
                            class="w-full pl-10 pr-4 py-2.5 bg-gray-50 dark:bg-slate-800 border border-gray-200 dark:border-gray-700 rounded-xl text-sm text-gray-900 dark:text-white focus:ring-2 focus:ring-primary/20 focus:border-primary focus:outline-none transition-all placeholder-gray-400">
                    </div>
                </div>

                <!-- Password -->
                <div>
                    <div class="flex justify-between items-center mb-1.5">
                        <label for="password" class="block text-sm font-semibold text-gray-700 dark:text-gray-300">Password</label>
                        <a href="#" class="text-xs font-medium text-primary hover:text-orange-600 transition-colors">Forgot password?</a>
                    </div>
                    <div class="relative group">
                        <ion-icon name="lock-closed-outline" class="absolute left-3.5 top-3 text-gray-400 group-focus-within:text-primary transition-colors"></ion-icon>
                        <input type="password" id="password" name="password" required autocomplete="current-password" 
                            class="w-full pl-10 pr-10 py-2.5 bg-gray-50 dark:bg-slate-800 border border-gray-200 dark:border-gray-700 rounded-xl text-sm text-gray-900 dark:text-white focus:ring-2 focus:ring-primary/20 focus:border-primary focus:outline-none transition-all placeholder-gray-400">
                        <button type="button" id="togglePassword" class="absolute right-3.5 top-3 text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-colors">
                            <ion-icon name="eye-outline" class="text-lg"></ion-icon>
                        </button>
                    </div>
                </div>

                <!-- Remember -->
                <div class="flex items-center">
                    <input type="checkbox" id="remember" name="remember" class="w-4 h-4 text-primary border-gray-300 rounded focus:ring-primary bg-gray-50 dark:bg-slate-800 dark:border-gray-600">
                    <label for="remember" class="ml-2 text-sm text-gray-600 dark:text-gray-400">Remember me for 30 days</label>
                </div>

                <!-- Submit -->
                <button type="submit" id="submitBtn" class="w-full py-2.5 bg-primary hover:bg-orange-600 text-white font-semibold rounded-xl shadow-lg shadow-orange-500/20 active:scale-[0.98] transition-all flex items-center justify-center gap-2">
                    <span class="btn-text">Sign In</span>
                    <div class="spinner"></div>
                </button>
            </form>
        </div>

        <!-- Footer -->
        <div class="text-center mt-8 text-xs text-gray-400 dark:text-gray-500">
            &copy; <?php echo date('Y'); ?> Cashbox Portal. <br>System by Ahamed Shaliq.
        </div>
    </div>

    <!-- JS Logic -->
    <script>
        function initLoginPage() {
            const loginForm = document.getElementById('loginForm');
            if (!loginForm) return;

            const emailInput = document.getElementById('email');
            const passwordInput = document.getElementById('password');
            const submitBtn = document.getElementById('submitBtn');
            const errorAlert = document.getElementById('errorAlert');
            const errorText = document.getElementById('errorText');
            const togglePasswordBtn = document.getElementById('togglePassword');

            if (togglePasswordBtn && !togglePasswordBtn.dataset.initialized) {
                togglePasswordBtn.dataset.initialized = "true";
                togglePasswordBtn.addEventListener('click', () => {
                    const type = passwordInput.getAttribute('type') === 'password' ? 'text' : 'password';
                    passwordInput.setAttribute('type', type);
                    const icon = togglePasswordBtn.querySelector('ion-icon');
                    icon.setAttribute('name', type === 'password' ? 'eye-outline' : 'eye-off-outline');
                });
            }

            if (!loginForm.dataset.initialized) {
                loginForm.dataset.initialized = "true";
                loginForm.addEventListener('submit', async (e) => {
                    e.preventDefault();
                    hideError();

                    const email = emailInput.value.trim();
                    const password = passwordInput.value;
                    const remember = document.getElementById('remember').checked;

                    setLoading(true);

                    try {
                        const formData = new FormData();
                        formData.append('email', email);
                        formData.append('password', password);
                        formData.append('remember', remember ? '1' : '0');
                        const csrfToken = document.querySelector('input[name="csrf_token"]')?.value;
                        if (csrfToken) formData.append('csrf_token', csrfToken);

                        const response = await fetch('login.php', { method: 'POST', body: formData });
                        const text = await response.text();
                        let result;
                        try { result = JSON.parse(text); } catch (err) { throw new Error('Server returned invalid response'); }

                        if (result.success) {
                            // Full page redirect on login for session sync
                            window.location.href = 'index.php';
                        } else {
                            showError(result.message || 'Invalid credentials.');
                        }
                    } catch (error) {
                        showError('An unexpected error occurred. Please check your connection.');
                    } finally {
                        setLoading(false);
                    }
                });
            }

            function showError(message) {
                if (errorText) errorText.textContent = message;
                errorAlert?.classList.remove('hidden');
            }

            function hideError() {
                errorAlert?.classList.add('hidden');
            }

            function setLoading(isLoading) {
                if (!submitBtn) return;
                if (isLoading) {
                    submitBtn.classList.add('btn-loading');
                    submitBtn.disabled = true;
                    submitBtn.classList.add('opacity-75', 'cursor-not-allowed');
                } else {
                    submitBtn.classList.remove('btn-loading');
                    submitBtn.disabled = false;
                    submitBtn.classList.remove('opacity-75', 'cursor-not-allowed');
                }
            }
        }

        // Support both initial load and SPA transitions
        document.addEventListener('DOMContentLoaded', initLoginPage);

        if (!window.login_initialized) {
            document.addEventListener('pageLoaded', (e) => {
                if (e.detail.url.includes('login.php')) {
                    initLoginPage();
                }
            });
            window.login_initialized = true;
        }

        // Theme Toggle Logic
        window.toggleTheme = function() {
            if (document.documentElement.classList.contains('dark')) {
                document.documentElement.classList.remove('dark');
                localStorage.theme = 'light';
            } else {
                document.documentElement.classList.add('dark');
                localStorage.theme = 'dark';
            }
        }
    </script>
</body>
</html>
