<?php
// Reusable Dual-Pane Date Range Picker Component
// Usage: 
// include 'includes/components/date_range_picker.php';
//
// Parameters:
// - $pickerId (required): Unique ID (e.g., 'search_range')
// - $startParam (opt): URL param for start date (default: 'date_from')
// - $endParam (opt): URL param for end date (default: 'date_to')

$pickerId = $pickerId ?? 'range';
$startParam = $startParam ?? 'date_from';
$endParam = $endParam ?? 'date_to';

// Get Current Values
$startDateVal = $_GET[$startParam] ?? date('Y-m-d');
$endDateVal = $_GET[$endParam] ?? date('Y-m-d');

// Helper to format consistent with other pickers
$displayStart = date('d/m/Y', strtotime($startDateVal));
$displayEnd = date('d/m/Y', strtotime($endDateVal));
$displayRange = $displayStart . ' - ' . $displayEnd;
?>

<!-- Range Trigger -->
<div class="relative z-40">
    <button id="btn_<?php echo $pickerId; ?>" type="button" class="flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 rounded-lg transition-colors whitespace-nowrap cursor-pointer">
        <ion-icon name="calendar-outline" class="text-gray-400"></ion-icon>
        <span id="text_<?php echo $pickerId; ?>"><?php echo $displayRange; ?></span>
        <ion-icon name="chevron-down-outline" class="text-xs text-gray-400 ml-1 transition-transform duration-300"></ion-icon>
    </button>
    
    <!-- Dropdown (Dual Pane) -->
    <div id="dropdown_<?php echo $pickerId; ?>" class="absolute top-full right-0 mt-2 w-[660px] bg-white dark:bg-slate-800 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700 z-50 p-5 opacity-0 invisible transform -translate-y-2 transition-all duration-200 flex flex-col">
        
        <!-- Header Controls (Dual) -->
        <div class="flex items-center justify-between mb-4">
            <!-- Left Pane Control & Global Navigation -->
            <div class="flex items-center gap-2 px-2">
                <button type="button" onclick="window.rangePickers['<?php echo $pickerId; ?>'].changePage('left', -1)" class="w-8 h-8 flex items-center justify-center rounded-lg bg-gray-50 dark:bg-slate-700 text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-back-outline"></ion-icon>
                </button>
                <div id="header_label_left_<?php echo $pickerId; ?>" class="font-bold text-gray-800 dark:text-white text-base flex items-center gap-1">
                    <!-- Populated by JS -->
                </div>
                 <button type="button" onclick="window.rangePickers['<?php echo $pickerId; ?>'].changePage('left', 1)" class="w-8 h-8 flex items-center justify-center rounded-lg bg-gray-50 dark:bg-slate-700 text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-forward-outline"></ion-icon>
                </button>
            </div>

            <!-- Right Pane Label (Independent) -->
             <div class="flex items-center gap-2 px-2">
                <button type="button" onclick="window.rangePickers['<?php echo $pickerId; ?>'].changePage('right', -1)" class="w-8 h-8 flex items-center justify-center rounded-lg bg-gray-50 dark:bg-slate-700 text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-back-outline"></ion-icon>
                </button>
                <div id="header_label_right_<?php echo $pickerId; ?>" class="font-bold text-gray-800 dark:text-white text-base flex items-center gap-1">
                    <!-- Populated by JS -->
                </div>
                 <button type="button" onclick="window.rangePickers['<?php echo $pickerId; ?>'].changePage('right', 1)" class="w-8 h-8 flex items-center justify-center rounded-lg bg-gray-50 dark:bg-slate-700 text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-forward-outline"></ion-icon>
                </button>
            </div>
        </div>

        <!-- content Container -->
        <div id="calendar_container_<?php echo $pickerId; ?>" class="flex gap-8 mb-4 min-h-[250px]">
             <!-- Populated by JS -->
        </div>
        
        <!-- Footer Actions -->
        <div class="flex flex-col gap-3 border-t border-gray-100 dark:border-gray-700 pt-4">
            <!-- Jump To Today -->
             <button type="button" onclick="window.rangePickers['<?php echo $pickerId; ?>'].goToToday()" class="w-full py-2 bg-gray-100 dark:bg-slate-700 text-gray-600 dark:text-gray-300 font-semibold rounded-xl text-sm hover:bg-gray-200 dark:hover:bg-slate-600 transition-colors">
                Jump to Today
            </button>
        </div>
    </div>
</div>

<script>
if (typeof CashboxRangePicker === 'undefined') {
    window.CashboxRangePicker = class CashboxRangePicker {
        constructor(pickerId, startParam, endParam, defaultStart, defaultEnd) {
            this.pickerId = pickerId;
            this.startParam = startParam;
            this.endParam = endParam;
            
            this.startDate = this.parseDate(defaultStart);
            this.endDate = this.parseDate(defaultEnd);
            
            // Decoupled Views
            this.leftViewDate = new Date(this.startDate); 
            this.leftViewDate.setDate(1);
            
            this.rightViewDate = new Date(this.startDate);
            this.rightViewDate.setMonth(this.rightViewDate.getMonth() + 1);
            this.rightViewDate.setDate(1);

            this.tempStart = new Date(this.startDate);
            this.tempEnd = new Date(this.endDate);
            
            // Interaction State
            this.currentView = 'days'; 
            this.leftMode = 'days';
            this.rightMode = 'days';

            if (!window.rangePickers) window.rangePickers = {};
            window.rangePickers[pickerId] = this;

            this.init();
        }

        parseDate(str) { 
            if(!str) return new Date();
            const p = str.split('-');
            if(p.length===3) return new Date(p[0], p[1]-1, p[2]);
            return new Date(str);
        }
        formatDate(d) { return `${d.getFullYear()}-${(d.getMonth()+1).toString().padStart(2,'0')}-${d.getDate().toString().padStart(2,'0')}`; }
        formatDisplay(d) { return `${d.getDate().toString().padStart(2,'0')}/${(d.getMonth()+1).toString().padStart(2,'0')}/${d.getFullYear()}`; }

        init() {
            const btn = document.getElementById(`btn_${this.pickerId}`);
            if(btn) {
                btn.onclick = (e) => { e.stopPropagation(); this.toggleDropdown(); };
            }
            document.addEventListener('click', (e) => {
                const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
                if (btn && dropdown && !btn.contains(e.target) && !dropdown.contains(e.target)) {
                    this.closeDropdown();
                }
            });
        }

        toggleDropdown() {
            const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
            if (!dropdown) return;
            const isHidden = dropdown.classList.contains('invisible');
            
            if (isHidden) {
                // Reset Logic: 
                this.leftViewDate = new Date(this.tempStart);
                this.leftViewDate.setDate(1);
                
                this.rightViewDate = new Date(this.tempEnd);
                this.rightViewDate.setDate(1);
                
                if (this.rightViewDate.getFullYear() === this.leftViewDate.getFullYear() && this.rightViewDate.getMonth() === this.leftViewDate.getMonth()) {
                     this.rightViewDate.setMonth(this.rightViewDate.getMonth() + 1);
                }

                this.leftMode = 'days';
                this.rightMode = 'days';
                
                this.render();
                dropdown.classList.remove('invisible', 'opacity-0', '-translate-y-2');
            } else {
                this.closeDropdown();
            }
        }

        closeDropdown() {
             const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
             if(dropdown) dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
        }
        
        changePage(side, offset) {
            const viewDate = (side === 'left') ? this.leftViewDate : this.rightViewDate;
            const mode = (side === 'left') ? this.leftMode : this.rightMode;

            if (mode === 'days') {
                viewDate.setMonth(viewDate.getMonth() + offset);
            } else if (mode === 'years') {
                viewDate.setFullYear(viewDate.getFullYear() + (offset * 12)); 
            } else {
                 viewDate.setFullYear(viewDate.getFullYear() + offset);
            }
            this.render();
        }
        
        setMode(side, mode) {
            if(side === 'left') this.leftMode = mode;
            else this.rightMode = mode;
            this.render();
        }

        render() {
            const container = document.getElementById(`calendar_container_${this.pickerId}`);
            const leftLabel = document.getElementById(`header_label_left_${this.pickerId}`);
            const rightLabel = document.getElementById(`header_label_right_${this.pickerId}`);
            if(!container) return;

            container.innerHTML = '';
            
            // Build Helper
            const buildSide = (side) => {
                const pane = document.createElement('div'); pane.className = 'flex-1 relative';
                
                const viewDate = (side === 'left') ? this.leftViewDate : this.rightViewDate;
                const mode = (side === 'left') ? this.leftMode : this.rightMode;
                const mInfo = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];
                const year = viewDate.getFullYear();

                // 1. UPDATE HEADER
                const label = (side === 'left') ? leftLabel : rightLabel;
                label.innerHTML = `
                    <span class="hover:text-[#E65100] cursor-pointer transition-colors" onclick="event.stopPropagation(); window.rangePickers['${this.pickerId}'].setMode('${side}', 'months')">${mInfo[viewDate.getMonth()]}</span>
                    <span class="hover:text-[#E65100] cursor-pointer transition-colors" onclick="event.stopPropagation(); window.rangePickers['${this.pickerId}'].setMode('${side}', 'years')">${year}</span>
                    <ion-icon name="caret-down" class="text-xs text-gray-400 ml-1"></ion-icon>
                `;

                // 2. RENDER CONTENT
                if (mode === 'days') {
                    // Header for Days
                    const dHeader = document.createElement('div'); dHeader.className="grid grid-cols-7 text-center mb-2";
                    ['S','M','T','W','T','F','S'].forEach(t => {
                       const s = document.createElement('div'); s.className="text-xs font-bold text-gray-400 dark:text-gray-500 py-1"; s.textContent=t; dHeader.appendChild(s);
                    });
                    pane.appendChild(dHeader);
                    
                    const grid = document.createElement('div'); grid.className="grid grid-cols-7 gap-1 text-center text-sm";
                    this.renderDays(viewDate, grid);
                    pane.appendChild(grid);
                    
                } else if (mode === 'months') {
                    label.textContent = `${year}`;
                    
                    const grid = document.createElement('div');
                    grid.className = "grid grid-cols-3 gap-2 w-full pt-4"; // Compact grid
                    
                    mInfo.forEach((m, idx) => {
                        const btn = document.createElement('button');
                        btn.type="button";
                        const isCurrent = (idx === viewDate.getMonth());
                        let cls = "h-10 rounded-lg text-sm font-bold transition-all ";
                        if(isCurrent) cls += "bg-orange-50 text-[#E65100] border border-orange-200";
                        else cls += "text-gray-600 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-slate-700";
                        
                        btn.className = cls;
                        btn.textContent = m.substring(0,3); // Short name
                        btn.onclick = (e) => {
                            e.stopPropagation();
                            viewDate.setMonth(idx);
                            this.setMode(side, 'days');
                        };
                        grid.appendChild(btn);
                    });
                    pane.appendChild(grid);

                } else if (mode === 'years') {
                    const startYear = year - 6;
                    const endYear = startYear + 11;
                    label.textContent = `${startYear} - ${endYear}`;
                    
                    const grid = document.createElement('div');
                    grid.className = "grid grid-cols-3 gap-2 w-full pt-4";
                    
                    for(let i=0; i<12; i++) {
                        const y = startYear + i;
                        const btn = document.createElement('button');
                        btn.type="button";
                        const isCurrent = (y === year);
                        let cls = "h-10 rounded-lg text-sm font-bold transition-all ";
                        if(isCurrent) cls += "bg-orange-50 text-[#E65100] border border-orange-200";
                        else cls += "text-gray-600 dark:text-gray-300 hover:bg-gray-50 dark:hover:bg-slate-700";
                        
                        btn.className = cls;
                        btn.textContent = y;
                        btn.onclick = (e) => {
                            e.stopPropagation();
                            viewDate.setFullYear(y);
                            this.setMode(side, 'days'); 
                        };
                        grid.appendChild(btn);
                    }
                    pane.appendChild(grid);
                }

                return pane;
            };

            container.appendChild(buildSide('left'));
            
            const divider = document.createElement('div'); divider.className = 'w-px bg-gray-100 dark:bg-gray-700 mx-4';
            container.appendChild(divider);
            
            container.appendChild(buildSide('right'));
        }

        renderDays(baseDate, grid) {
            const year = baseDate.getFullYear();
            const month = baseDate.getMonth();
            const firstDay = new Date(year, month, 1).getDay();
            const daysInMonth = new Date(year, month+1, 0).getDate();
            
            const today = new Date();
            today.setHours(0,0,0,0);

            // Filler
            for(let i=0; i<firstDay; i++) {
                grid.appendChild(document.createElement('div'));
            }

            // Days
            for(let i=1; i<=daysInMonth; i++) {
                const d = new Date(year, month, i);
                const btn = document.createElement('button');
                btn.type = 'button';
                btn.textContent = i;
                
                const tStart = this.tempStart.getTime(); 
                const tEnd = this.tempEnd.getTime();     
                const curr = d.getTime();             
                
                if (d > today) {
                    btn.className = "h-9 w-full flex items-center justify-center text-sm text-gray-300 dark:text-gray-600 cursor-not-allowed";
                    btn.onclick = (e) => e.stopPropagation(); 
                } else {
                    let cls = "h-9 w-full flex items-center justify-center text-sm font-medium transition-colors rounded-lg cursor-pointer ";
                    
                    if (curr === tStart || curr === tEnd) {
                        cls += "bg-[#E65100] text-white shadow-md shadow-orange-500/30 z-10 relative ";
                    } else if (curr > tStart && curr < tEnd) {
                        cls += "bg-orange-50 dark:bg-orange-900/20 text-orange-700 dark:text-orange-300 rounded-none ";
                    } else {
                        cls += "text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-slate-700 ";
                    }

                    btn.className = cls;
                    btn.onclick = (e) => { e.stopPropagation(); this.handleDateClick(d); };
                }
                grid.appendChild(btn);
            }
        }

        handleDateClick(date) {
            const dTime = date.getTime();
            const sTime = this.tempStart.getTime();
            const eTime = this.tempEnd.getTime();

            if (sTime !== eTime) {
                // Was Range -> Reset to Single
                this.tempStart = new Date(date);
                this.tempEnd = new Date(date);
            } else {
                // Was Single -> Complete Range
                if (dTime < sTime) {
                    this.tempStart = new Date(date); // Swap if clicked before
                    // Keep end same? No, logic above said "Was Single -> Set End"
                    // If dTime < sTime, dTime is new Start, sTime is End?
                    // Previous logic: this.tempEnd = new Date(sTime);
                    // Yes.
                    this.tempEnd = new Date(sTime);
                } else {
                    this.tempEnd = new Date(date);
                }
            }
            this.render();
            // Auto Update URL when range is complete OR single?
            // Let's update on every valid click/state change to keep UI in sync
            this.updateUrlState();
        }

        goToToday() {
            const now = new Date();
            now.setHours(0,0,0,0);
            this.tempStart = new Date(now);
            this.tempEnd = new Date(now);
            
            this.leftViewDate = new Date(now);
            this.leftViewDate.setDate(1);
            
            this.rightViewDate = new Date(now);
            this.rightViewDate.setMonth(now.getMonth() + 1);
            this.rightViewDate.setDate(1);

            this.leftMode = 'days';
            this.rightMode = 'days';
            this.render();
            this.updateUrlState();
        }

        async updateUrlState() {
            // Update Internal State & URL
            this.startDate = new Date(this.tempStart);
            this.endDate = new Date(this.tempEnd);

            const label = `${this.formatDisplay(this.startDate)} - ${this.formatDisplay(this.endDate)}`;
            document.getElementById(`text_${this.pickerId}`).textContent = label;

            const s = this.formatDate(this.startDate);
            const e = this.formatDate(this.endDate);
            
            const url = new URL(window.location.href);
            url.searchParams.set(this.startParam, s);
            url.searchParams.set(this.endParam, e);
            window.history.pushState({}, '', url);

            if (typeof window.handleDateChange === 'function') {
                await window.handleDateChange(url.toString(), this.pickerId, s, e);
            }
        }
    }
}

new CashboxRangePicker('<?php echo $pickerId; ?>', '<?php echo $startParam; ?>', '<?php echo $endParam; ?>', '<?php echo $startDateVal; ?>', '<?php echo $endDateVal; ?>');
</script>
