<?php
// Reusable Date Picker Component
// Usage: 
// $pickerId = 'start'; $urlParam = 'start_date'; $btnClass = '...'; include 'includes/components/date_picker.php';
//
// Parameters:
// - $pickerId (required): Unique ID for this instance (e.g., 'live', 'from', 'to')
// - $urlParam (optional): URL query parameter to update (default: 'date')
// - $defaultLabel (optional): Default text if no date selected
// - $btnClass (optional): Tailwind classes for the button
// - $showLabel (optional): Boolean to show/hide the text label (default: true)

$pickerId = $pickerId ?? 'live';
$urlParam = $urlParam ?? 'date';
$btnClass = $btnClass ?? 'flex items-center gap-2 px-3 py-2 text-sm font-medium text-gray-600 dark:text-gray-300 hover:text-gray-900 dark:hover:text-white hover:bg-white dark:hover:bg-slate-700 rounded-lg transition-colors whitespace-nowrap cursor-pointer';
$defaultDate = $_GET[$urlParam] ?? date('Y-m-d');
$displayDate = date('d/m/Y', strtotime($defaultDate));
$showApplyButton = $showApplyButton ?? true;
$autoApply = $autoApply ?? false;
?>

<!-- Date Filter Button -->
<div class="relative z-40">
    <button id="btn_<?php echo $pickerId; ?>" type="button" class="<?php echo $btnClass; ?>">
        <ion-icon name="calendar-outline" class="text-gray-400"></ion-icon>
        <span id="text_<?php echo $pickerId; ?>"><?php echo (time() - strtotime($defaultDate) < 86400) ? 'Today : '.$displayDate : $displayDate; ?></span>
        <ion-icon name="chevron-down-outline" class="text-xs text-gray-400 ml-1 transition-transform duration-300"></ion-icon>
    </button>
    
    <!-- Date Dropdown -->
    <div id="dropdown_<?php echo $pickerId; ?>" class="absolute top-full left-0 mt-2 w-[300px] bg-white dark:bg-slate-800 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700 z-50 p-4 opacity-0 invisible transform -translate-y-2 transition-all duration-200">
        
        <!-- Header -->
        <div class="flex items-center justify-between mb-4">
            <h3 id="month_label_<?php echo $pickerId; ?>" class="font-bold text-gray-800 dark:text-white flex items-center gap-2">Loading...</h3>
            <div class="flex gap-1">
                <button type="button" onclick="window.pickers['<?php echo $pickerId; ?>'].changeMonth(-1)" class="w-8 h-8 rounded-lg bg-gray-50 dark:bg-slate-700 flex items-center justify-center text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-back-outline"></ion-icon>
                </button>
                <button type="button" onclick="window.pickers['<?php echo $pickerId; ?>'].changeMonth(1)" class="w-8 h-8 rounded-lg bg-gray-50 dark:bg-slate-700 flex items-center justify-center text-gray-600 dark:text-gray-300 hover:bg-gray-100 dark:hover:bg-slate-600 transition-colors">
                    <ion-icon name="chevron-forward-outline"></ion-icon>
                </button>
            </div>
        </div>

        <!-- Days Header -->
        <div id="days_header_<?php echo $pickerId; ?>" class="grid grid-cols-7 text-center mb-2">
                <?php foreach(['S','M','T','W','T','F','S'] as $day): ?>
                <div class="text-xs font-bold text-gray-400 dark:text-gray-500 py-1"><?php echo $day; ?></div>
                <?php endforeach; ?>
        </div>

        <!-- Calendar Grid -->
        <div id="grid_<?php echo $pickerId; ?>" class="grid grid-cols-7 gap-1 text-center mb-4">
            <!-- Populated by JS -->
        </div>
        
        <!-- Today Shortcut -->
            <button type="button" onclick="window.pickers['<?php echo $pickerId; ?>'].goToToday()" class="w-full mb-2 py-1.5 bg-gray-100 dark:bg-slate-700 text-gray-600 dark:text-gray-300 font-semibold rounded-lg text-xs hover:bg-gray-200 dark:hover:bg-slate-600 transition-colors">
            Jump to Today
        </button>

        <!-- Apply Button -->
        <?php if($showApplyButton ?? true): ?>
        <button type="button" onclick="window.pickers['<?php echo $pickerId; ?>'].applySelection()" class="w-full py-2.5 bg-[#E65100] hover:bg-[#F57C00] text-white font-bold rounded-xl shadow-lg shadow-orange-500/20 transition-all transform hover:-translate-y-0.5 text-sm">
            Apply Now
        </button>
        <?php endif; ?>
    </div>
</div>

<script>
    // Define class only if not already defined (Singleton Pattern for Class Definition)
    if (typeof CashboxDatePicker === 'undefined') {
        window.CashboxDatePicker = class CashboxDatePicker {
            constructor(pickerId, urlParam, defaultDate, autoApply = false) {
                this.pickerId = pickerId;
                this.urlParam = urlParam;
                this.defaultDate = defaultDate;
                this.autoApply = autoApply;
                this.calendarView = 'days';
                
                // Register instance
                if (!window.pickers) window.pickers = {};
                window.pickers[pickerId] = this;

                // Parse Initial Date (Simple YYYY-MM-DD parser)
                this.selectedDate = this.parseDate(defaultDate);
                this.calendarDate = new Date(this.selectedDate);

                // Initialize
                this.init();
            }

            parseDate(dateStr) {
                if (!dateStr) return new Date();
                const parts = dateStr.split('-');
                if(parts.length === 3) return new Date(parts[0], parts[1]-1, parts[2]);
                return new Date();
            }

            init() {
                // Attach Toggle Event using ID
                const btn = document.getElementById(`btn_${this.pickerId}`);
                if(btn) {
                    btn.onclick = (e) => {
                        e.stopPropagation();
                        this.toggleDropdown();
                    };
                }

                // Global Click to Close
                document.addEventListener('click', (e) => {
                    const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
                    // Note: button element might be different if re-rendered, so re-fetch
                    const currentBtn = document.getElementById(`btn_${this.pickerId}`);
                    if (currentBtn && dropdown && !currentBtn.contains(e.target) && !dropdown.contains(e.target)) {
                        this.closeDropdown();
                    }
                });

                // Initial Render
                this.renderLabel();
                this.renderCalendar();
            }

            toggleDropdown() {
                const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
                const btn = document.getElementById(`btn_${this.pickerId}`);
                const icon = btn ? btn.querySelector('ion-icon[name="chevron-down-outline"]') : null;
                
                if (!dropdown) return;

                const isHidden = dropdown.classList.contains('invisible');

                // Close all other dropdowns
                document.querySelectorAll('[id^="dropdown_"]').forEach(el => {
                    if (el.id !== `dropdown_${this.pickerId}`) {
                        el.classList.add('invisible', 'opacity-0', '-translate-y-2');
                    }
                });

                if (isHidden) {
                    dropdown.classList.remove('invisible', 'opacity-0', '-translate-y-2');
                    if(icon) icon.classList.add('rotate-180');
                    this.renderCalendar(); // Re-render calendar when opening
                } else {
                    dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
                    if(icon) icon.classList.remove('rotate-180');
                }
            }

            closeDropdown() {
                const dropdown = document.getElementById(`dropdown_${this.pickerId}`);
                const btn = document.getElementById(`btn_${this.pickerId}`);
                const icon = btn ? btn.querySelector('ion-icon[name="chevron-down-outline"]') : null;
                
                if(dropdown) dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
                if(icon) icon.classList.remove('rotate-180');
            }

            renderLabel() {
                const label = document.getElementById(`text_${this.pickerId}`);
                if(!label) return;

                const day = this.selectedDate.getDate().toString().padStart(2, '0');
                const month = (this.selectedDate.getMonth() + 1).toString().padStart(2, '0');
                const year = this.selectedDate.getFullYear();
                const displayFormatted = `${day}/${month}/${year}`;
                
                const today = new Date();
                let value = displayFormatted;
                if (this.selectedDate.toDateString() === today.toDateString()) {
                    value = `Today : ${displayFormatted}`;
                }
                label.textContent = value;
            }

            renderCalendar() {
                const grid = document.getElementById(`grid_${this.pickerId}`);
                const label = document.getElementById(`month_label_${this.pickerId}`);
                const daysHeader = document.getElementById(`days_header_${this.pickerId}`);
                
                if(!grid || !label) return;
                grid.innerHTML = '';

                const year = this.calendarDate.getFullYear();
                const month = this.calendarDate.getMonth();
                const monthNames = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

                // Build Header
                const mClass = this.calendarView === 'months' ? "text-orange-600 font-extrabold" : "hover:text-orange-600 cursor-pointer transition-colors";
                const yClass = this.calendarView === 'years' ? "text-orange-600 font-extrabold" : "hover:text-orange-600 cursor-pointer transition-colors";
                
                // Using global access for click handling inside HTML string
                label.innerHTML = `
                   <span class="${mClass}" onclick="event.stopPropagation(); window.pickers['${this.pickerId}'].setCalView('months')">${monthNames[month]} <ion-icon name="chevron-down-outline" class="text-[10px] opacity-70"></ion-icon></span>
                   <span class="mx-1 opacity-30 text-gray-400">|</span>
                   <span class="${yClass}" onclick="event.stopPropagation(); window.pickers['${this.pickerId}'].setCalView('years')">${year} <ion-icon name="chevron-down-outline" class="text-[10px] opacity-70"></ion-icon></span>
                `;

                // Render Views
                if (this.calendarView === 'days') {
                     if(daysHeader) daysHeader.classList.remove('hidden');
                     
                     grid.className = "grid grid-cols-7 gap-1 text-center mb-4";
                     
                     const firstDay = new Date(year, month, 1).getDay(); 
                     const daysInMonth = new Date(year, month + 1, 0).getDate();
                     const daysInPrevMonth = new Date(year, month, 0).getDate();
                     const today = new Date(); today.setHours(0,0,0,0);

                     // Prev Month
                     for (let i = 0; i < firstDay; i++) {
                         const btn = this.createBtn(daysInPrevMonth - firstDay + 1 + i, 'text-gray-300 dark:text-gray-600', () => this.changeMonth(-1));
                         grid.appendChild(btn);
                     }
                     // Current Month
                     for (let i = 1; i <= daysInMonth; i++) {
                         const date = new Date(year, month, i);
                         let cls = "text-gray-700 dark:text-gray-200 hover:bg-gray-100 dark:hover:bg-slate-700";
                         let disabled = false;
                         
                         if (date > today) {
                             cls = "text-gray-300 dark:text-gray-600 opacity-50 cursor-not-allowed";
                             disabled = true;
                         } else if (this.selectedDate.toDateString() === date.toDateString()) {
                             cls = "bg-[#E65100] text-white shadow-md shadow-orange-500/30";
                         } else if (today.toDateString() === date.toDateString()) {
                             cls += " border border-orange-500 text-orange-600";
                         }
                         
                         grid.appendChild(this.createBtn(i, cls, () => this.selectDate(i), disabled));
                     }
                } 
                else if (this.calendarView === 'months') {
                    if(daysHeader) daysHeader.classList.add('hidden');
                    grid.className = "grid grid-cols-3 gap-2 text-center mb-4";
                    monthNames.forEach((m, idx) => {
                        const isSel = (idx === month);
                        const cls = `py-2 rounded-lg text-sm ${isSel ? 'bg-orange-50 text-orange-600 border border-orange-200' : 'hover:bg-gray-50 dark:hover:bg-slate-700 text-gray-700 dark:text-gray-300'}`;
                        const btn = document.createElement('button'); btn.type="button"; btn.className=cls; btn.textContent=m.substring(0,3);
                        btn.onclick=(e)=>{e.stopPropagation(); this.calendarDate.setMonth(idx); this.calendarView='days'; this.renderCalendar();};
                        grid.appendChild(btn);
                    });
                }
                else if (this.calendarView === 'years') {
                    if(daysHeader) daysHeader.classList.add('hidden');
                    grid.className = "grid grid-cols-3 gap-2 text-center mb-4";
                    const startYear = year - 5;
                    const currentYear = new Date().getFullYear();
                    for(let i=0; i<12; i++){
                        const y = startYear+i;
                        if(y>currentYear) continue;
                        const isSel = (y===year);
                        const cls = `py-2 rounded-lg text-sm ${isSel ? 'bg-orange-50 text-orange-600 border border-orange-200' : 'hover:bg-gray-50 dark:hover:bg-slate-700 text-gray-700 dark:text-gray-300'}`;
                         const btn = document.createElement('button'); btn.type="button"; btn.className=cls; btn.textContent=y;
                        btn.onclick=(e)=>{e.stopPropagation(); this.calendarDate.setFullYear(y); this.calendarView='months'; this.renderCalendar();};
                        grid.appendChild(btn);
                    }
                }
            }

            createBtn(text, classes, onClick, disabled=false) {
                const btn = document.createElement('button');
                btn.type = "button";
                btn.className = `h-8 w-8 rounded-lg flex items-center justify-center text-sm font-medium transition-colors ${classes}`;
                btn.textContent = text;
                if(disabled) btn.disabled = true;
                else btn.onclick = (e) => { e.stopPropagation(); onClick(); };
                return btn;
            }

            setCalView(view) { this.calendarView = view; this.renderCalendar(); }
            changeMonth(offset) { this.calendarDate.setMonth(this.calendarDate.getMonth() + offset); this.renderCalendar(); }
            changeYear(offset) { this.calendarDate.setFullYear(this.calendarDate.getFullYear() + offset); this.renderCalendar(); }
            
            selectDate(day) {
                // Only select, do not apply or close
                this.selectedDate = new Date(this.calendarDate.getFullYear(), this.calendarDate.getMonth(), day);
                // We re-render to show the highlight state
                this.renderCalendar();
                // Optionally update label immediately or wait? 
                // User asked "choosing a date... then apply". 
                // Usually label updates on selection.
                // But let's keep label showing "Selected but not applied"? 
                // Actually, standard behavior is label updates to reflect 'pending' selection inside the picker context, 
                // but the MAIN label (outside) only updates on apply?
                // The current renderLabel targets `text_${id}` which IS the button label.
                // If we update it, the user sees visual confirmation.
                // Let's UPDATE it so they know what they picked.
                this.renderLabel();

                if (this.autoApply) {
                    this.applySelection();
                }
            }

            async applySelection() {
                this.closeDropdown();
                
                const day = this.selectedDate.getDate().toString().padStart(2, '0');
                const month = (this.selectedDate.getMonth() + 1).toString().padStart(2, '0');
                const year = this.selectedDate.getFullYear();
                const formatted = `${year}-${month}-${day}`;
                
                // Update URL (history)
                const currentUrl = new URL(window.location.href);
                currentUrl.searchParams.set(this.urlParam, formatted);
                window.history.pushState({path: currentUrl.toString()}, '', currentUrl.toString());

                // Trigger Custom Handler
                if (typeof window.handleDateChange === 'function') {
                    await window.handleDateChange(currentUrl.toString(), this.pickerId, formatted);
                } else {
                    window.location.reload();
                }
            }
            
            
            goToToday() {
                const now = new Date();
                this.selectedDate = new Date(now);
                this.calendarDate = new Date(now);
                this.calendarView = 'days';
                this.renderLabel();
                this.renderCalendar();
                // Removed this.applySelection() to align with manual apply workflow
            }
        };
    }

    // Initialize this instance
    new CashboxDatePicker('<?php echo $pickerId; ?>', '<?php echo $urlParam; ?>', '<?php echo $defaultDate; ?>', <?php echo $autoApply ? 'true' : 'false'; ?>);
</script>
