<?php
include 'includes/session.php';
$pageTitle = "Detailed Audit Tracker";
$currentPage = "audit_tracker";

// Styles
$inputClass = "w-full bg-white dark:bg-slate-800 border border-gray-200 dark:border-gray-700 rounded-xl px-4 py-2.5 outline-none focus:ring-2 focus:ring-orange-500/20 focus:border-orange-500 transition-all text-sm shadow-sm font-medium text-gray-600 dark:text-gray-300 placeholder-gray-400";
$badgeClass = "inline-flex items-center px-2.5 py-0.5 rounded-full text-xs font-medium border";

require_once 'includes/header.php';
require_once 'includes/components/custom_select_renderer.php';

// Mock Data for Audit Events with Diffs
$events = [
    [
        'id' => 'evt_9921',
        'time' => '2026-01-16 15:45:22',
        'actor' => 'Ali K.',
        'actor_role' => 'Manager',
        'target' => 'Sarah J.',
        'target_role' => 'Staff',
        'system' => 'Portal',
        'action' => 'User Update',
        'entity' => 'User',
        'summary' => 'Promoted to Supervisor',
        'reason' => 'Quarterly promotion cycle',
        'changes' => [
            ['field' => 'Role', 'old' => 'Cashier', 'new' => 'Supervisor'],
            ['field' => 'Wage', 'old' => '£11.50', 'new' => '£13.00']
        ]
    ],
    [
        'id' => 'evt_9920',
        'time' => '2026-01-16 14:12:10',
        'actor' => 'Sarah J.',
        'actor_role' => 'Supervisor',
        'target' => 'John D.',
        'target_role' => 'Cashier',
        'system' => 'Cashbox Till',
        'action' => 'Store Access',
        'entity' => 'Permissions',
        'summary' => 'Modified store access lists',
        'reason' => 'Covering shifts in Newport',
        'changes' => [
            ['field' => 'Store Access', 'old' => ['Cardiff', 'Bristol'], 'new' => ['Cardiff', 'Bristol', 'Newport'], 'type' => 'array']
        ]
    ],
    [
        'id' => 'evt_9919',
        'time' => '2026-01-16 11:30:05',
        'actor' => 'System',
        'actor_role' => 'Bot',
        'target' => 'N/A',
        'target_role' => 'N/A',
        'system' => 'Portal',
        'action' => 'Config Change',
        'entity' => 'Global Settings',
        'summary' => 'Updated VAT Rate',
        'reason' => 'Government policy update',
        'changes' => [
            ['field' => 'Standard VAT', 'old' => '20.0%', 'new' => '20.0% (Locked)']
        ]
    ],
    [
        'id' => 'evt_9918',
        'time' => '2026-01-15 09:15:00',
        'actor' => 'Ali K.',
        'actor_role' => 'Manager',
        'target' => 'Mike R.',
        'target_role' => 'Part-time',
        'system' => 'Cashbox Admin',
        'action' => 'Disable User',
        'entity' => 'User',
        'summary' => 'Account disabled indefinitely',
        'reason' => 'Suspicious activity detected on Till #2',
        'changes' => [
            ['field' => 'Status', 'old' => 'Active', 'new' => 'Disabled'],
            ['field' => 'Login Access', 'old' => 'Enabled', 'new' => 'Revoked']
        ]
    ]
];

function renderArrayDiff($old, $new) {
    if (!is_array($old)) $old = [];
    if (!is_array($new)) $new = [];
    
    $added = array_diff($new, $old);
    $removed = array_diff($old, $new);
    $kept = array_intersect($old, $new);
    
    $html = '<div class="flex flex-wrap gap-1.5">';
    foreach ($removed as $item) {
        $html .= '<span class="inline-flex items-center px-2 py-0.5 rounded bg-red-100 text-red-700 dark:bg-red-900/30 dark:text-red-400 text-xs font-mono line-through" title="Removed">'.$item.'</span>';
    }
    foreach ($added as $item) {
        $html .= '<span class="inline-flex items-center px-2 py-0.5 rounded bg-green-100 text-green-700 dark:bg-green-900/30 dark:text-green-400 text-xs font-mono font-bold" title="Added">'.$item.'</span>';
    }
    foreach ($kept as $item) {
        $html .= '<span class="inline-flex items-center px-2 py-0.5 rounded bg-gray-100 text-gray-600 dark:bg-slate-700 dark:text-gray-400 text-xs font-mono">'.$item.'</span>';
    }
    $html .= '</div>';
    return $html;
}

?>

<!-- Sidebar -->
<?php include 'includes/sidebar.php'; ?>

<main class="flex-1 flex flex-col relative z-0 overflow-y-auto w-full">
    
    <!-- Topbar -->
    <?php include 'includes/topbar.php'; ?>

    <!-- Page Content -->
    <div class="flex-1 p-6 md:p-8 pb-20 overflow-y-auto custom-scrollbar">
        <div class="max-w-[1600px] mx-auto space-y-8">

            <!-- Header & Advanced Filters -->
            <div class="glass-panel p-8 rounded-[2.5rem] relative overflow-hidden group border-none shadow-sm">
                 <div class="absolute top-0 right-0 p-12 opacity-[0.03] pointer-events-none group-hover:opacity-[0.07] transition-opacity">
                    <ion-icon name="git-compare" class="text-[12rem]"></ion-icon>
                </div>
                <div class="relative z-10">
                    <div class="flex flex-col lg:flex-row lg:items-center justify-between gap-8 mb-10">
                        <div class="flex items-center gap-5">
                            <div class="w-14 h-14 rounded-2xl bg-primary/10 flex items-center justify-center text-primary shadow-sm border border-primary/10">
                                <ion-icon name="git-compare-outline" class="text-3xl"></ion-icon>
                            </div>
                            <div>
                                <h1 class="text-3xl font-bold text-gray-800 dark:text-white tracking-tight">Audit Tracker</h1>
                                <p class="text-gray-500 dark:text-gray-400 text-sm mt-1">Granular tracking of data changes with diff comparisons.</p>
                            </div>
                        </div>
                        <div class="flex items-center gap-3">
                             <button class="px-5 py-3 bg-white dark:bg-slate-800 border border-gray-200 dark:border-gray-700 text-gray-700 dark:text-gray-200 rounded-2xl text-sm font-bold hover:bg-gray-50 dark:hover:bg-slate-700 hover:text-primary transition-all shadow-sm flex items-center gap-2 group border-none">
                                <ion-icon name="download-outline" class="text-lg group-hover:scale-110 transition-transform"></ion-icon> 
                                <span>Export Detailed Report</span>
                            </button>
                        </div>
                    </div>

                    <div class="space-y-6">
                        <div class="grid grid-cols-1 md:grid-cols-2 lg:grid-cols-4 gap-6">
                            <!-- Search -->
                            <div class="col-span-1 md:col-span-2">
                                <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Search Context</label>
                                <div class="relative group h-[46px]">
                                    <div class="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                                        <ion-icon name="search-outline" class="text-gray-400 text-lg group-focus-within:text-primary transition-colors"></ion-icon>
                                    </div>
                                    <input type="text" class="w-full h-full bg-white/50 dark:bg-slate-900/50 border border-gray-100 dark:border-gray-800 rounded-2xl px-5 pl-12 py-3 outline-none focus:ring-4 focus:ring-orange-500/10 focus:border-primary transition-all text-sm font-medium" placeholder="Search by Event ID, Actor, or Target Name...">
                                </div>
                            </div>
                            
                            <!-- System -->
                            <div>
                                <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Source System</label>
                                 <?php echo renderCustomSelect('All Systems', 'system_id', ['all' => 'All Systems', 'portal' => 'Web Portal', 'till' => 'Cashbox Till', 'admin' => 'Admin API'], 'all'); ?>
                            </div>

                            <!-- Action -->
                            <div>
                                 <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Action Type</label>
                                 <?php echo renderCustomSelect('All Actions', 'action_id', ['all' => 'All Actions', 'update' => 'Updates', 'delete' => 'Deletions', 'config' => 'Config Changes'], 'all'); ?>
                            </div>
                        </div>
                        
                        <div class="grid grid-cols-1 md:grid-cols-3 gap-6 pt-6 border-t border-gray-100 dark:border-gray-800/50">
                             <!-- Actor -->
                            <div>
                                <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Actor (Who did it)</label>
                                <input type="text" class="<?php echo $inputClass; ?>" placeholder="e.g. Ali K.">
                            </div>
                             <!-- Target -->
                            <div>
                                <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Target (Who was affected)</label>
                                <input type="text" class="<?php echo $inputClass; ?>" placeholder="e.g. Sarah J.">
                            </div>
                             <!-- Date -->
                            <div>
                                <label class="block text-[11px] font-bold text-gray-400 dark:text-gray-500 uppercase tracking-widest mb-2.5 px-1">Date Range</label>
                                <div class="relative">
                                    <div class="absolute inset-y-0 left-0 pl-4 flex items-center pointer-events-none">
                                        <ion-icon name="calendar-outline" class="text-gray-400"></ion-icon>
                                    </div>
                                    <input type="text" class="<?php echo $inputClass; ?> pl-11" value="Last 7 Days">
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
            </div>

            <!-- Events Table -->
            <div class="glass-panel rounded-3xl overflow-hidden shadow-sm border-none">
                <div class="overflow-x-auto custom-scrollbar">
                    <table class="w-full text-left border-collapse">
                        <thead>
                            <tr class="bg-gray-50/50 dark:bg-slate-800/50 border-b border-gray-100 dark:border-gray-700">
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest">Time / ID</th>
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest">Actor</th>
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest">Action & Target</th>
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest text-center">System</th>
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest">Summary</th>
                                <th class="py-5 px-6 text-[11px] font-bold text-gray-500 dark:text-gray-400 uppercase tracking-widest text-right">Changes</th>
                            </tr>
                        </thead>
                        <tbody class="divide-y divide-gray-50 dark:divide-gray-700/30">
                            <?php foreach ($events as $evt): ?>
                            <tr class="group hover:bg-gray-50/50 dark:hover:bg-slate-800/30 transition-all">
                                <!-- Time / ID -->
                                <td class="py-5 px-6">
                                    <div class="text-sm font-bold text-gray-800 dark:text-white font-mono"><?php echo date('H:i', strtotime($evt['time'])); ?></div>
                                    <div class="text-[10px] text-gray-500 font-bold tracking-tighter mt-0.5"><?php echo date('M d, Y', strtotime($evt['time'])); ?></div>
                                    <div class="text-[10px] text-gray-400 font-mono mt-1 opacity-60">ID: <?php echo $evt['id']; ?></div>
                                </td>

                                <!-- Actor -->
                                <td class="py-5 px-6">
                                    <div class="flex items-center gap-3">
                                        <div class="w-10 h-10 rounded-2xl bg-orange-100 dark:bg-orange-500/10 flex items-center justify-center text-xs font-bold text-orange-600 dark:text-orange-400 border border-white dark:border-slate-800 shadow-sm group-hover:scale-105 transition-transform">
                                            <?php echo substr($evt['actor'], 0, 1); ?>
                                        </div>
                                        <div>
                                            <div class="text-sm font-bold text-gray-800 dark:text-white"><?php echo $evt['actor']; ?></div>
                                            <div class="text-[10px] text-gray-400 font-bold uppercase tracking-widest"><?php echo $evt['actor_role']; ?></div>
                                        </div>
                                    </div>
                                </td>

                                <!-- Action & Target -->
                                <td class="py-5 px-6">
                                    <div class="flex flex-col gap-1.5">
                                        <span class="text-[10px] font-extrabold text-primary uppercase tracking-widest drop-shadow-sm"><?php echo $evt['action']; ?></span>
                                        <div class="text-sm font-bold text-gray-800 dark:text-white flex items-center gap-2">
                                            <div class="w-4 h-px bg-gray-200 dark:bg-gray-700"></div>
                                            <?php echo $evt['target']; ?>
                                        </div>
                                    </div>
                                </td>

                                <!-- System -->
                                <td class="py-5 px-6 text-center">
                                     <span class="inline-flex items-center px-3 py-1 rounded-lg bg-gray-100 dark:bg-slate-700/50 text-[10px] font-bold font-mono text-gray-600 dark:text-gray-300 border border-gray-200 dark:border-gray-700/50 uppercase">
                                        <?php echo $evt['system']; ?>
                                     </span>
                                </td>

                                <!-- Summary -->
                                <td class="py-5 px-6">
                                    <div class="text-sm text-gray-700 dark:text-gray-300 font-extrabold tracking-tight"><?php echo $evt['summary']; ?></div>
                                    <div class="text-[11px] text-gray-500 italic mt-1 bg-gray-50 dark:bg-slate-800/40 px-3 py-1 rounded-lg border border-gray-100 dark:border-gray-700/50 inline-block max-w-[200px] truncate" title="<?php echo htmlspecialchars($evt['reason']); ?>">"<?php echo $evt['reason']; ?>"</div>
                                </td>

                                <!-- Actions -->
                                <td class="py-5 px-6 text-right">
                                    <button onclick='openDiffModal(<?php echo json_encode($evt); ?>)' class="px-4 py-2 bg-primary/5 hover:bg-primary text-primary hover:text-white dark:bg-primary/10 dark:hover:bg-primary rounded-xl text-[11px] font-extrabold uppercase tracking-widest transition-all shadow-sm ring-1 ring-primary/20 hover:ring-primary hover:shadow-primary/20 transform hover:-translate-y-0.5">
                                        View Diff
                                    </button>
                                </td>
                            </tr>
                            <?php endforeach; ?>
                        </tbody>
                    </table>
                </div>
            </div>

        </div>
    </div>

    <!-- Diff Modal -->
    <div id="diffModal" class="fixed inset-0 z-[100] hidden">
        <div class="absolute inset-0 bg-slate-900/60 backdrop-blur-md transition-opacity" onclick="closeModal('diffModal')"></div>
        <div class="absolute top-1/2 left-1/2 transform -translate-x-1/2 -translate-y-1/2 w-full max-w-2xl bg-white dark:bg-slate-900 rounded-[2.5rem] shadow-[0_35px_60px_-15px_rgba(0,0,0,0.3)] border border-gray-100 dark:border-gray-800 overflow-hidden flex flex-col max-h-[90vh]">
            <!-- Header -->
            <div class="px-8 py-6 border-b border-gray-100 dark:border-gray-800 flex justify-between items-center bg-gray-50/50 dark:bg-slate-800/30 relative">
                <div class="absolute top-0 right-1/4 w-32 h-16 bg-primary/10 rounded-full blur-[60px]"></div>
                <div class="relative z-10 flex items-center gap-4">
                    <div class="w-12 h-12 rounded-2xl bg-primary flex items-center justify-center text-white shadow-lg shadow-orange-500/20">
                        <ion-icon name="git-compare-outline" class="text-2xl"></ion-icon>
                    </div>
                    <div>
                         <div class="flex items-center gap-2 mb-1">
                            <span class="text-[10px] font-mono font-extrabold text-gray-400 uppercase tracking-widest" id="modalEventId">#evt_0000</span>
                            <div class="w-1 h-1 rounded-full bg-gray-300 dark:bg-gray-600"></div>
                            <span class="text-[10px] font-extrabold px-2.5 py-0.5 rounded-full bg-blue-100 text-blue-700 dark:bg-blue-900/40 dark:text-blue-300 uppercase tracking-widest" id="modalAction">Action</span>
                        </div>
                         <h3 class="text-xl font-bold text-gray-800 dark:text-white">Change Comparison</h3>
                    </div>
                </div>
                <button onclick="closeModal('diffModal')" class="p-3 rounded-2xl hover:bg-gray-200 dark:hover:bg-slate-700 transition-all group overflow-hidden relative">
                    <ion-icon name="close-outline" class="text-2xl text-gray-500 group-hover:scale-110 group-hover:rotate-90 transition-all"></ion-icon>
                </button>
            </div>
            
            <!-- Metadata Bar -->
            <div class="px-8 py-4 bg-white dark:bg-slate-900/90 border-b border-gray-50 dark:border-gray-800 flex flex-wrap gap-6 text-xs text-gray-500 font-medium">
                <div class="flex items-center gap-2">
                    <ion-icon name="person-outline" class="text-gray-400"></ion-icon> 
                    <span id="modalActor" class="text-gray-900 dark:text-white font-bold">Actor</span>
                </div>
                <div class="flex items-center gap-2">
                    <ion-icon name="arrow-forward-outline" class="text-primary"></ion-icon> 
                    <span id="modalTarget" class="text-gray-900 dark:text-white font-bold">Target</span>
                </div>
                <div class="flex items-center gap-2 ml-auto text-[10px] font-bold uppercase tracking-widest opacity-60">
                    <ion-icon name="time-outline"></ion-icon> <span id="modalTime">Time</span>
                </div>
            </div>
            
            <!-- Scrollable Content -->
            <div class="flex-1 overflow-y-auto p-8 custom-scrollbar space-y-8">
                
                <!-- Reason Block -->
                <div class="p-6 rounded-3xl bg-gray-50/50 dark:bg-slate-800/20 border border-gray-100/50 dark:border-gray-800/50 relative overflow-hidden group">
                    <div class="absolute top-0 right-0 p-6 opacity-[0.05] group-hover:scale-110 transition-transform">
                        <ion-icon name="chatbubble-ellipses-outline" class="text-5xl"></ion-icon>
                    </div>
                    <div class="relative z-10">
                        <div class="text-[10px] font-extrabold text-gray-400 uppercase tracking-widest mb-2">Internal Context / Reason</div>
                        <p id="modalReason" class="text-sm text-gray-700 dark:text-gray-300 italic font-medium leading-relaxed">"No reason provided"</p>
                    </div>
                </div>

                <!-- Diff Table -->
                <div class="rounded-3xl border border-gray-100 dark:border-gray-800 overflow-hidden shadow-2xl relative">
                    <div class="absolute top-0 left-0 w-full h-1.5 bg-gradient-to-r from-red-500/20 via-primary/20 to-green-500/20"></div>
                    <table class="w-full text-left">
                        <thead class="bg-gray-50/50 dark:bg-slate-800/50 text-[10px] font-extrabold uppercase tracking-widest text-gray-400 dark:text-gray-500">
                            <tr>
                                <th class="px-6 py-4 w-1/4 border-b dark:border-gray-800">Field</th>
                                <th class="px-6 py-4 w-1/3 border-b dark:border-gray-800 border-r dark:border-r-gray-800 bg-red-50/20 dark:bg-red-900/10 text-red-500">Old State</th>
                                <th class="px-6 py-4 w-1/3 border-b dark:border-gray-800 bg-green-50/20 dark:bg-green-900/10 text-green-500">New State</th>
                            </tr>
                        </thead>
                        <tbody id="diffTableBody" class="divide-y divide-gray-50 dark:divide-gray-800/30">
                            <!-- JS Generated Rows -->
                        </tbody>
                    </table>
                </div>
            </div>
            
            <!-- Footer Actions -->
            <div class="p-6 border-t border-gray-50 dark:border-gray-800 bg-gray-50/50 dark:bg-slate-800/20 flex justify-between items-center px-8">
                <button class="px-4 py-2 text-xs font-bold text-gray-400 hover:text-gray-600 dark:hover:text-gray-200 transition-all flex items-center gap-2">
                    <ion-icon name="print-outline"></ion-icon> Print View
                </button>
                <div class="flex gap-3">
                    <button onclick="closeModal('diffModal')" class="px-6 py-3 bg-white dark:bg-slate-700 border border-gray-200 dark:border-gray-600 rounded-2xl text-[11px] font-extrabold uppercase tracking-widest shadow-sm hover:shadow-md transition-all">Dismiss</button>
                    <button class="px-6 py-3 bg-primary hover:bg-orange-600 text-white rounded-2xl text-[11px] font-extrabold uppercase tracking-widest shadow-lg shadow-orange-500/20 transition-all">Verify Action</button>
                </div>
            </div>
        </div>
    </div>

    <script>
        document.addEventListener('DOMContentLoaded', () => {
             initCustomSelects(); 
        });

        function closeModal(id) {
            document.getElementById(id).classList.add('hidden');
        }

        function openDiffModal(data) {
            // Populate Header
            document.getElementById('modalEventId').textContent = '#' + data.id;
            document.getElementById('modalAction').textContent = data.action;
            document.getElementById('modalActor').textContent = data.actor;
            document.getElementById('modalTarget').textContent = data.target;
            document.getElementById('modalTime').textContent = data.time;
            document.getElementById('modalReason').textContent = '"' + data.reason + '"';

            // Generate Diff Rows
            const tbody = document.getElementById('diffTableBody');
            tbody.innerHTML = '';
            
            data.changes.forEach(change => {
                const tr = document.createElement('tr');
                tr.className = 'bg-white dark:bg-slate-900 group';
                
                let oldValHtml = change.old;
                let newValHtml = change.new;
                
                if (change.type === 'array') {
                    const oldArr = Array.isArray(change.old) ? change.old : [change.old];
                    const newArr = Array.isArray(change.new) ? change.new : [change.new];
                    
                    const removed = oldArr.filter(x => !newArr.includes(x));
                    const added = newArr.filter(x => !oldArr.includes(x));
                    const kept = oldArr.filter(x => newArr.includes(x));

                    oldValHtml = oldArr.map(x => `<span class="inline-block bg-gray-100 dark:bg-slate-800 px-2 py-0.5 rounded text-[10px] font-bold mr-1 mb-1 border border-gray-200 dark:border-gray-700">${x}</span>`).join('');
                    
                    newValHtml = '';
                    if (removed.length > 0) {
                        newValHtml += removed.map(x => `<span class="inline-block bg-red-100 text-red-700 dark:bg-red-900/30 px-2 py-0.5 rounded text-[10px] font-extrabold mr-1 mb-1 line-through border border-red-200 dark:border-red-800/50" title="Removed">${x}</span>`).join('');
                    }
                    if (added.length > 0) {
                        newValHtml += added.map(x => `<span class="inline-block bg-green-100 text-green-700 dark:bg-green-900/40 px-2 py-0.5 rounded text-[10px] font-extrabold mr-1 mb-1 border border-green-200 dark:border-green-800/50" title="Added">${x}</span>`).join('');
                    }
                    if (kept.length > 0) {
                        newValHtml += kept.map(x => `<span class="inline-block bg-gray-50 text-gray-400 border border-gray-200 dark:border-gray-700 px-2 py-0.5 rounded text-[10px] font-bold mr-1 mb-1 opacity-40">${x}</span>`).join('');
                    }
                } else {
                    oldValHtml = `<span class="opacity-80">${oldValHtml}</span>`;
                    newValHtml = `<span class="font-extrabold text-gray-800 dark:text-white px-2 py-1 bg-primary/5 dark:bg-primary/20 rounded-lg ring-1 ring-primary/20">${newValHtml}</span>`;
                }

                tr.innerHTML = `
                    <td class="px-6 py-4 font-extrabold text-[11px] uppercase tracking-wider text-gray-600 dark:text-gray-400 border-r border-gray-50 dark:border-gray-800/50">${change.field}</td>
                    <td class="px-6 py-4 text-gray-600 dark:text-gray-400 font-mono text-xs border-r border-gray-50 dark:border-gray-800/50 bg-red-50/10 dark:bg-red-900/5">${oldValHtml}</td>
                    <td class="px-6 py-4 text-gray-800 dark:text-white font-mono text-xs bg-green-50/10 dark:bg-green-900/10">${newValHtml}</td>
                `;
                tbody.appendChild(tr);
            });

            document.getElementById('diffModal').classList.remove('hidden');
        }

        // --- Custom Select Logic ---
        function initCustomSelects() {
            document.querySelectorAll('.custom-select-trigger').forEach(trigger => {
                trigger.addEventListener('click', (e) => {
                    e.stopPropagation();
                    const container = trigger.closest('.custom-select-container');
                    const dropdown = container.querySelector('.custom-select-dropdown');
                    closeAllSelects(trigger);

                    const isHidden = dropdown.classList.contains('invisible');
                    if (isHidden) {
                        dropdown.classList.remove('opacity-0', 'invisible', '-translate-y-2');
                        trigger.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.add('rotate-180', 'text-primary');
                    } else {
                        dropdown.classList.add('opacity-0', 'invisible', '-translate-y-2');
                        trigger.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.remove('rotate-180', 'text-primary');
                    }
                });
            });

            document.querySelectorAll('.custom-select-option').forEach(option => {
                option.addEventListener('click', (e) => {
                    e.stopPropagation();
                    const container = option.closest('.custom-select-container');
                    const hiddenInput = container.querySelector('input[type="hidden"]');
                    const triggerLabel = container.querySelector('.select-label');
                    const dropdown = container.querySelector('.custom-select-dropdown');
                    const trigger = container.querySelector('.custom-select-trigger');
                    
                    hiddenInput.value = option.dataset.value;
                    triggerLabel.textContent = option.dataset.label;
                    
                    container.querySelectorAll('.custom-select-option').forEach(opt => {
                        opt.classList.remove('active');
                        opt.querySelector('.check-icon')?.classList.replace('opacity-100', 'opacity-0');
                    });
                    
                    option.classList.add('active');
                    option.querySelector('.check-icon')?.classList.replace('opacity-0', 'opacity-100');
                    
                    dropdown.classList.add('opacity-0', 'invisible', '-translate-y-2');
                    trigger.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.remove('rotate-180', 'text-primary');
                });
            });
            document.addEventListener('click', () => closeAllSelects());
       }

       function closeAllSelects(exceptTrigger) {
            document.querySelectorAll('.custom-select-container').forEach(container => {
                const dropdown = container.querySelector('.custom-select-dropdown');
                const trigger = container.querySelector('.custom-select-trigger');
                if (exceptTrigger && trigger === exceptTrigger) return;
                dropdown.classList.add('opacity-0', 'invisible', '-translate-y-2');
                trigger.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.remove('rotate-180', 'text-primary');
            });
       }
    </script>
    
    <!-- Footer -->
    <?php include 'includes/footer.php'; ?>
</main>
