function initStoreFormPage() {
    console.log("--- initStoreFormPage Triggered ---");
    // Check if we are actually on the store form page
    const form = document.getElementById('storeForm');
    if (!form) {
        console.log("initStoreFormPage: #storeForm not found, skipping.");
        return;
    }
    console.log("initStoreFormPage: Initializing components...");

    initTabs();
    initLogoPreview();
    initReceiptLogoPreview();
    initReportLogoPreview();
    initConfigToggles();
    initStatusToggleOnLoad();
    initCustomSelects();
    checkEditMode();
    populateCloneSelect();
    forceRemoveBlur();
    console.log("initStoreFormPage: Successfully finished.");
}

function forceRemoveBlur() {
    const elements = [
        document.getElementById('sidebar'),
        document.getElementById('topbar'),
        document.getElementById('storeFormContent'),
        document.getElementById('storeDevicesContent'),
        document.querySelector('main')
    ];
    elements.forEach(el => {
        if (el) {
            el.classList.remove('blur-sm', 'blur', 'transition-all', 'duration-300');
            el.style.filter = ''; // Also clear inline styles if any
        }
    });
}

// 1. Immediate execution (for when script is re-appended)
console.log("store_form.js: Script executed.");
initStoreFormPage();

// 2. DOMContentLoaded (standard load)
document.addEventListener('DOMContentLoaded', () => {
    console.log("store_form.js: DOMContentLoaded fired.");
    initStoreFormPage();
});

// 3. pageLoaded (SPA transition)
if (!window.storeForm_initialized) {
    console.log("store_form.js: Adding pageLoaded listener.");
    document.addEventListener('pageLoaded', (e) => {
        console.log("store_form.js: pageLoaded event caught.", e.detail.url);
        initStoreFormPage();
    });
    window.storeForm_initialized = true;
} else {
    console.log("store_form.js: pageLoaded listener already exists.");
}

// --- Tab System ---
function initTabs() {
    const tabContainer = document.getElementById('formTabs');
    if (!tabContainer || tabContainer.dataset.delegated) return;

    tabContainer.addEventListener('click', (e) => {
        const tab = e.target.closest('.tab-btn');
        if (!tab) return;

        const tabs = tabContainer.querySelectorAll('.tab-btn');
        const contents = document.querySelectorAll('.tab-content');

        // Remove active classes
        tabs.forEach(t => {
            t.classList.remove('active', 'border-primary', 'text-primary');
            t.classList.add('border-transparent', 'text-gray-500', 'dark:text-gray-400');
        });

        // Add active classes
        tab.classList.add('active', 'border-primary', 'text-primary');
        tab.classList.remove('border-transparent', 'text-gray-500', 'dark:text-gray-400');

        // Hide/Show content
        contents.forEach(c => c.classList.add('hidden'));
        const targetId = tab.dataset.target;
        const target = document.getElementById(targetId);
        if (target) {
            target.classList.remove('hidden');
            target.classList.add('animate-fade-in-up');
        }
    });

    tabContainer.dataset.delegated = "true";
}

// --- Logo Preview ---
function initLogoPreview() {
    const input = document.getElementById('logoInput');
    const preview = document.getElementById('logoPreview');
    const placeholder = document.getElementById('logoPlaceholder');
    const overlay = document.getElementById('logoOverlay'); // Overlay element
    const removeBtn = document.getElementById('removeLogoBtn');

    if (input && preview) {
        input.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                // Validate File Type (JPG/PNG/GIF/SVG)
                const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/pjpeg', 'image/x-png', 'image/gif', 'image/svg+xml'];
                const fileType = file.type.toLowerCase();
                const fileName = file.name.toLowerCase();
                const validExtensions = ['.jpg', '.jpeg', '.png', '.gif', '.svg'];

                const isValidMime = validTypes.includes(fileType);
                const isValidExt = validExtensions.some(ext => fileName.endsWith(ext));

                if (!isValidMime && !isValidExt) {
                    if (window.showToast) {
                        showToast('Only JPG, PNG, GIF, and SVG files are allowed.', 'error');
                    } else {
                        console.warn('Toast system missing: Invalid file type.');
                    }
                    input.value = ''; // Clear input
                    return;
                }

                const reader = new FileReader();
                reader.onload = (e) => {
                    preview.src = e.target.result;
                    preview.classList.remove('hidden');
                    if (placeholder) placeholder.classList.add('hidden');
                    if (removeBtn) removeBtn.classList.remove('hidden'); // Show remove button
                }
                reader.readAsDataURL(file);
            }
        });
    }

    // Expose remove function to global scope or bind it here
    window.removeLogo = function (e) {
        e.stopPropagation(); // Stop click from triggering hidden input
        e.preventDefault();

        // Clear input
        if (input) input.value = '';

        // Reset preview
        if (preview) {
            preview.src = '';
            preview.classList.add('hidden');
        }

        // Show placeholder
        if (placeholder) placeholder.classList.remove('hidden');

        // Hide remove button
        if (removeBtn) removeBtn.classList.add('hidden');
    }
}

// --- Receipt Logo Preview ---
function initReceiptLogoPreview() {
    const input = document.getElementById('receiptLogoInput');
    const preview = document.getElementById('receiptLogoPreview');
    const placeholder = document.getElementById('receiptLogoPlaceholder');
    const removeBtn = document.getElementById('removeReceiptLogoBtn');

    if (input && preview) {
        input.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                // Validate File Type (JPG/PNG only)
                const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/pjpeg', 'image/x-png'];
                const fileType = file.type.toLowerCase();
                const fileName = file.name.toLowerCase();
                const validExtensions = ['.jpg', '.jpeg', '.png'];

                const isValidMime = validTypes.includes(fileType);
                const isValidExt = validExtensions.some(ext => fileName.endsWith(ext));

                if (!isValidMime && !isValidExt) {
                    if (window.showToast) {
                        showToast('Only JPG and PNG files are allowed.', 'error');
                    } else {
                        console.warn('Toast system missing: Only JPG and PNG files are allowed.');
                    }
                    input.value = ''; // Clear input
                    return;
                }

                const reader = new FileReader();
                reader.onload = (e) => {
                    preview.src = e.target.result;
                    preview.classList.remove('hidden');
                    if (placeholder) placeholder.classList.add('hidden');
                    if (removeBtn) removeBtn.classList.remove('hidden');
                }
                reader.readAsDataURL(file);
            }
        });
    }

    // Expose remove function
    window.removeReceiptLogo = function (e) {
        e.stopPropagation();
        e.preventDefault();

        if (input) input.value = '';

        if (preview) {
            preview.src = '';
            preview.classList.add('hidden');
        }

        if (placeholder) placeholder.classList.remove('hidden');
        if (removeBtn) removeBtn.classList.add('hidden');
    }
}

// --- Report Logo Preview ---
function initReportLogoPreview() {
    const input = document.getElementById('reportLogoInput');
    const preview = document.getElementById('reportLogoPreview');
    const placeholder = document.getElementById('reportLogoPlaceholder');
    const removeBtn = document.getElementById('removeReportLogoBtn');

    if (input && preview) {
        input.addEventListener('change', (e) => {
            const file = e.target.files[0];
            if (file) {
                // Validate File Type (JPG/PNG only)
                const validTypes = ['image/jpeg', 'image/png', 'image/jpg', 'image/pjpeg', 'image/x-png'];
                const fileType = file.type.toLowerCase();
                const fileName = file.name.toLowerCase();
                const validExtensions = ['.jpg', '.jpeg', '.png'];

                const isValidMime = validTypes.includes(fileType);
                const isValidExt = validExtensions.some(ext => fileName.endsWith(ext));

                if (!isValidMime && !isValidExt) {
                    if (window.showToast) {
                        showToast('Only JPG and PNG files are allowed.', 'error');
                    } else {
                        console.warn('Toast system missing: Only JPG and PNG files are allowed.');
                    }
                    input.value = ''; // Clear input
                    return;
                }

                const reader = new FileReader();
                reader.onload = (e) => {
                    preview.src = e.target.result;
                    preview.classList.remove('hidden');
                    if (placeholder) placeholder.classList.add('hidden');
                    if (removeBtn) removeBtn.classList.remove('hidden');
                }
                reader.readAsDataURL(file);
            }
        });
    }

    // Expose remove function
    window.removeReportLogo = function (e) {
        e.stopPropagation();
        e.preventDefault();

        if (input) input.value = '';

        if (preview) {
            preview.src = '';
            preview.classList.add('hidden');
        }

        if (placeholder) placeholder.classList.remove('hidden');
        if (removeBtn) removeBtn.classList.add('hidden');
    }
}

// --- Status Toggle Logic ---
// Removed as Tax is always enabled / managed via specific fields now
/*
function initTaxToggle() {
    const toggle = document.getElementById('taxToggle');
    const fields = document.getElementById('taxFields');

    if (toggle && fields) {
        toggle.addEventListener('change', () => {
            if (toggle.checked) {
                fields.classList.remove('opacity-50', 'pointer-events-none', 'grayscale');
            } else {
                fields.classList.add('opacity-50', 'pointer-events-none', 'grayscale');
            }
        });

        // Init state
        if (!toggle.checked) {
            fields.classList.add('opacity-50', 'pointer-events-none', 'grayscale');
        }
    }
}
*/

// --- Config Toggles Logic ---
function initConfigToggles() {
    setupToggle('input[name="coupon_applicable"]', 'couponFields');
    setupToggle('input[name="bag_fee"]', 'bagFeeFields');
    setupToggle('input[name="is_cashback_enabled"]', 'cashbackFields');
    setupToggle('input[name="is_credit_enabled"]', 'creditCardFields');

    // Receipt Header/Footer Toggles
    setupToggle('input[name="enable_receipt_header"]', 'receiptHeaderFields', true);
    setupToggle('input[name="enable_receipt_footer"]', 'receiptFooterFields', true);

    // Reports Toggles

    // Receipt Feature Toggles
    setupToggle('input[name="show_receipt_qr"]', 'qrLinkField');
    setupToggle('input[name="show_receipt_logo"]', 'receiptLogoWrapper', true);

    // Security Toggles (Max Cash & Txn Lock)
    setupInputToggle('input[name="enable_max_cash_warning"]', 'input[name="max_cash_warning"]');
    setupInputToggle('input[name="enable_txn_value_lock"]', 'input[name="txn_value_lock"]');

    function setupInputToggle(toggleSelector, inputSelector) {
        const toggle = document.querySelector(toggleSelector);
        const input = document.querySelector(inputSelector);

        if (toggle && input) {
            function update() {
                if (toggle.checked) {
                    input.disabled = false;
                    input.classList.remove('opacity-50', 'bg-gray-50', 'pointer-events-none');
                    input.classList.add('bg-white', 'dark:bg-slate-800');
                } else {
                    input.disabled = true;
                    input.classList.add('opacity-50', 'bg-gray-50', 'pointer-events-none');
                    input.classList.remove('bg-white', 'dark:bg-slate-800');
                }
            }
            toggle.addEventListener('change', update);
            update();
        }
    }

    function setupToggle(selector, targetId, useDisabling = false) {
        const toggle = document.querySelector(selector);
        const target = document.getElementById(targetId);

        if (toggle && target) {
            function updateState() {
                if (toggle.checked) {
                    if (useDisabling) {
                        target.classList.remove('opacity-50', 'pointer-events-none');
                        // Optional: Re-enable inputs
                        target.querySelectorAll('input').forEach(i => i.disabled = false);
                    } else {
                        target.classList.remove('hidden');
                    }
                } else {
                    if (useDisabling) {
                        target.classList.add('opacity-50', 'pointer-events-none');
                        // Optional: Disable inputs
                        target.querySelectorAll('input').forEach(i => i.disabled = true);
                    } else {
                        target.classList.add('hidden');
                    }
                }
            }

            // Event listener
            toggle.addEventListener('change', updateState);
            // Initial state check
            updateState();
        }
    }

    // Advanced Credit Card Logic (No Limit -> Disable Amount)
    const noLimitToggle = document.querySelector('input[name="credit_no_limit"]');
    const limitInput = document.querySelector('input[name="credit_limit_amount"]');
    const limitWrapper = document.getElementById('creditLimitWrapper');

    if (noLimitToggle && limitInput && limitWrapper) {
        function updateLimitState() {
            if (noLimitToggle.checked) {
                limitInput.disabled = true;
                limitWrapper.classList.add('opacity-50', 'pointer-events-none');
                limitInput.value = ''; // Optional: clear value when no limit
            } else {
                limitInput.disabled = false;
                limitWrapper.classList.remove('opacity-50', 'pointer-events-none');
            }
        }

        noLimitToggle.addEventListener('change', updateLimitState);
        updateLimitState(); // Init
    }
    // Receipt Field Defaults Logic
    setupReceiptFieldToggle('use_store_name', 'receipt_title', () => {
        return document.querySelector('input[placeholder="e.g. Cardiff Central"]').value;
    });

    setupReceiptFieldToggle('use_store_address', 'receipt_header_1', () => {
        const addr1 = document.querySelector('input[placeholder="Street name, Building number"]').value;
        const city = document.querySelector('input[placeholder="e.g. Cardiff"]').value;
        return [addr1, city].filter(Boolean).join(', ');
    });

    setupReceiptFieldToggle('use_store_contact', 'receipt_header_2', () => {
        const mobile = document.querySelector('input[type="tel"]').value;
        return mobile ? `Tel: ${mobile}` : '';
    });

    function setupReceiptFieldToggle(toggleName, targetName, valueGetter) {
        const toggle = document.querySelector(`input[name="${toggleName}"]`);
        const target = document.querySelector(`input[name="${targetName}"]`);

        if (toggle && target) {
            function update() {
                if (toggle.checked) {
                    target.value = valueGetter();
                    target.readOnly = true;
                    target.classList.add('bg-gray-50', 'text-gray-500');
                } else {
                    target.readOnly = false;
                    target.classList.remove('bg-gray-50', 'text-gray-500');
                }
            }
            toggle.addEventListener('change', update);
            // Also update on source change? Complex to track source inputs, sticking to manual refresh on toggle for now.
            // But usually we should listen to source inputs too if toggle is checked.
            // For simplicity, let's just update when toggle changes. The user can toggle off/on to refresh.
            update();
        }
    }
}

// --- Status Toggle Logic ---
function initStatusToggleOnLoad() {
    const statusCheck = document.querySelector('input[name="store_status"]');
    if (statusCheck) {
        statusCheck.addEventListener('change', (e) => {
            const label = e.target.parentElement.querySelector('.status-label');
            if (label) label.textContent = e.target.checked ? 'Active' : 'Inactive';
        });
    }
}

// --- Custom Select Logic ---
function initCustomSelects() {
    // Toggle Dropdown
    document.querySelectorAll('.custom-select-trigger').forEach(trigger => {
        trigger.addEventListener('click', (e) => {
            e.stopPropagation();
            const container = trigger.closest('.custom-select-container');
            const dropdown = container.querySelector('.custom-select-dropdown');

            // Close all other dropdowns
            closeAllSelects(trigger);

            dropdown.classList.toggle('opacity-0');
            dropdown.classList.toggle('invisible');
            dropdown.classList.toggle('-translate-y-2');

            // Rotate chevron
            const chevron = trigger.querySelector('ion-icon[name="chevron-down-outline"]');
            if (chevron) {
                if (!dropdown.classList.contains('invisible')) {
                    chevron.classList.add('rotate-180');
                    chevron.classList.add('text-primary');
                } else {
                    chevron.classList.remove('rotate-180');
                    chevron.classList.remove('text-primary');
                }
            }
        });
    });

    // Option Click
    document.querySelectorAll('.custom-select-option').forEach(option => {
        option.addEventListener('click', (e) => {
            e.stopPropagation();
            const container = option.closest('.custom-select-container');
            const hiddenInput = container.querySelector('input[type="hidden"]');
            const triggerLabel = container.querySelector('.select-label');
            const dropdown = container.querySelector('.custom-select-dropdown');
            const trigger = container.querySelector('.custom-select-trigger');

            // Update value
            const val = option.dataset.value;
            const label = option.dataset.label;

            hiddenInput.value = val;
            triggerLabel.textContent = label; // Or just label if needed

            // Update active styling
            container.querySelectorAll('.custom-select-option').forEach(opt => {
                opt.classList.remove('active');
                const check = opt.querySelector('.check-icon');
                if (check) {
                    check.classList.remove('opacity-100');
                    check.classList.add('opacity-0');
                }
            });
            option.classList.add('active');
            const check = option.querySelector('.check-icon');
            if (check) {
                check.classList.remove('opacity-0');
                check.classList.add('opacity-100');
            }

            // Close dropdown
            dropdown.classList.add('opacity-0');
            dropdown.classList.add('invisible');
            dropdown.classList.add('-translate-y-2');

            // Reset chevron
            const chevron = trigger.querySelector('ion-icon[name="chevron-down-outline"]');
            if (chevron) {
                chevron.classList.remove('rotate-180');
                chevron.classList.remove('text-primary');
            }
        });
    });

    // Click Outside - Add only once globally
    if (!window.storeForm_clickListenerAdded) {
        document.addEventListener('click', () => {
            closeAllSelects();
        });
        window.storeForm_clickListenerAdded = true;
    }
}

function closeAllSelects(exceptTrigger) {
    document.querySelectorAll('.custom-select-container').forEach(container => {
        const dropdown = container.querySelector('.custom-select-dropdown');
        const trigger = container.querySelector('.custom-select-trigger');

        if (exceptTrigger && trigger === exceptTrigger) return;

        if (!dropdown.classList.contains('invisible')) {
            dropdown.classList.add('opacity-0');
            dropdown.classList.add('invisible');
            dropdown.classList.add('-translate-y-2');

            // Reset chevron
            const chevron = trigger.querySelector('ion-icon[name="chevron-down-outline"]');
            if (chevron) {
                chevron.classList.remove('rotate-180');
                chevron.classList.remove('text-primary');
            }
        }
    });
}

// --- Mock Data & Edit Mode ---
const MOCK_STORES = {
    '1001': {
        name: 'FNF Cardiff',
        type: 'grocery',
        currency: 'GBP',
        timezone: 'Europe/London',
        status: 'active',
        ownerMobile: '7123 456789',
        ownerMobileCode: '+44',
        ownerEmail: 'manager@fnfcardiff.co.uk',

        addressLine1: '123 Queen Street',
        addressLine2: '',
        city: 'Cardiff',
        postcode: 'CF10 2BH',
        county: 'South Glamorgan',
        county: 'South Glamorgan',
        country: 'United Kingdom',

        // taxEnabled: true, // Removed
        // taxName: 'VAT', // Removed
        vatNumber: 'GB 123 456 789',
        companyRegNo: '09876543',
        taxLabel: 'VAT',
        legalName: 'FNF Cardiff Ltd',
        serviceCharge: 0,
        // taxRate: 20, // Removed
        taxModel: 'inclusive',

        // Receipt Config
        enableReceiptHeader: true,
        useStoreName: true,
        useStoreAddress: true,
        useStoreContact: true,

        showReceiptLogo: true,
        showReportLogo: true,
        showReceiptQr: true,
        receiptQrLink: 'https://fnf.co.uk/feedback',

        receiptTitle: 'FNF Cardiff',
        receiptHeaderLines: {
            line1: '123 Queen Street',
            line2: 'Cardiff CF10 2BH'
        },

        // receiptFooter: 'Thank you for shopping!\nVisit www.fnf.co.uk', // Removed
        enableReceiptFooter: true,
        receiptFooterLines: {
            msg: 'Thank you for shopping!',
            policy: 'Returns within 30 days',
            extra: 'www.fnf.co.uk'
        },

        receiptOptions: {
            show_vat_on_receipt: true,
            show_crn_on_receipt: false,
            show_legal_name_on_receipt: true
        },

        config: {
            is_18_plus: false,
            is_credit_enabled: true,
            is_void_enabled: true,
            is_hold_enabled: true,
            coupon_applicable: false,
            bag_fee: true,
            float: 150.00,
            cashback_fee: 0.00,
            cashback_limit: 50.00,
            denom: '10, 20'
        },
        settings: {
            forcePin: true,
            timeout: 5,
            negativeStock: false,
            txnLock: 100.00
        }
    },
    '1002': {
        name: 'FNF Swansea',
        type: 'grocery',
        currency: 'GBP',
        timezone: 'Europe/London',
        status: 'inactive',
        ownerMobile: '7987 654321',
        ownerMobileCode: '+44',
        ownerEmail: 'manager@fnfswansea.co.uk',

        addressLine1: '45 High St',
        addressLine2: '',
        city: 'Swansea',
        postcode: 'S1 1AA',
        county: 'Swansea',
        country: 'United Kingdom',

        // taxEnabled: true,
        // taxName: 'VAT',
        vatNumber: 'GB 987 654 321',
        companyRegNo: '09876544',
        taxLabel: 'VAT',
        legalName: 'FNF Swansea Ltd',
        serviceCharge: 0,
        taxRate: 20,
        taxModel: 'inclusive',

        // Receipt Config
        enableReceiptHeader: true,
        useStoreName: false,
        useStoreAddress: false,
        useStoreContact: false,

        showReceiptLogo: false,
        showReceiptQr: false,
        receiptQrLink: '',

        receiptTitle: 'FNF Swansea',
        receiptHeaderLines: {
            line1: '45 High St',
            line2: 'Swansea S1 1AA'
        },

        // receiptFooter: 'Thank you!',
        enableReceiptFooter: true,
        receiptFooterLines: {
            msg: 'Thank you!',
            policy: 'No Returns',
            extra: 'Visit us again'
        },

        receiptOptions: {
            show_vat_on_receipt: false,
            show_crn_on_receipt: true,
            show_legal_name_on_receipt: false
        },

        config: {
            is_18_plus: false,
            is_credit_enabled: true,
            is_void_enabled: false,
            is_hold_enabled: true,
            coupon_applicable: true,
            bag_fee: false
        },
        settings: {
            forcePin: true,
            timeout: 10,
            negativeStock: true,
            txnLock: 0
        }
    }
};

function checkEditMode() {
    const urlParams = new URLSearchParams(window.location.search);
    const id = urlParams.get('id');

    if (id && MOCK_STORES[id]) {
        populateForm(MOCK_STORES[id]);

        const titleEl = document.getElementById('pageHeaderTitle');
        if (titleEl) titleEl.textContent = `Store Create/Edit`;
    }
}

function populateForm(data) {
    // General - Basics
    setVal('input[placeholder="e.g. Cardiff Central"]', data.name);
    setCustomSelect('currency_code', data.currency);

    // Store Type (Radio)
    const typeRadio = document.querySelector(`input[name="store_type"][value="${data.type}"]`);
    if (typeRadio) typeRadio.checked = true;

    // Status (Toggle)
    const statusCheck = document.querySelector('input[name="store_status"]');
    if (statusCheck) {
        const isActive = (data.status === 'active');
        statusCheck.checked = isActive;
        // Trigger generic change if needed, or update label manully
        const label = statusCheck.parentElement.querySelector('.status-label');
        if (label) label.textContent = isActive ? 'Active' : 'Inactive';
    }

    // Contact
    setVal('input[type="tel"]', data.ownerMobile);
    setCustomSelect('owner_mobile_code', data.ownerMobileCode);
    setVal('input[type="email"]', data.ownerEmail);

    // Location
    setVal('input[placeholder="Street name, Building number"]', data.addressLine1);
    setVal('input[placeholder="Apartment, Suite, Unit, etc. (Optional)"]', data.addressLine2);
    setVal('input[placeholder="e.g. Cardiff"]', data.city);
    setVal('input[placeholder="e.g. CF10 2BH"]', data.postcode);
    setVal('input[placeholder="e.g. South Glamorgan"]', data.county);
    setCustomSelect('country', data.country);
    setCustomSelect('timezone', data.timezone);

    // Config
    if (data.config) {
        for (const [key, val] of Object.entries(data.config)) {
            // Bool radios (1/0)
            if (typeof val === 'boolean') {
                const r = document.querySelector(`input[name="${key}"]`);
                // Checkbox toggle type
                if (r && r.type === 'checkbox') {
                    r.checked = val;
                    const event = new Event('change');
                    r.dispatchEvent(event);
                }
            }
        }
    }

    // Tax
    // Removed legacy tax fields (toggle, name, rate)

    // Tax Model (Inclusive/Exclusive) - Removed
    /*
    const taxInclusive = document.querySelector('input[name="tax_inclusive"]');
    if (taxInclusive) {
        taxInclusive.checked = (data.taxModel === 'inclusive');
        const evt = new Event('change');
        taxInclusive.dispatchEvent(evt);
    }
    */

    setVal('input[name="vat_number"]', data.vatNumber);
    setVal('input[name="company_reg_number"]', data.companyRegNo);
    // setVal('input[name="tax_label"]', data.taxLabel);
    setVal('input[name="legal_name"]', data.legalName);
    // setVal('input[name="service_charge"]', data.serviceCharge);

    // Receipt Options - Toggles
    const receiptHeaderToggle = document.querySelector('input[name="enable_receipt_header"]');
    if (receiptHeaderToggle && data.enableReceiptHeader !== undefined) {
        receiptHeaderToggle.checked = data.enableReceiptHeader;
        receiptHeaderToggle.dispatchEvent(new Event('change'));
    }

    const receiptFooterToggle = document.querySelector('input[name="enable_receipt_footer"]');
    if (receiptFooterToggle && data.enableReceiptFooter !== undefined) {
        receiptFooterToggle.checked = data.enableReceiptFooter;
        receiptFooterToggle.dispatchEvent(new Event('change'));
    }

    // Default Toggles
    const useNameToggle = document.querySelector('input[name="use_store_name"]');
    if (useNameToggle) {
        useNameToggle.checked = !!data.useStoreName;
        useNameToggle.dispatchEvent(new Event('change'));
    }
    const useAddressToggle = document.querySelector('input[name="use_store_address"]');
    if (useAddressToggle) {
        useAddressToggle.checked = !!data.useStoreAddress;
        useAddressToggle.dispatchEvent(new Event('change'));
    }
    const useContactToggle = document.querySelector('input[name="use_store_contact"]');
    if (useContactToggle) {
        useContactToggle.checked = !!data.useStoreContact;
        useContactToggle.dispatchEvent(new Event('change'));
    }

    // Receipt Features
    const showLogoToggle = document.querySelector('input[name="show_receipt_logo"]');
    if (showLogoToggle) {
        showLogoToggle.checked = !!data.showReceiptLogo;
        showLogoToggle.dispatchEvent(new Event('change'));
    }

    const showQrToggle = document.querySelector('input[name="show_receipt_qr"]');
    if (showQrToggle) {
        showQrToggle.checked = !!data.showReceiptQr;
        showQrToggle.dispatchEvent(new Event('change'));
    }
    setVal('input[name="receipt_qr_link"]', data.receiptQrLink);

    // Receipt Fields
    // Old textareas removed
    /*
    document.querySelectorAll('textarea')[0].value = data.receiptHeader;
    document.querySelectorAll('textarea')[1].value = data.receiptFooter;
    */

    setVal('input[name="receipt_title"]', data.receiptTitle);

    if (data.receiptHeaderLines) {
        setVal('input[name="receipt_header_1"]', data.receiptHeaderLines.line1);
        setVal('input[name="receipt_header_2"]', data.receiptHeaderLines.line2);
    }

    if (data.receiptFooterLines) {
        setVal('input[name="receipt_footer_msg"]', data.receiptFooterLines.msg);
        setVal('input[name="receipt_footer_policy"]', data.receiptFooterLines.policy);
        setVal('input[name="receipt_footer_extra"]', data.receiptFooterLines.extra);
    }

    // Receipt Options
    if (data.receiptOptions) {
        for (const [key, val] of Object.entries(data.receiptOptions)) {
            const toggle = document.querySelector(`input[name="${key}"]`);
            if (toggle) {
                toggle.checked = val;
                toggle.dispatchEvent(new Event('change'));
            }
        }
    }
}

function setCustomSelect(name, value) {
    const input = document.querySelector(`input[name="${name}"]`);
    if (!input) return;

    input.value = value;
    const container = input.closest('.custom-select-container');
    // Find option by value
    const option = container.querySelector(`.custom-select-option[data-value="${value}"]`);

    if (option) {
        // Update trigger label
        const label = container.querySelector('.select-label');
        if (label) label.textContent = option.dataset.label; // Use full label from data attribute

        // Update active classes
        container.querySelectorAll('.custom-select-option').forEach(opt => {
            opt.classList.remove('active');
            const check = opt.querySelector('.check-icon');
            if (check) {
                check.classList.remove('opacity-100');
                check.classList.add('opacity-0');
            }
        });
        option.classList.add('active');
        const check = option.querySelector('.check-icon');
        if (check) {
            check.classList.remove('opacity-0');
            check.classList.add('opacity-100');
        }
    }
}

function setVal(selector, value, index = 0) {
    const els = document.querySelectorAll(selector);
    if (els[index] && value !== undefined) els[index].value = value;
}

// --- Clone (Simplified for now) ---
function openCloneModal() {
    const modal = document.getElementById('cloneModal');
    const overlay = document.getElementById('cloneModalOverlay');
    const content = document.getElementById('cloneModalContent');
    if (!modal) return;

    modal.classList.remove('hidden');

    // Apply Blur to Specific Elements
    const elementsToBlur = [
        document.getElementById('sidebar'),
        document.getElementById('topbar'),
        document.getElementById('storeFormContent')
    ];

    elementsToBlur.forEach(el => {
        if (el) {
            el.classList.add('transition-all', 'duration-300', 'blur-sm');
        }
    });

    requestAnimationFrame(() => {
        overlay.classList.remove('opacity-0');
        content.classList.remove('opacity-0', 'scale-95');
        content.classList.add('scale-100');
    });
}

function closeCloneModal() {
    const modal = document.getElementById('cloneModal');
    const overlay = document.getElementById('cloneModalOverlay');
    const content = document.getElementById('cloneModalContent');
    if (!modal) return;

    if (overlay) overlay.classList.add('opacity-0');
    if (content) {
        content.classList.remove('scale-100');
        content.classList.add('opacity-0', 'scale-95');
    }

    // Remove Blur
    const elementsToBlur = [
        document.getElementById('sidebar'),
        document.getElementById('topbar'),
        document.getElementById('storeFormContent')
    ];

    elementsToBlur.forEach(el => {
        if (el) {
            el.classList.remove('blur-sm');
        }
    });

    setTimeout(() => {
        modal.classList.add('hidden');
    }, 200);
}

function populateCloneSelect() {
    const container = document.getElementById('cloneStoreOptions');
    if (!container) return;

    container.innerHTML = ''; // Clear existing

    Object.keys(MOCK_STORES).forEach(id => {
        const store = MOCK_STORES[id];
        const opt = document.createElement('div');
        opt.className = 'custom-select-option group flex items-center justify-between w-full px-3 py-2 text-sm text-gray-700 dark:text-gray-200 hover:bg-gray-50 dark:hover:bg-slate-700/50 rounded-lg cursor-pointer transition-colors';
        opt.dataset.value = id;
        opt.dataset.label = store.name;
        opt.innerHTML = `
            <span class="truncate">${store.name}</span>
            <ion-icon name="checkmark-circle" class="text-primary text-lg ml-2 opacity-0 check-icon"></ion-icon>
        `;

        // Add Click Listener
        opt.addEventListener('click', (e) => {
            e.stopPropagation();
            const wrapper = container.closest('.custom-select-container');
            const hiddenInput = wrapper.querySelector('input[type="hidden"]');
            const triggerLabel = wrapper.querySelector('.select-label');
            const dropdown = wrapper.querySelector('.custom-select-dropdown');
            const trigger = wrapper.querySelector('.custom-select-trigger');

            // Update Value
            hiddenInput.value = id;
            triggerLabel.textContent = store.name;

            // Update Active State
            container.querySelectorAll('.custom-select-option').forEach(o => {
                o.classList.remove('active');
                const check = o.querySelector('.check-icon');
                if (check) {
                    check.classList.remove('opacity-100');
                    check.classList.add('opacity-0');
                }
            });
            opt.classList.add('active');
            const check = opt.querySelector('.check-icon');
            if (check) {
                check.classList.remove('opacity-0');
                check.classList.add('opacity-100');
            }

            // Close Dropdown
            dropdown.classList.add('opacity-0', 'invisible', '-translate-y-2');

            // Reset Chevron
            const chevron = trigger.querySelector('ion-icon[name="chevron-down-outline"]');
            if (chevron) {
                chevron.classList.remove('rotate-180', 'text-primary');
            }
        });

        container.appendChild(opt);
    });
}

function applyClone() {
    if (window.showToast) {
        showToast("Settings cloned (mock).", 'success');
    } else {
        console.log("Settings cloned (mock).");
    }
    closeCloneModal();
}

// --- Device Charges Logic ---
function calculateCharges() {
    const override = document.getElementById('manualOverride');
    // If manual override is enabled, do not auto-calculate
    if (override && override.checked) return;

    const packageType = document.getElementById('selectedPackage').value;
    const totalField = document.getElementById('totalCharges');

    // Package Rates
    const rates = {
        'starter': 20.00,
        'retail': 20.00,
        'standard': 25.00,
        'growth': 40.00
    };

    if (rates[packageType] !== undefined) {
        if (totalField) totalField.value = rates[packageType].toFixed(2);
        return;
    }

    const posInput = document.getElementById('posCount');
    const displayInput = document.getElementById('displayCount');
    const invInput = document.getElementById('inventoryAddon');

    if (posInput && displayInput && totalField) {
        // Enforce Minimum POS Count of 1
        let minPos = 1;
        if (posInput.value && parseInt(posInput.value) < minPos) {
            posInput.value = minPos;
        }

        const pos = parseInt(posInput.value) || 0;
        const display = parseInt(displayInput.value) || 0;
        const hasInv = invInput ? invInput.checked : false;

        // Define Bundle Configurations for Matching
        const bundles = [
            { pos: 2, display: 0, inv: false, price: 20.00 }, // Starter
            { pos: 1, display: 0, inv: true, price: 20.00 },  // Retail Starter
            { pos: 2, display: 1, inv: true, price: 25.00 },  // Standard
            { pos: 4, display: 2, inv: true, price: 40.00 }   // Growth
        ];

        let total = 0;
        const match = bundles.find(b => b.pos === pos && b.display === display && b.inv === hasInv);

        if (match) {
            total = match.price;
        } else {
            // Fallback Linear Calculation
            const posRate = 10.00;
            const displayRate = 5.00;
            const invRate = 5.00;

            total = (pos * posRate) + (display * displayRate);
            if (hasInv) total += invRate;
        }

        totalField.value = total.toFixed(2);
    }
}

function toggleManualOverride() {
    const override = document.getElementById('manualOverride');
    const totalField = document.getElementById('totalCharges');

    if (override && totalField) {
        if (override.checked) {
            // Enable editing
            totalField.removeAttribute('readonly');
            // Visual feedback could be added here
        } else {
            // Disable editing and recalculate
            totalField.setAttribute('readonly', 'true');
            calculateCharges();
        }
    }
}

function selectPackage(type, card) {
    document.getElementById('selectedPackage').value = type;

    // Visual Updates
    document.querySelectorAll('.package-card').forEach(c => {
        c.classList.remove('border-primary', 'bg-orange-50/50', 'dark:bg-orange-900/10');
        // Reset base style
        c.classList.add('border-gray-200', 'dark:border-gray-700', 'bg-white', 'dark:bg-slate-800');

        c.querySelector('.absolute .rounded-full').classList.remove('bg-primary', 'border-primary');
        c.querySelector('.absolute .rounded-full').classList.add('border-gray-300', 'dark:border-gray-600');
        c.querySelector('ion-icon[name="checkmark"]').classList.add('opacity-0');
        c.querySelector('ion-icon[name="checkmark"]').classList.remove('text-white');
    });

    if (card) {
        card.classList.remove('border-gray-200', 'dark:border-gray-700', 'bg-white', 'dark:bg-slate-800');
        // Apply active style
        card.classList.add('border-primary', 'bg-orange-50/50', 'dark:bg-orange-900/10');

        card.querySelector('.absolute .rounded-full').classList.remove('border-gray-300', 'dark:border-gray-600');
        card.querySelector('.absolute .rounded-full').classList.add('bg-primary', 'border-primary');
        card.querySelector('ion-icon[name="checkmark"]').classList.remove('opacity-0');
        card.querySelector('ion-icon[name="checkmark"]').classList.add('text-white');
    }

    // Logic Updates
    const wrapper = document.getElementById('customConfigWrapper');
    const note = document.getElementById('chargeCalculationNote');
    const invInput = document.getElementById('inventoryAddon');

    // Package Configurations
    const packageConfig = {
        'starter': { pos: 2, display: 0, inv: false, label: "Starter Kit Rate" },
        'retail': { pos: 1, display: 0, inv: true, label: "Retail Starter Rate" },
        'standard': { pos: 2, display: 1, inv: true, label: "Standard Bundle Rate" },
        'growth': { pos: 4, display: 2, inv: true, label: "Growth Pack Rate" }
    };

    if (packageConfig[type]) {
        wrapper.classList.add('hidden');
        document.getElementById('posCount').value = packageConfig[type].pos;
        document.getElementById('displayCount').value = packageConfig[type].display;
        if (invInput) invInput.checked = packageConfig[type].inv;

        note.textContent = packageConfig[type].label;
        document.getElementById('manualOverride').checked = false;
        toggleManualOverride();
    } else {
        wrapper.classList.remove('hidden');
        calculateCharges();
    }

    calculateCharges();
}
