document.addEventListener('DOMContentLoaded', () => {
    // --- SIDEBAR LOGIC ---
    const sidebar = document.getElementById('sidebar');
    const toggleBtn = document.getElementById('sidebarToggle');
    const sidebarTexts = document.querySelectorAll('.sidebar-text');

    const isCollapsed = localStorage.getItem('sidebarState') === 'collapsed';
    if (isCollapsed) applyCollapsedState(true);

    if (toggleBtn) {
        toggleBtn.addEventListener('click', () => {
            const willCollapse = !sidebar.classList.contains('w-20');
            applyCollapsedState(willCollapse);
        });
    }

    function applyCollapsedState(collapsed) {
        if (!sidebar) return;
        if (collapsed) {
            sidebar.classList.remove('w-64');
            sidebar.classList.add('w-20');
            sidebarTexts.forEach(el => {
                el.classList.add('opacity-0', 'w-0', 'px-0');
                setTimeout(() => el.style.display = 'none', 300);
            });
            localStorage.setItem('sidebarState', 'collapsed');
        } else {
            sidebar.classList.remove('w-20');
            sidebar.classList.add('w-64');
            sidebarTexts.forEach(el => {
                el.style.display = '';
                setTimeout(() => el.classList.remove('opacity-0', 'w-0', 'px-0'), 10);
            });
            localStorage.setItem('sidebarState', 'expanded');
        }
    }

    // --- FULLSCREEN LOGIC ---
    const fullscreenBtn = document.getElementById('fullscreenToggle');
    if (fullscreenBtn) {
        fullscreenBtn.addEventListener('click', () => {
            if (!document.fullscreenElement) {
                document.documentElement.requestFullscreen().catch(err => console.error(err));
            } else if (document.exitFullscreen) {
                document.exitFullscreen();
            }
        });
        document.addEventListener('fullscreenchange', () => {
            const icon = fullscreenBtn.querySelector('ion-icon');
            if (icon) icon.setAttribute('name', document.fullscreenElement ? 'contract-outline' : 'scan-outline');
        });
    }

    // --- DROPDOWN LOGIC ---
    setupDropdown('storeSelectBtn', 'storeDropdown', 'currentStore', '.store-option');
    setupDropdown('typeSelectBtn', 'typeDropdown', 'currentType', '.type-option');
    setupDropdown('typeFilterBtn', 'typeFilterDropdown', 'currentTypeFilter', '.type-filter-option');
    setupDropdown('statusFilterBtn', 'statusFilterDropdown', 'currentStatusFilter', '.status-filter-option');
    setupDropdown('deviceStoreFilterBtn', 'deviceStoreFilterDropdown', 'deviceStoreFilterText', '.device-store-option');
    setupDropdown('deviceStatusFilterBtn', 'deviceStatusFilterDropdown', 'deviceStatusFilterText', '.device-status-option');

    // Global Click Outside Dropdowns
    document.addEventListener('click', (e) => {
        const dropdowns = [
            { btn: 'storeSelectBtn', menu: 'storeDropdown' },
            { btn: 'typeSelectBtn', menu: 'typeDropdown' },
            { btn: 'typeFilterBtn', menu: 'typeFilterDropdown' },
            { btn: 'statusFilterBtn', menu: 'statusFilterDropdown' },
            { btn: 'deviceStoreFilterBtn', menu: 'deviceStoreFilterDropdown' },
            { btn: 'deviceStatusFilterBtn', menu: 'deviceStatusFilterDropdown' }
        ];

        dropdowns.forEach(d => {
            const btn = document.getElementById(d.btn);
            const menu = document.getElementById(d.menu);
            if (btn && menu && !btn.contains(e.target) && !menu.contains(e.target)) {
                closeDropdown(menu, btn);
            }
        });
    });

    // --- SIDEBAR SCROLL PERSISTENCE ---
    const sidebarScrollArea = document.querySelector('#sidebar .overflow-y-auto');
    if (sidebarScrollArea) {
        // Restore scroll
        const savedScroll = localStorage.getItem('sidebarScrollPos');
        if (savedScroll) sidebarScrollArea.scrollTop = parseInt(savedScroll, 10);

        // Save scroll on change (debounced)
        let scrollTimeout;
        sidebarScrollArea.addEventListener('scroll', () => {
            clearTimeout(scrollTimeout);
            scrollTimeout = setTimeout(() => {
                localStorage.setItem('sidebarScrollPos', sidebarScrollArea.scrollTop);
            }, 100);
        });
    }

    // Initialize SPA navigation
    initializeSmoothNavigation();
});

// --- HELPER FUNCTIONS (GLOBAL) ---

window.setupDropdown = function (triggerId, dropdownId, labelId, optionClass) {
    const btn = document.getElementById(triggerId);
    const dropdown = document.getElementById(dropdownId);
    const labelEl = document.getElementById(labelId);
    const options = document.querySelectorAll(optionClass);

    if (!btn || !dropdown) return;

    btn.addEventListener('click', (e) => {
        e.stopPropagation();
        // Close others
        document.querySelectorAll('[id$="Dropdown"]').forEach(el => {
            if (el.id !== dropdownId) el.classList.add('invisible', 'opacity-0', '-translate-y-2');
        });

        const isHidden = dropdown.classList.contains('invisible');
        if (isHidden) {
            dropdown.classList.remove('invisible', 'opacity-0', '-translate-y-2');
            btn.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.add('rotate-180');
        } else {
            closeDropdown(dropdown, btn);
        }
    });

    options.forEach(opt => {
        opt.addEventListener('click', (e) => {
            e.stopPropagation();
            const value = opt.dataset.value;
            const label = opt.dataset.label;

            if (labelEl) labelEl.textContent = label;

            options.forEach(o => {
                const check = o.querySelector('.check-icon');
                if (o === opt) {
                    if (check) {
                        check.classList.remove('opacity-0');
                        check.classList.add('opacity-100');
                    }
                    o.classList.add('bg-gray-50', 'dark:bg-slate-700/50', 'active');
                } else {
                    if (check) {
                        check.classList.remove('opacity-100');
                        check.classList.add('opacity-0');
                    }
                    o.classList.remove('bg-gray-50', 'dark:bg-slate-700/50', 'active');
                }
            });

            closeDropdown(dropdown, btn);

            // Dashboard Callback
            if (triggerId === 'typeSelectBtn' && window.updateDashboardData) {
                window.updateDashboardData(value);
            }

            // Stores Table Callback
            if ((triggerId === 'typeFilterBtn' || triggerId === 'statusFilterBtn') && window.updateStoresTable) {
                const activeType = document.querySelector('#typeFilterDropdown .active')?.dataset.value || 'all';
                const activeStatus = document.querySelector('#statusFilterDropdown .active')?.dataset.value || 'active';
                window.updateStoresTable(activeType, activeStatus);
            }
        });
    });
};

window.closeDropdown = function (dropdown, btn) {
    dropdown.classList.add('invisible', 'opacity-0', '-translate-y-2');
    btn?.querySelector('ion-icon[name="chevron-down-outline"]')?.classList.remove('rotate-180');
};

// --- SPA NAVIGATION SYSTEM ---

function initializeSmoothNavigation() {
    document.body.addEventListener('click', (e) => {
        const link = e.target.closest('a');
        if (!link || !link.href) return;

        const url = new URL(link.href);
        // Only internal links, ignore target="_blank", fragments, and special handles
        if (url.origin !== window.location.origin) return;
        if (link.target && link.target !== '_self') return;
        if (link.hasAttribute('data-no-ajax') || link.href.includes('logout.php') || link.hasAttribute('download')) return;
        if (e.metaKey || e.ctrlKey || e.shiftKey || e.button !== 0) return;

        // Fragment check: if same page but different hash, let browser handle it
        if (url.pathname === window.location.pathname && url.search === window.location.search && url.hash !== window.location.hash) return;
        // Same URL (excluding hash), do nothing or reload? Let's just avoid re-fetching the same content
        if (url.pathname === window.location.pathname && url.search === window.location.search && !url.hash) {
            e.preventDefault();
            return;
        }

        e.preventDefault();
        loadPage(link.href, true);
    });

    window.addEventListener('popstate', () => {
        loadPage(window.location.href, false);
    });
}

async function loadPage(url, pushToHistory = true) {
    console.log(`--- SPA: Loading Page: ${url} ---`);
    const main = document.querySelector('main');
    if (!main) return;

    // Use a more robust loading state than just opacity
    main.classList.add('page-loading-active');
    main.style.opacity = '0.5';

    try {
        const response = await fetch(url);
        if (!response.ok) {
            console.error(`SPA: Fetch failed with status ${response.status}`);
            throw new Error('Network response was not ok');
        }

        const html = await response.text();
        const parser = new DOMParser();
        const doc = parser.parseFromString(html, 'text/html');

        const newMain = doc.querySelector('main');
        if (newMain) {
            main.innerHTML = newMain.innerHTML;
            document.title = doc.title;

            if (pushToHistory) {
                window.history.pushState({}, '', url);
            }

            // Re-run scripts
            await executePageScripts(doc);

            // Reset opacity immediately after content is ready
            main.style.opacity = '1';

            // Trigger Tailwind Re-scan (for Play CDN)
            if (window.tailwind) {
                window.tailwind.config = window.tailwind.config;
            }

            // Sync UI
            updateSidebarActiveStates(url);
            main.scrollTop = 0;
        }
    } catch (error) {
        console.error('SPA: Failed to load page:', error);
        window.location.href = url;
    } finally {
        // Fallback safety: ensure page is visible, interactive and unblurred
        main.classList.remove('page-loading-active');
        main.style.opacity = '1';
        main.style.pointerEvents = 'auto';
        document.body.classList.remove('overflow-hidden');

        // Remove any lingering blur from modals on persistent elements
        const persistents = [
            document.body,
            document.getElementById('sidebar'),
            document.getElementById('topbar'),
            main
        ];
        persistents.forEach(el => {
            if (el) {
                el.classList.remove('blur-sm', 'blur', 'transition-all', 'duration-300');
                el.style.filter = '';
                el.style.pointerEvents = 'auto';
            }
        });

        console.log(`--- SPA: Load Finished ---`);
    }
}

async function executePageScripts(doc) {
    const scripts = doc.querySelectorAll('script');
    const promises = [];

    scripts.forEach(oldScript => {
        const src = oldScript.getAttribute('src');
        // Core libraries are already loaded; skip them to avoid double listeners
        if (src && (src.includes('layout.js') || src.includes('apexcharts') || src.includes('tailwindcss') || src.includes('ionicons'))) return;

        const newScript = document.createElement('script');

        // Copy all attributes
        Array.from(oldScript.attributes).forEach(attr => {
            newScript.setAttribute(attr.name, attr.value);
        });

        if (src) {
            // Remove existing script with same src to avoid duplication and force re-execution
            const existing = document.head.querySelector(`script[src^="${src.split('?')[0]}"]`);
            if (existing) {
                document.head.removeChild(existing);
            }

            newScript.src = src;
            // Create a promise to wait for this specific script
            // Create a promise to wait for this specific script with a timeout safety
            const scriptTask = new Promise((resolve) => {
                newScript.onload = () => {
                    // After script loads, explicitly try to call global init if it exists
                    // Wrap in try-catch to prevent hanging the whole SPA if one script fails
                    try {
                        if (src.includes('store_form.js') && typeof window.initStoreFormPage === 'function') {
                            window.initStoreFormPage();
                        } else if (src.includes('dashboard.js') && typeof window.initDashboard === 'function') {
                            window.initDashboard();
                        } else if (src.includes('stores.js') && typeof window.initStoresPage === 'function') {
                            window.initStoresPage();
                        }
                    } catch (e) {
                        console.error(`Error during init for ${src}:`, e);
                    }
                    resolve();
                };
                newScript.onerror = () => {
                    console.error(`Failed to load script: ${src}`);
                    resolve();
                };
            });

            // Fallback: If script takes > 2s, proceed anyway
            const timeoutTask = new Promise(resolve =>
                setTimeout(() => {
                    console.warn(`SPA: Script load timed out for ${src}`);
                    resolve();
                }, 2000)
            );

            promises.push(Promise.race([scriptTask, timeoutTask]));
        } else {
            // For inline scripts, we MUST extract the text and set it
            newScript.textContent = oldScript.textContent;
        }

        // Append to head to trigger execution
        document.head.appendChild(newScript);

        // Cleanup immediately if it's an inline script (keeps DOM clean)
        if (!src) {
            document.head.removeChild(newScript);
        }
    });

    // Wait for all external scripts to finish loading
    if (promises.length > 0) {
        await Promise.all(promises);
    }

    // Fire a custom event to notify scripts that the content has been updated
    document.dispatchEvent(new CustomEvent('pageLoaded', { detail: { url: window.location.href } }));
}

function updateSidebarActiveStates(url) {
    let filename = 'index.php';
    try {
        const urlObj = new URL(url, window.location.origin);
        filename = urlObj.pathname.split('/').pop() || 'index.php';
    } catch (e) {
        filename = url.split('/').pop().split('?')[0] || 'index.php';
    }

    const sidebarLinks = document.querySelectorAll('#sidebar a');
    const sidebarButtons = document.querySelectorAll('#sidebar button');

    // Reset all to default state first
    sidebarLinks.forEach(l => {
        l.classList.remove('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
        l.classList.add('text-gray-600', 'dark:text-gray-400');
    });
    sidebarButtons.forEach(b => {
        if (!b.onclick || !b.onclick.toString().includes('toggleSubmenu')) return;
        b.classList.remove('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
        b.classList.add('text-gray-600', 'dark:text-gray-400');
    });

    // Help identify category for highlighting parent menu/submenus
    const isStoreRelated = ['stores.php', 'store_form.php', 'store_devices.php', 'store_health.php'].includes(filename);
    const isTxnRelated = ['transactions_live.php', 'transactions_search.php', 'transactions_penny.php', 'transactions_void.php', 'transactions_refund.php'].includes(filename);

    sidebarLinks.forEach(link => {
        const linkHref = link.getAttribute('href');
        if (!linkHref) return;

        const linkFile = linkHref.split('?')[0];
        let isActive = (linkFile === filename);


        if (isActive) {
            link.classList.add('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
            link.classList.remove('text-gray-600', 'dark:text-gray-400');

            // Handle submenus
            const submenu = link.closest('[id$="-submenu"]');
            if (submenu) {
                // Expand
                submenu.classList.remove('max-h-0');
                submenu.classList.add('max-h-96');
                const chevronId = submenu.id.replace('-submenu', '-chevron');
                document.getElementById(chevronId)?.classList.add('rotate-180');

                // Highlight parent button
                const parentBtn = document.querySelector(`button[onclick*="${submenu.id}"]`);
                if (parentBtn) {
                    parentBtn.classList.add('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
                    parentBtn.classList.remove('text-gray-600', 'dark:text-gray-400');
                }
            }
        }
    });

    // Special Pass: Ensure parent categories are highlighted/expanded if related page is active
    if (isStoreRelated) {
        const storesSubmenu = document.getElementById('stores-submenu');
        if (storesSubmenu) {
            storesSubmenu.classList.remove('max-h-0');
            storesSubmenu.classList.add('max-h-96');
            document.getElementById('stores-chevron')?.classList.add('rotate-180');
        }
        const storesBtn = document.querySelector('button[onclick*="stores-submenu"]');
        if (storesBtn) {
            storesBtn.classList.add('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
            storesBtn.classList.remove('text-gray-600', 'dark:text-gray-400');
        }
    }
    if (isTxnRelated) {
        const txnSubmenu = document.getElementById('transactions-submenu');
        if (txnSubmenu) {
            txnSubmenu.classList.remove('max-h-0');
            txnSubmenu.classList.add('max-h-96');
            document.getElementById('transactions-chevron')?.classList.add('rotate-180');
        }
        const txnBtn = document.querySelector('button[onclick*="transactions-submenu"]');
        if (txnBtn) {
            txnBtn.classList.add('bg-orange-50', 'dark:bg-primary/10', 'text-primary');
            txnBtn.classList.remove('text-gray-600', 'dark:text-gray-400');
        }
    }
}

// --- TOAST NOTIFICATION SYSTEM ---
window.showToast = function (message, type = 'info') {
    let container = document.getElementById('toast-container');
    if (!container) {
        container = document.createElement('div');
        container.id = 'toast-container';
        container.className = 'fixed top-24 right-4 z-[9999] flex flex-col gap-2 pointer-events-none';
        document.body.appendChild(container);
    }

    const toast = document.createElement('div');
    toast.className = `transform transition-all duration-300 ease-out translate-x-full opacity-0 pointer-events-auto flex items-center p-4 rounded-xl shadow-lg border backdrop-blur-md min-w-[300px] max-w-md`;

    let iconName = '';
    let themeClasses = '';

    switch (type) {
        case 'success':
            iconName = 'checkmark-circle';
            themeClasses = 'bg-green-50/95 border-green-200 text-green-800 dark:bg-green-900/90 dark:border-green-800 dark:text-green-100';
            break;
        case 'error':
            iconName = 'alert-circle';
            themeClasses = 'bg-red-50/95 border-red-200 text-red-800 dark:bg-red-900/90 dark:border-red-800 dark:text-red-100';
            break;
        case 'warning':
            iconName = 'warning';
            themeClasses = 'bg-amber-50/95 border-amber-200 text-amber-800 dark:bg-amber-900/90 dark:border-amber-800 dark:text-amber-100';
            break;
        default:
            iconName = 'information-circle';
            themeClasses = 'bg-white/95 border-gray-200 text-gray-800 dark:bg-slate-800/90 dark:border-gray-700 dark:text-gray-100';
    }

    toast.classList.add(...themeClasses.split(' '));

    toast.innerHTML = `
        <ion-icon name="${iconName}" class="text-2xl mr-3 shrink-0"></ion-icon>
        <span class="text-sm font-medium flex-1">${message}</span>
        <button onclick="this.parentElement.style.opacity='0'; setTimeout(()=>this.parentElement.remove(), 300)" class="ml-3 text-current opacity-60 hover:opacity-100 transition-opacity">
            <ion-icon name="close"></ion-icon>
        </button>
    `;

    container.appendChild(toast);
    setTimeout(() => toast.classList.remove('translate-x-full', 'opacity-0'), 50);
    setTimeout(() => {
        if (toast.parentElement) {
            toast.classList.add('opacity-0', 'translate-x-4');
            setTimeout(() => toast.remove(), 300);
        }
    }, 4000);
};
