class SingleDatePicker {
    constructor(id, defaultDate, options = {}) {
        this.id = id;
        this.date = new Date(defaultDate);
        this.options = options; // Support { onApply: function, showTime: boolean }
        this.container = document.getElementById(id + '_container');
        if (this.container) this.init();
    }

    init() {
        // Render basic trigger structure
        this.container.innerHTML = `
            <style>
                .no-scrollbar::-webkit-scrollbar { display: none; }
                .no-scrollbar { -ms-overflow-style: none; scrollbar-width: none; }
            </style>
            <button id="btn_${this.id}" type="button" class="w-full flex items-center justify-between bg-white dark:bg-slate-900 border border-gray-200 dark:border-gray-600 rounded-xl px-3 py-2 text-xs font-medium text-gray-700 dark:text-gray-200 hover:border-orange-500 hover:ring-2 hover:ring-orange-500/10 transition-all h-[38px]">
                <div class="flex items-center gap-2 overflow-hidden">
                    <ion-icon name="calendar" class="text-[#E65100] shrink-0"></ion-icon>
                    <span id="label_${this.id}" class="truncate">${this.formatDisplay(this.date)}</span>
                </div>
            </button>
            <div id="dropdown_${this.id}" class="absolute top-full right-0 mt-2 w-[320px] bg-white dark:bg-slate-800 rounded-2xl shadow-xl border border-gray-100 dark:border-gray-700 z-50 p-4 hidden">
                
                <!-- Header -->
                <div class="flex items-center justify-between mb-4">
                     <div class="flex items-center gap-2">
                         <span class="font-bold text-sm text-gray-800 dark:text-gray-200 cursor-pointer hover:bg-gray-50 dark:hover:bg-slate-700 rounded px-2 py-1 transition-colors" id="month_label_${this.id}"></span>
                         <span class="text-gray-300">|</span>
                         <span class="font-bold text-sm text-gray-800 dark:text-gray-200 cursor-pointer hover:bg-gray-50 dark:hover:bg-slate-700 rounded px-2 py-1 transition-colors" id="year_label_${this.id}"></span>
                     </div>
                     <div class="flex items-center gap-1">
                         <button id="prev_${this.id}" class="p-1.5 hover:bg-gray-100 dark:hover:bg-slate-700 rounded-lg text-gray-500 transition-colors"><ion-icon name="chevron-back" class="text-xs"></ion-icon></button>
                         <button id="next_${this.id}" class="p-1.5 hover:bg-gray-100 dark:hover:bg-slate-700 rounded-lg text-gray-500 transition-colors"><ion-icon name="chevron-forward" class="text-xs"></ion-icon></button>
                     </div>
                </div>

                <!-- Calendar Grid -->
                <div id="cal_${this.id}" class="grid grid-cols-7 gap-1 text-center text-[10px] mb-4 pb-4 border-b border-gray-100 dark:border-gray-700"></div>

                <!-- Time Section (Scrolling Columns) -->
                <div class="mb-4" id="time_section_${this.id}">
                    <div class="flex items-center justify-between mb-2 px-1">
                         <span class="text-[10px] font-bold text-gray-400 uppercase tracking-wider">Time Selection</span>
                         <div class="text-xs font-black text-[#E65100] bg-orange-50 dark:bg-orange-900/20 px-2 py-0.5 rounded" id="time_display_${this.id}">
                             ${this.date.getHours().toString().padStart(2, '0')}:${this.date.getMinutes().toString().padStart(2, '0')}
                         </div>
                    </div>
                    <div class="flex gap-2 h-32 relative bg-gray-50 dark:bg-slate-900 rounded-xl border border-gray-100 dark:border-gray-700 overflow-hidden">
                        <!-- Hours Column -->
                        <div class="flex-1 overflow-y-auto no-scrollbar snap-y snap-mandatory py-2 relative" id="hours_col_${this.id}">
                            <div class="text-[10px] font-bold text-gray-300 text-center mb-1 sticky top-0 bg-gray-50 dark:bg-slate-900 z-10 w-full pb-1 uppercase">Hour</div>
                            <div id="hours_list_${this.id}" class="space-y-1 px-1">
                                <!-- Populated by JS -->
                            </div>
                        </div>
                        
                        <!-- Separator -->
                        <div class="flex flex-col justify-center h-full pb-6">
                            <span class="text-gray-300 font-bold text-lg">:</span>
                        </div>

                        <!-- Minutes Column -->
                        <div class="flex-1 overflow-y-auto no-scrollbar snap-y snap-mandatory py-2 relative" id="mins_col_${this.id}">
                            <div class="text-[10px] font-bold text-gray-300 text-center mb-1 sticky top-0 bg-gray-50 dark:bg-slate-900 z-10 w-full pb-1 uppercase">Min</div>
                            <div id="mins_list_${this.id}" class="space-y-1 px-1">
                                <!-- Populated by JS -->
                            </div>
                        </div>
                    </div>
                </div>

                <!-- Footer Buttons -->
                <div class="flex flex-col gap-2">
                    <button id="today_${this.id}" class="w-full py-2 bg-gray-50 dark:bg-slate-700 hover:bg-gray-100 dark:hover:bg-slate-600 text-gray-600 dark:text-gray-300 text-xs font-bold rounded-xl transition-colors">Jump to Today</button>
                    <button id="apply_${this.id}" class="w-full py-2 bg-[#E65100] hover:bg-[#F57C00] text-white text-xs font-bold rounded-xl shadow-lg shadow-orange-500/20 transition-all transform hover:-translate-y-0.5">Apply Now</button>
                </div>
            </div>`;

        const btn = document.getElementById(`btn_${this.id}`), dd = document.getElementById(`dropdown_${this.id}`);
        this.viewDate = new Date(this.date);

        // Toggle Open
        btn.onclick = (e) => {
            e.stopPropagation();
            // Close others (assuming closeAllDropdowns is available globally, or handle locally)
            if (window.closeAllDropdowns) window.closeAllDropdowns(`dropdown_${this.id}`);

            const isHidden = dd.classList.contains('hidden');
            if (isHidden) { dd.classList.remove('hidden'); this.render(); this.scrollToTime(); }
            else { dd.classList.add('hidden'); }
        };

        // Navigation
        document.getElementById(`prev_${this.id}`).onclick = (e) => { e.stopPropagation(); this.viewDate.setMonth(this.viewDate.getMonth() - 1); this.render(); };
        document.getElementById(`next_${this.id}`).onclick = (e) => { e.stopPropagation(); this.viewDate.setMonth(this.viewDate.getMonth() + 1); this.render(); };

        // Buttons
        document.getElementById(`today_${this.id}`).onclick = (e) => {
            e.stopPropagation();
            const now = new Date();
            this.date = new Date(now);
            this.viewDate = new Date(now);
            this.render();
            this.scrollToTime();
        };

        document.getElementById(`apply_${this.id}`).onclick = (e) => {
            e.stopPropagation();
            this.updateLabel();
            dd.classList.add('hidden');
            if (this.options.onApply) this.options.onApply(this.date);
        };

        // Stop propagation inside dropdown
        dd.onclick = (e) => e.stopPropagation();

        // Initial Scroll
        this.render();
    }

    formatDisplay(d) { return d.toLocaleDateString('en-GB') + ' - ' + d.getHours().toString().padStart(2, '0') + ':' + d.getMinutes().toString().padStart(2, '0'); }
    updateLabel() { document.getElementById(`label_${this.id}`).innerText = this.formatDisplay(this.date); }

    scrollToTime() {
        const h = this.date.getHours();
        const m = this.date.getMinutes();

        const hEl = document.getElementById(`h_opt_${this.id}_${h}`);
        const mEl = document.getElementById(`m_opt_${this.id}_${m}`);

        if (hEl) hEl.scrollIntoView({ block: 'center', behavior: 'smooth' });
        if (mEl) mEl.scrollIntoView({ block: 'center', behavior: 'smooth' });
    }

    render() {
        const y = this.viewDate.getFullYear(), m = this.viewDate.getMonth();
        const mNames = ["January", "February", "March", "April", "May", "June", "July", "August", "September", "October", "November", "December"];

        document.getElementById(`month_label_${this.id}`).innerText = mNames[m];
        document.getElementById(`year_label_${this.id}`).innerText = y;

        // Calendar Grid
        const grid = document.getElementById(`cal_${this.id}`); grid.innerHTML = "";
        ['S', 'M', 'T', 'W', 'T', 'F', 'S'].forEach(d => grid.innerHTML += `<div class="text-gray-400 py-1 font-bold mb-1">${d}</div>`);

        const firstDay = new Date(y, m, 1).getDay(), days = new Date(y, m + 1, 0).getDate();

        for (let i = 0; i < firstDay; i++) grid.innerHTML += `<div></div>`;
        for (let i = 1; i <= days; i++) {
            const btn = document.createElement('button');
            btn.innerText = i;
            const isSelected = (i === this.date.getDate() && m === this.date.getMonth() && y === this.date.getFullYear());
            const isToday = (i === new Date().getDate() && m === new Date().getMonth() && y === new Date().getFullYear());

            let baseClass = "h-8 w-full rounded-lg flex items-center justify-center font-bold text-xs transition-colors ";
            if (isSelected) baseClass += "bg-[#E65100] text-white shadow-md shadow-orange-500/20";
            else if (isToday) baseClass += "bg-orange-50 dark:bg-orange-900/20 text-[#E65100]";
            else baseClass += "hover:bg-gray-100 dark:hover:bg-slate-700 text-gray-700 dark:text-gray-300";

            btn.className = baseClass;
            btn.onclick = (e) => {
                e.stopPropagation();
                this.date.setDate(i);
                this.date.setMonth(m);
                this.date.setFullYear(y);
                this.render();
            };
            grid.appendChild(btn);
        }

        // Time Columns Render
        this.renderTimeColumn('hours', 24);
        this.renderTimeColumn('mins', 60);

        // Update Time Display
        document.getElementById(`time_display_${this.id}`).innerText =
            this.date.getHours().toString().padStart(2, '0') + ':' + this.date.getMinutes().toString().padStart(2, '0');
    }

    renderTimeColumn(type, count) {
        const list = document.getElementById(`${type}_list_${this.id}`);
        if (!list) return;
        list.innerHTML = '';
        for (let i = 0; i < count; i++) {
            const val = i;
            const isSelected = (type === 'hours' ? this.date.getHours() : this.date.getMinutes()) === val;

            const el = document.createElement('div');
            el.id = `${type.charAt(0)}_opt_${this.id}_${val}`;
            el.className = `snap-center py-2 text-center text-xs font-bold cursor-pointer rounded-lg transition-all ${isSelected ? 'bg-[#E65100] text-white shadow-lg shadow-orange-500/20 scale-105' : 'text-gray-600 dark:text-gray-400 hover:bg-white dark:hover:bg-slate-800'}`;
            el.innerText = val.toString().padStart(2, '0');
            el.onclick = (e) => {
                e.stopPropagation();
                if (type === 'hours') this.date.setHours(val); else this.date.setMinutes(val);
                this.render();
            };
            list.appendChild(el);
        }
    }
}
