function initDashboard() {
    // Check if we are on the dashboard page
    if (!document.querySelector('#weeklyActivityChart')) return;

    // --- Global Chart Instances ---
    let weeklyChart = null;
    let comparisonChart = null;
    let monthlyChart = null;
    let breakdownChart = null;
    let currentSlide = 0;

    // --- Shared Options ---
    const commonOptions = {
        chart: {
            fontFamily: 'Inter, sans-serif',
            toolbar: { show: false },
            zoom: { enabled: false },
            background: 'transparent'
        },
        dataLabels: { enabled: false },
        stroke: { curve: 'smooth', width: 2 },
        theme: { mode: 'light' },
        grid: {
            borderColor: 'rgba(128,128,128, 0.2)',
            strokeDashArray: 5,
        }
    };

    // --- STATIC DATA DEFINITIONS (Carousel) ---
    const slide1 = {
        series: [
            { name: 'Cash', data: [480, 350, 320, 480, 150, 390, 400] },
            { name: 'Card', data: [240, 120, 260, 360, 240, 240, 340] }
        ],
        colors: ['#FF9909', '#FCD34D'],
        title: 'Cash and Card Comparison',
        type: 'bar'
    };
    const slide2 = {
        series: [{ name: 'Grand Total', data: [490, 280, 350, 200, 380, 460, 350] }],
        colors: ['#FF9909'],
        title: 'Sales Comparison',
        type: 'bar'
    };
    const slide3 = {
        series: [
            { name: 'Vapes', data: [450, 450, 450, 450, 450, 450, 450] },
            { name: 'Alcohols', data: [220, 220, 220, 220, 220, 220, 220] },
            { name: 'Tobacco', data: [450, 450, 450, 450, 450, 450, 450] }
        ],
        colors: ['#FCD34D', '#FF9909', '#3B82F6'],
        title: 'Vapes, Alcohol & Tobacco Sales',
        type: 'bar'
    };
    const slides = [slide1, slide2, slide3];


    // --- 1. Weekly Activity Carousel ---
    function renderWeeklyChart() {
        if (!slides[currentSlide]) currentSlide = 0;
        const config = slides[currentSlide];
        const isDark = document.documentElement.classList.contains('dark');
        const textColor = isDark ? '#e2e8f0' : '#475569';

        const options = {
            ...commonOptions,
            series: config.series,
            chart: {
                type: config.type,
                height: 350,
                parentHeightOffset: 0,
                toolbar: { show: false },
                background: 'transparent'
            },
            colors: config.colors,
            plotOptions: {
                bar: { horizontal: false, columnWidth: '55%', borderRadius: 4 },
            },
            xaxis: {
                categories: ['Sat', 'Sun', 'Mon', 'Tue', 'Wed', 'Thu', 'Fri'],
                axisBorder: { show: false },
                axisTicks: { show: false },
                labels: { style: { colors: textColor } }
            },
            yaxis: { labels: { style: { colors: textColor } } },
            legend: { position: 'bottom', labels: { colors: textColor } },
            tooltip: { theme: isDark ? 'dark' : 'light' }
        };

        const titleEl = document.getElementById('weeklyChartTitle');
        if (titleEl) titleEl.innerText = config.title;

        const el = document.querySelector("#weeklyActivityChart");
        if (el) {
            if (weeklyChart) weeklyChart.destroy();
            weeklyChart = new ApexCharts(el, options);
            weeklyChart.render();
        }
    }

    // Button Handlers
    window.nextWeeklyChart = function () {
        currentSlide = (currentSlide + 1) % slides.length;
        renderWeeklyChart();
    }
    window.prevWeeklyChart = function () {
        currentSlide = (currentSlide - 1 + slides.length) % slides.length;
        renderWeeklyChart();
    }


    // --- 2. Comparison Chart ---
    function renderComparisonChart() {
        const el = document.querySelector("#comparisonChart");
        if (el) {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#e2e8f0' : '#475569';
            const options = {
                ...commonOptions,
                series: [
                    { name: 'Payouts', data: [50, 30] },
                    { name: 'Refunds', data: [40, 20] },
                    { name: 'Discounts', data: [350, 250] }
                ],
                chart: { type: 'bar', height: 320, stacked: true, toolbar: { show: false }, background: 'transparent' },
                colors: ['#ef4444', '#3b82f6', '#FF9909'],
                xaxis: {
                    categories: ['02/11/2025', '09/11/2025'],
                    labels: { style: { colors: textColor } }
                },
                yaxis: { labels: { style: { colors: textColor } } },
                legend: { labels: { colors: textColor } },
                tooltip: { theme: isDark ? 'dark' : 'light' },
                plotOptions: { bar: { columnWidth: '40%', borderRadius: 4 } }
            };

            if (comparisonChart) comparisonChart.destroy();
            comparisonChart = new ApexCharts(el, options);
            comparisonChart.render();
        }
    }

    // --- 3. Monthly Sales ---
    function renderMonthlyChart() {
        const el = document.querySelector("#monthlySalesChart");
        if (el) {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#e2e8f0' : '#475569';
            const options = {
                ...commonOptions,
                series: [{ name: 'Sales', data: [150, 300, 240, 600, 220, 450, 250] }],
                chart: { type: 'area', height: 320, toolbar: { show: false }, background: 'transparent' },
                colors: ['#FF9909'],
                fill: {
                    type: 'gradient',
                    gradient: { shadeIntensity: 1, opacityFrom: 0.7, opacityTo: 0.1, stops: [0, 90, 100] }
                },
                xaxis: {
                    categories: ['Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec', 'Jan'],
                    labels: { style: { colors: textColor } }
                },
                yaxis: { labels: { style: { colors: textColor } } },
                dataLabels: { enabled: false },
                tooltip: { theme: isDark ? 'dark' : 'light' }
            };

            if (monthlyChart) monthlyChart.destroy();
            monthlyChart = new ApexCharts(el, options);
            monthlyChart.render();
        }
    }

    // --- 4. Breakdown Chart ---
    function renderBreakdownChart() {
        const el = document.querySelector("#salesBreakdownChart");
        if (el) {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#e2e8f0' : '#475569';
            const options = {
                series: [30, 15, 25, 30],
                labels: ['Payouts', 'Refund', 'Discounts', 'Subtotal'],
                chart: { type: 'donut', height: 320, background: 'transparent' },
                colors: ['#ef4444', '#64748b', '#1e293b', '#FF9909'],
                legend: { position: 'bottom', labels: { colors: textColor } },
                dataLabels: { enabled: false },
                stroke: { show: false },
                plotOptions: {
                    pie: {
                        donut: {
                            size: '70%',
                            labels: {
                                show: true,
                                name: { color: textColor },
                                value: { color: textColor },
                                total: {
                                    show: true,
                                    label: 'Total',
                                    color: textColor,
                                    formatter: (w) => w.globals.seriesTotals.reduce((a, b) => a + b, 0)
                                }
                            }
                        }
                    }
                }
            };

            if (breakdownChart) breakdownChart.destroy();
            breakdownChart = new ApexCharts(el, options);
            breakdownChart.render();
        }
    }

    // --- 5. Top Dishes (Restaurant) ---
    function renderTopDishesChart(seriesData) {
        const el = document.querySelector("#topDishesChart");
        if (el && seriesData) {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#e2e8f0' : '#475569';
            const options = {
                ...commonOptions,
                series: [{ name: 'Orders', data: seriesData.data }],
                chart: { type: 'bar', height: 320, toolbar: { show: false }, background: 'transparent' },
                colors: ['#FF9909'],
                plotOptions: {
                    bar: { horizontal: true, borderRadius: 4, barHeight: '60%' }
                },
                xaxis: {
                    categories: seriesData.categories,
                    labels: { style: { colors: textColor } }
                },
                yaxis: { labels: { style: { colors: textColor } } },
                tooltip: { theme: isDark ? 'dark' : 'light' }
            };

            el.innerHTML = '';
            const chart = new ApexCharts(el, options);
            chart.render();
        }
    }

    // --- 6. Peak Hours (Restaurant) ---
    function renderPeakHoursChart(seriesData) {
        const el = document.querySelector("#peakHoursChart");
        if (el && seriesData) {
            const isDark = document.documentElement.classList.contains('dark');
            const textColor = isDark ? '#e2e8f0' : '#475569';
            const options = {
                ...commonOptions,
                series: [{ name: 'Visitors', data: seriesData.data }],
                chart: { type: 'area', height: 220, toolbar: { show: false }, background: 'transparent' },
                colors: ['#3B82F6'],
                fill: {
                    type: 'gradient',
                    gradient: { shadeIntensity: 1, opacityFrom: 0.7, opacityTo: 0.1, stops: [0, 90, 100] }
                },
                xaxis: {
                    categories: seriesData.categories,
                    labels: { style: { colors: textColor }, rotate: -45, rotateAlways: false }
                },
                yaxis: { show: false },
                dataLabels: { enabled: false },
                tooltip: { theme: isDark ? 'dark' : 'light' }
            };

            el.innerHTML = '';
            const chart = new ApexCharts(el, options);
            chart.render();
        }
    }

    // --- 7. Grocery Rendering Functions ---
    function renderGroceryWidgets(data) {
        const tbody = document.getElementById('lowStockBody');
        if (tbody && data.lowStock) {
            tbody.innerHTML = data.lowStock.map(item => `
                <tr class="border-b border-gray-50 dark:border-gray-800 last:border-none hover:bg-gray-50 dark:hover:bg-slate-800/50 transition-colors">
                    <td class="p-3">
                        <div class="font-medium text-gray-800 dark:text-gray-200 text-sm">${item.name}</div>
                        <div class="text-xs text-gray-400">${item.category}</div>
                    </td>
                    <td class="p-3 text-center text-sm font-bold text-gray-700 dark:text-gray-300">
                        ${item.stock} <span class="text-xs text-gray-400 font-normal">${item.unit}</span>
                    </td>
                    <td class="p-3 text-center">
                        <span class="inline-flex items-center px-2 py-0.5 rounded text-xs font-medium ${item.status === 'Critical' ? 'bg-red-100 text-red-600 dark:bg-red-900/20 dark:text-red-400' : 'bg-orange-100 text-orange-600 dark:bg-orange-900/20 dark:text-orange-400'}">
                            ${item.status}
                        </span>
                    </td>
                </tr>
            `).join('');
        }

        const list = document.getElementById('expiringList');
        if (list && data.expiring) {
            list.innerHTML = data.expiring.map(item => `
                <div class="flex items-center gap-3 p-3 rounded-xl bg-gray-50 dark:bg-slate-800/50 border border-gray-100 dark:border-gray-700">
                    <div class="w-10 h-10 rounded-lg bg-red-50 dark:bg-red-900/20 flex items-center justify-center text-red-500 font-bold text-xs shrink-0 flex-col leading-tight">
                        <span>${item.dateDay}</span>
                        <span class="text-[8px] uppercase">${item.dateMonth}</span>
                    </div>
                    <div class="flex-1 min-w-0">
                        <div class="font-medium text-gray-800 dark:text-gray-200 text-sm truncate">${item.name}</div>
                        <div class="text-xs text-red-500 font-medium">${item.daysLeft} days left</div>
                    </div>
                    <button class="text-gray-400 hover:text-red-500 transition-colors"><ion-icon name="trash-outline"></ion-icon></button>
                </div>
            `).join('');
        }
    }


    // --- MOCK DATA ---
    const mockData = {
        grocery: {
            weeklySlides: [
                {
                    series: [
                        { name: 'Cash', data: [480, 350, 320, 480, 150, 390, 400] },
                        { name: 'Card', data: [240, 120, 260, 360, 240, 240, 340] }
                    ],
                    colors: ['#FF9909', '#FCD34D'],
                    title: 'Cash vs Card Sales',
                    type: 'bar'
                },
                {
                    series: [{ name: 'Grand Total', data: [490, 280, 350, 200, 380, 460, 350] }],
                    colors: ['#FF9909'],
                    title: 'Sales Comparison',
                    type: 'bar'
                },
                {
                    series: [
                        { name: 'Vapes', data: [450, 450, 450, 450, 450, 450, 450] },
                        { name: 'Alcohols', data: [220, 220, 220, 220, 220, 220, 220] },
                        { name: 'Tobacco', data: [450, 450, 450, 450, 450, 450, 450] }
                    ],
                    colors: ['#FCD34D', '#FF9909', '#3B82F6'],
                    title: 'Vapes, Alcohol & Tobacco',
                    type: 'bar'
                }
            ],
            comparison: [
                { name: 'Payouts', data: [50, 30] },
                { name: 'Refunds', data: [40, 20] },
                { name: 'Discounts', data: [350, 250] }
            ],
            monthly: [{ name: 'Sales', data: [150, 300, 240, 600, 220, 450, 250] }],
            breakdown: [30, 15, 25, 30],
            cards: [
                { label: 'Walkins', value: '250', sub: '£5000.02', color: 'indigo' },
                { label: 'Payouts', value: '21', sub: '£3500.54', color: 'orange' },
                { label: 'Penny Sale', value: '5', sub: '£9.10', color: 'green' },
                { label: 'TXN Voids', value: '8', sub: '£250.78', color: 'red' }
            ],
            lowStock: [
                { name: 'Milk (2L)', category: 'Dairy', stock: 4, unit: 'btl', status: 'Critical' },
                { name: 'Bread (White)', category: 'Bakery', stock: 8, unit: 'loaf', status: 'Low' },
                { name: 'Eggs (Dozen)', category: 'Dairy', stock: 5, unit: 'box', status: 'Low' },
                { name: 'Bananas', category: 'Produce', stock: 12, unit: 'kg', status: 'Low' },
            ],
            expiring: [
                { name: 'Yogurt (Strawberry)', dateDay: '18', dateMonth: 'Jan', daysLeft: 2 },
                { name: 'Chicken Breast', dateDay: '19', dateMonth: 'Jan', daysLeft: 3 },
                { name: 'Fresh Salad Bag', dateDay: '17', dateMonth: 'Jan', daysLeft: 1 },
            ]
        },
        restaurant: {
            weeklySlides: [
                {
                    series: [
                        { name: 'Food', data: [800, 600, 750, 500, 450, 900, 1100] },
                        { name: 'Beverage', data: [400, 300, 350, 200, 150, 500, 600] }
                    ],
                    colors: ['#ef4444', '#3b82f6'],
                    title: 'Food vs Beverage Sales',
                    type: 'bar'
                },
                {
                    series: [{ name: 'Total Covers', data: [120, 90, 100, 80, 70, 150, 180] }],
                    colors: ['#FF9909'],
                    title: 'Weekly Covers (Guests)',
                    type: 'bar'
                },
                {
                    series: [
                        { name: 'Starters', data: [150, 120, 140, 100, 80, 200, 250] },
                        { name: 'Mains', data: [400, 350, 380, 300, 250, 500, 600] },
                        { name: 'Desserts', data: [100, 80, 90, 60, 50, 150, 180] }
                    ],
                    colors: ['#FCD34D', '#FF9909', '#10B981'],
                    title: 'Course Breakdown Sales',
                    type: 'bar'
                }
            ],
            comparison: [
                { name: 'Dine-In', data: [8500, 9200] },
                { name: 'Takeaway', data: [3200, 2800] },
                { name: 'Delivery', data: [4500, 5100] }
            ],
            monthly: [{ name: 'Revenue', data: [12000, 14500, 13200, 16000, 15500, 18000, 12000] }],
            breakdown: [60, 25, 15, 0],
            cards: [
                { label: 'Total Orders', value: '145', sub: '£3,450.00', color: 'blue' },
                { label: 'Avg Spend', value: '£42.50', sub: 'Per Head', color: 'green' },
                { label: 'Tips', value: '£250.00', sub: 'Today', color: 'yellow' },
                { label: 'Wastage', value: '3', sub: '£45.00', color: 'red' }
            ],
            topDishes: {
                categories: ['Burger', 'Pasta', 'Steak', 'Pizza', 'Salad'],
                data: [120, 95, 80, 150, 60]
            },
            peakHours: {
                categories: ['12pm', '1pm', '2pm', '5pm', '6pm', '7pm', '8pm', '9pm'],
                data: [40, 65, 30, 50, 80, 95, 120, 85]
            },
            lowStock: [
                { name: 'House Wine (Red)', category: 'Bar', stock: 3, unit: 'btl', status: 'Critical' },
                { name: 'Napkins (Box)', category: 'Supplies', stock: 5, unit: 'box', status: 'Low' },
                { name: 'Steak (Ribeye)', category: 'Kitchen', stock: 12, unit: 'pcs', status: 'Low' },
            ],
            expiring: [
                { name: 'Heavy Cream', dateDay: '18', dateMonth: 'Jan', daysLeft: 2 },
                { name: 'Fresh Basil', dateDay: '16', dateMonth: 'Jan', daysLeft: 0 },
            ]
        }
    };

    // --- GLOBAL UPDATE FUNCTION ---
    window.updateDashboardData = function (type) {
        const data = mockData[type] || mockData['grocery'];
        const restaurantDash = document.getElementById('restaurant-dashboard');

        if (type === 'restaurant') {
            if (restaurantDash) restaurantDash.classList.remove('hidden');
            renderTopDishesChart(data.topDishes);
            renderPeakHoursChart(data.peakHours);
        } else {
            if (restaurantDash) restaurantDash.classList.add('hidden');
        }

        renderGroceryWidgets(data);

        if (data.weeklySlides) {
            slides[0] = data.weeklySlides[0];
            slides[1] = data.weeklySlides[1];
            slides[2] = data.weeklySlides[2];
            currentSlide = 0;
            renderWeeklyChart();
        }

        if (comparisonChart && data.comparison) comparisonChart.updateSeries(data.comparison);
        if (monthlyChart && data.monthly) monthlyChart.updateSeries(data.monthly);
        if (breakdownChart && data.breakdown) {
            breakdownChart.updateSeries(data.breakdown);
            breakdownChart.updateOptions({ labels: type === 'restaurant' ? ['Card', 'Cash', 'Online', 'Other'] : ['Payouts', 'Refund', 'Discounts', 'Subtotal'] });
        }

        const visibleCards = document.querySelectorAll('.grid > .glass-panel');
        data.cards.forEach((cardData, index) => {
            if (visibleCards[index]) {
                const cardEl = visibleCards[index];
                const labelEl = cardEl.querySelector('.text-sm.font-bold.uppercase');
                if (labelEl) labelEl.textContent = cardData.label;
                const valueEl = cardEl.querySelector('.text-2xl');
                if (valueEl) valueEl.textContent = cardData.value;
                const subEl = cardEl.querySelector('.text-lg.font-semibold');
                if (subEl) subEl.textContent = cardData.sub;
            }
        });
    };

    // --- INITIAL RENDER ---
    // Wrap in a timeout to ensure DOM layout is complete before ApexCharts measures size
    setTimeout(() => {
        requestAnimationFrame(() => {
            renderWeeklyChart();
            renderComparisonChart();
            renderMonthlyChart();
            renderBreakdownChart();

            // Re-trigger defaults if URL params exist
            const urlParams = new URLSearchParams(window.location.search);
            const scope = urlParams.get('scope') || 'grocery';
            window.updateDashboardData(scope);
        });
    }, 100);
}

// Support both initial load and SPA transitions
document.addEventListener('DOMContentLoaded', initDashboard);

if (!window.dashboard_initialized) {
    document.addEventListener('pageLoaded', (e) => {
        if (e.detail.url.includes('index.php') || e.detail.url.endsWith('/')) {
            initDashboard();
        }
    });
    window.dashboard_initialized = true;
}

// Theme toggle refresh
document.getElementById('themeToggle')?.addEventListener('click', () => {
    setTimeout(() => location.reload(), 100);
});
