function initAdminDashboard() {
    // Check if we are on the admin dashboard page
    if (!document.querySelector('#revenueTrendChart')) return;

    // --- Shared Options ---
    const commonOptions = {
        chart: {
            fontFamily: 'Inter, sans-serif',
            toolbar: { show: false },
            zoom: { enabled: false },
            background: 'transparent'
        },
        dataLabels: { enabled: false },
        stroke: { curve: 'smooth', width: 2 },
        theme: { mode: 'light' },
        grid: {
            borderColor: 'rgba(128,128,128, 0.1)',
            strokeDashArray: 5,
        }
    };

    const isDark = document.documentElement.classList.contains('dark');
    const textColor = isDark ? '#e2e8f0' : '#475569';

    // --- 1. Total Revenue Trend (Area Chart) ---
    function renderRevenueTrendChart() {
        const el = document.querySelector("#revenueTrendChart");
        if (el) {
            const options = {
                ...commonOptions,
                series: [{ 
                    name: 'Total Revenue', 
                    data: [12500, 15000, 14200, 18500, 21000, 19500, 24000] 
                }],
                chart: { 
                    type: 'area', 
                    height: 350, 
                    toolbar: { show: false }, 
                    background: 'transparent' 
                },
                colors: ['#FF9909'], // Primary Orange
                fill: {
                    type: 'gradient',
                    gradient: { 
                        shadeIntensity: 1, 
                        opacityFrom: 0.6, 
                        opacityTo: 0.05, 
                        stops: [0, 90, 100] 
                    }
                },
                xaxis: {
                    categories: ['Mon', 'Tue', 'Wed', 'Thu', 'Fri', 'Sat', 'Sun'],
                    labels: { style: { colors: textColor } },
                    axisBorder: { show: false },
                    axisTicks: { show: false }
                },
                yaxis: { 
                    labels: { 
                        style: { colors: textColor },
                        formatter: (value) => { return '£' + (value / 1000).toFixed(0) + 'k' }
                    } 
                },
                tooltip: { theme: isDark ? 'dark' : 'light' }
            };

            new ApexCharts(el, options).render();
        }
    }

    // --- 2. System Health (Radial Bar / Donut) ---
    function renderSystemHealthChart() {
        const el = document.querySelector("#systemHealthChart");
        if (el) {
            const options = {
                series: [42, 65, 28], // CPU, RAM, DB
                labels: ['CPU Load', 'RAM Usage', 'DB IOPS'],
                chart: { 
                    type: 'radialBar', 
                    height: 320, 
                    background: 'transparent' 
                },
                colors: ['#3B82F6', '#10B981', '#F59E0B'],
                plotOptions: {
                    radialBar: {
                        hollow: { size: '55%' },
                        track: { 
                            background: isDark ? '#334155' : '#f1f5f9',
                        },
                        dataLabels: {
                            name: { fontSize: '14px', color: textColor },
                            value: { fontSize: '16px', color: textColor, fontWeight: 'bold' },
                            total: {
                                show: true,
                                label: 'System',
                                color: textColor,
                                formatter: function (w) { return 'Good' }
                            }
                        }
                    }
                },
                stroke: { lineCap: 'round' }
            };

            new ApexCharts(el, options).render();
        }
    }

    // --- 3. OpEx vs Revenue (Bar Chart) ---
    function renderOpexChart() {
        const el = document.querySelector("#opexComparisonChart");
        if (el) {
            const options = {
                ...commonOptions,
                series: [
                    { name: 'Revenue', data: [85, 92, 105, 98, 115, 120] },
                    { name: 'OpEx', data: [25, 28, 30, 29, 35, 32] }
                ],
                chart: { 
                    type: 'bar', 
                    height: 320, 
                    toolbar: { show: false }, 
                    background: 'transparent' 
                },
                colors: ['#10B981', '#EF4444'], // Green vs Red
                plotOptions: {
                    bar: { 
                        horizontal: false, 
                        columnWidth: '45%', 
                        borderRadius: 4 
                    },
                },
                dataLabels: { enabled: false },
                xaxis: {
                    categories: ['Aug', 'Sep', 'Oct', 'Nov', 'Dec', 'Jan'],
                    labels: { style: { colors: textColor } },
                    axisBorder: { show: false },
                    axisTicks: { show: false }
                },
                yaxis: { 
                    labels: { 
                        style: { colors: textColor },
                        formatter: (value) => { return '£' + value + 'k' }
                    } 
                },
                legend: { position: 'top', labels: { colors: textColor } },
                tooltip: { theme: isDark ? 'dark' : 'light' }
            };

            new ApexCharts(el, options).render();
        }
    }

    // Render All
    requestAnimationFrame(() => {
        renderRevenueTrendChart();
        renderSystemHealthChart();
        renderOpexChart();
    });
}

// Support both initial load and SPA transitions
document.addEventListener('DOMContentLoaded', initAdminDashboard);

if (!window.admin_dashboard_initialized) {
    document.addEventListener('pageLoaded', (e) => {
        if (e.detail.url.includes('admin_dashboard.php')) {
            initAdminDashboard();
        }
    });
    window.admin_dashboard_initialized = true;
}

// Theme toggle refresh
document.getElementById('themeToggle')?.addEventListener('click', () => {
    setTimeout(() => location.reload(), 100);
});
