<?php
// index.php

// 1) Suppress deprecated/notice errors and start buffering immediately
error_reporting(E_ALL & ~E_DEPRECATED & ~E_NOTICE);
ob_start();

// 2) Include FPDF and set up your DB connection
require('fpdf/fpdf.php');

$host     = "localhost";
$user     = "root";
$password = "";
$database = "bookerlistdb";
$conn = new mysqli($host, $user, $password, $database);
if ($conn->connect_error) {
    die("DB Connection failed: " . $conn->connect_error);
}

// 3) POST → Save data & output PDF, then exit before any HTML is sent
if ($_SERVER['REQUEST_METHOD'] === 'POST' && isset($_POST['store_id'])) {
    $store_id = (int)$_POST['store_id'];
    $posted   = $_POST['products'] ?? [];
    $now      = date('Y-m-d H:i:s');

    // Upsert each item (assign to variables so bind_param gets references)
    $upsert = $conn->prepare("
      INSERT INTO store_products (store_id, product_id, quantity, created_date)
      VALUES (?, ?, ?, ?)
      ON DUPLICATE KEY UPDATE quantity = VALUES(quantity)
    ");
    foreach ($posted as $p) {
        $pid  = (int)$p['id'];
        $qty  = (int)$p['qty'];
        $date = $now;
        $upsert->bind_param("iiis", $store_id, $pid, $qty, $date);
        $upsert->execute();
    }
    $upsert->close();

    // Fetch the store name
    $sQ = $conn->prepare("SELECT store_name FROM stores WHERE store_ID = ?");
    $sQ->bind_param("i", $store_id);
    $sQ->execute();
    $store = $sQ->get_result()->fetch_assoc() ?: [];
    $sQ->close();
    $store_name = $store['store_name'] ?? "Store #{$store_id}";

    // Clean out all buffered output
    ob_end_clean();

    // Generate the PDF
    $pdf = new FPDF('P','mm','A4');
    $pdf->AddPage();

    // Title and metadata
    $pdf->SetFont('Arial','B',16);
    $pdf->Cell(0,10, iconv('UTF-8','ISO-8859-1//TRANSLIT',$store_name), 0,1,'C');
    $pdf->SetFont('Arial','',12);
    $pdf->Cell(0,7, "Created: {$now}", 0,1,'C');
    $pdf->Ln(4);

    // Gather items
    $items = [];
    foreach ($posted as $p) {
        $pid = (int)$p['id'];
        $q = $conn->prepare("
          SELECT product_name, rrp, category
          FROM products
          WHERE product_id = ?
        ");
        $q->bind_param("i", $pid);
        $q->execute();
        $row = $q->get_result()->fetch_assoc() ?: [];
        $q->close();

        // Normalize RRP
        $num   = floatval(preg_replace('/[^\d\.]/','',$row['rrp'] ?? '0'));
        $price = '£'.number_format($num,2);

        $items[] = [
          'name'     => $row['product_name'] ?? '',
          'price'    => $price,
          'category' => $row['category'] ?? 'Uncategorized',
          'qty'      => (int)$p['qty'],
        ];
    }

    // Group by category and print
    $groups = [];
    foreach ($items as $it) {
        $groups[$it['category']][] = $it;
    }

    foreach ($groups as $cat => $rows) {
        $pdf->SetFont('Arial','B',14);
        $pdf->Cell(0,8, iconv('UTF-8','ISO-8859-1//TRANSLIT',$cat), 0,1);
        $pdf->Ln(1);

        $pdf->SetFont('Arial','',12);
        foreach ($rows as $r) {
            $line = "{$r['name']} - {$r['price']} - {$r['qty']}";
            $pdf->Cell(0,7, iconv('UTF-8','ISO-8859-1//TRANSLIT',$line), 0,1);
        }
        $pdf->Ln(4);
    }

    // Send PDF inline
    $pdf->Output('I', "store_{$store_id}_" . date('Ymd_His') . ".pdf");
    exit;
}

// 4) On GET: flush the buffer so we can render the page HTML
ob_end_flush();

// 5) Include header and set page title
$page_title = 'Store Product Entry';
include 'header.php';

// Fetch data for the interactive page
$stores   = $conn->query("SELECT * FROM stores");
$products = $conn->query("SELECT * FROM products");
// fetch distinct categories for the dropdown
$catRes = $conn->query("SELECT DISTINCT category FROM products");
$categories = [];
while ($c = $catRes->fetch_assoc()) {
    $categories[] = $c['category'];
}

?>
<!DOCTYPE html>
<html lang="en">
<head>
  <meta charset="UTF-8">
  <meta name="viewport" content="width=device-width,initial-scale=1">
  <title><?= htmlspecialchars($page_title) ?></title>
  <link href="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/css/bootstrap.min.css" rel="stylesheet">
  <style>
    #suggestions { max-height:200px; overflow-y:auto; }
    #suggestions div { padding:5px; cursor:pointer; }
    #suggestions div:hover { background:#f0f0f0; }
    td.qty-cell { cursor: pointer; }
    td.qty-cell:hover { background-color: #eef; }
  </style>
</head>
<body class="bg-light p-4">
  <div class="container">
    <h2 class="text-center mb-4"><?= htmlspecialchars($page_title) ?></h2>

    <div class="row g-3 mb-3">
      <div class="col-md-3">
        <select id="store_select" class="form-select">
          <option value="">Select Store</option>
          <?php while($s=$stores->fetch_assoc()): ?>
            <option value="<?= (int)$s['store_ID'] ?>">
              <?= htmlspecialchars($s['store_name']) ?> — <?= htmlspecialchars($s['location']) ?>
            </option>
          <?php endwhile; ?>
        </select>
      </div>
      <div class="col-md-3 position-relative">
        <input id="product_input" class="form-control" placeholder="Search product…" autocomplete="off">
        <div id="suggestions" class="position-absolute bg-white border rounded w-100" style="z-index:99;"></div>
      </div>
      <div class="col-md-2"><input id="price" class="form-control" placeholder="Price" readonly></div>
      <div class="col-md-2"><input id="category" class="form-control" placeholder="Category" readonly></div>
      <div class="col-md-2"><input id="qty" type="number" class="form-control" placeholder="Qty"></div>
      <div class="col-12 text-end">
        <button class="btn btn-primary" onclick="newProduct()">+ New Product</button>
        <button class="btn btn-success" onclick="addProduct()">+ Add</button>
      </div>
    </div>

    <table class="table table-bordered">
      <tbody id="product_grid"></tbody>
    </table>

    <button class="btn btn-primary" onclick="submitForm()">Save &amp; Download PDF</button>
  </div>

  <!-- Alert Modal -->
  <div class="modal fade" id="alertModal" tabindex="-1" aria-hidden="true">
    <div class="modal-dialog"><div class="modal-content">
      <div class="modal-header">
        <h5 id="alertModalLabel" class="modal-title">Notice</h5>
        <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
      </div>
      <div id="alertModalBody" class="modal-body"></div>
      <div id="alertModalFooter" class="modal-footer">
        <button type="button" class="btn btn-primary" data-bs-dismiss="modal">OK</button>
      </div>
    </div></div>
  </div>

  <!-- New Product Modal -->
<div class="modal fade" id="newProductModal" tabindex="-1" aria-hidden="true">
  <div class="modal-dialog"><div class="modal-content">
    <div class="modal-header">
      <h5 class="modal-title">Add New Product</h5>
      <button type="button" class="btn-close" data-bs-dismiss="modal"></button>
    </div>
    <form id="newProductForm">
      <div class="modal-body">
        <div class="mb-3">
          <label for="new_product_code" class="form-label">Product Code</label>
          <input type="text" class="form-control" id="new_product_code" name="product_code" required>
        </div>
        <div class="mb-3">
          <label for="new_product_name" class="form-label">Product Name</label>
          <input type="text" class="form-control" id="new_product_name" name="product_name" required>
        </div>
        <div class="mb-3">
          <label for="new_rrp" class="form-label">RRP</label>
          <input type="text" class="form-control" id="new_rrp" name="rrp" required>
        </div>
        <div class="mb-3">
          <label for="new_category" class="form-label">Category</label>
          <input list="categoryList" class="form-control" id="new_category" name="category"
                 placeholder="Type or select to search" required>
          <datalist id="categoryList">
            <?php foreach ($categories as $cat): ?>
              <option value="<?= htmlspecialchars($cat) ?>">
            <?php endforeach; ?>
          </datalist>
        </div>
      </div>
      <div class="modal-footer">
        <button type="button" class="btn btn-secondary" data-bs-dismiss="modal">Cancel</button>
        <button type="submit" class="btn btn-primary">Save Product</button>
      </div>
    </form>
  </div></div>
</div>


  <script src="https://cdn.jsdelivr.net/npm/bootstrap@5.3.2/dist/js/bootstrap.bundle.min.js"></script>
 <script>
"use strict";

/* =========================
   Helpers (sanitization)
========================= */
function escapeHTML(str){
  return String(str).replace(/[&<>"']/g, s => (
    { "&":"&amp;", "<":"&lt;", ">":"&gt;", '"':"&quot;", "'":"&#39;" }[s]
  ));
}
const enc = encodeURIComponent;
const dec = decodeURIComponent;

/* =========================
   Globals
========================= */
let newProductModal;
window.productData = {};       // { [name]: { id, rrp_raw, category } }
window.selectedProducts = [];  // [{id, qty, name, rrp_raw, category}]
let searchIndex = [];          // [{name, lc, initials, tokens}]

/* Build initials: "Euro Shopper Milk Chocolate" -> "esmc" */
function makeInitials(name){
  return name
    .split(/[\s\-_/]+/)
    .filter(Boolean)
    .map(w => w[0])
    .join('')
    .toLowerCase();
}
function tokenize(name){
  return name.toLowerCase().split(/[\s\-_/]+/).filter(Boolean);
}

/* Match logic:
   - If query looks like initials (letters only, optionally spaced), match against initials startsWith
   - Else multi-word AND: every term must be present (substring) in lc
   - Also allow simple substring of whole query
*/
function matchProducts(query){
  const raw = query.trim().toLowerCase();
  if (!raw) return [];

  const initialsQuery = raw.replace(/\s+/g,'');             // "e s" -> "es"
  const isInitialsLike = /^[a-z]+$/.test(initialsQuery) && initialsQuery.length <= 8;

  const terms = raw.split(/\s+/).filter(Boolean);

  return searchIndex.filter(item => {
    if (isInitialsLike) {
      // prioritize initials startsWith
      if (item.initials.startsWith(initialsQuery)) return true;
      // soft: initials contains in order (subsequence)
      let i = 0;
      for (let c of item.initials){
        if (c === initialsQuery[i]) i++;
        if (i === initialsQuery.length) return true;
      }
    }

    // multi-word AND over name (substring)
    const allInName = terms.every(t => item.lc.includes(t));
    if (allInName) return true;

    // fallback: whole-query substring (already covered by allInName if single word)
    return false;
  });
}

/* =========================
   Boot
========================= */
document.addEventListener('DOMContentLoaded', () => {
  /* Alert modal */
  const alertModal = new bootstrap.Modal(document.getElementById('alertModal'));
  window.showAlert = msg => {
    document.getElementById('alertModalBody').innerHTML = msg;
    alertModal.show();
  };

  /* Hydrate productData from PHP */
  <?php foreach($products->fetch_all(MYSQLI_ASSOC) as $r): ?>
    productData[<?= json_encode($r['product_name']) ?>] = {
      id:       <?= (int)$r['product_id'] ?>,
      rrp_raw:  <?= json_encode($r['rrp']) ?>,
      category: <?= json_encode($r['category']) ?>
    };
  <?php endforeach; ?>

  /* Build search index once */
  searchIndex = Object.keys(productData).map(name => ({
    name,
    lc: name.toLowerCase(),
    initials: makeInitials(name),
    tokens: tokenize(name)
  }));

  /* Autocomplete UI (no inline onclicks; safe with quotes) */
  const inp   = document.getElementById('product_input');
  const box   = document.getElementById('suggestions');
  const price = document.getElementById('price');
  const cat   = document.getElementById('category');
  const qty   = document.getElementById('qty');

  inp.setAttribute('autocomplete','off');

  function renderSuggestions(list){
    if (!list.length){
      box.innerHTML = '<div class="text-muted px-2 py-1">No match</div>';
      return;
    }
    // limit to 200 for performance
    const html = list.slice(0,200).map(it =>
      `<div class="suggestion px-2 py-1" data-name="${enc(it.name)}">${escapeHTML(it.name)}</div>`
    ).join('');
    box.innerHTML = html;
  }

  inp.addEventListener('input', () => {
    const q = inp.value;
    if (!q.trim()){ box.innerHTML = ''; return; }
    renderSuggestions(matchProducts(q));
  });

  // Click selection
  box.addEventListener('click', (e) => {
    const el = e.target.closest('.suggestion');
    if (!el) return;
    const name = dec(el.dataset.name);
    selectProduct(name);
    box.innerHTML = '';
  });

  // Close suggestions on outside click / Esc
  document.addEventListener('click', (e) => {
    if (!box.contains(e.target) && e.target !== inp) box.innerHTML = '';
  });
  inp.addEventListener('keydown', (e) => {
    if (e.key === 'Escape') box.innerHTML = '';
  });

  /* Bootstrap modal */
  newProductModal = new bootstrap.Modal(document.getElementById('newProductModal'));

  /* New product AJAX */
  document.getElementById('newProductForm')
    .addEventListener('submit', handleNewProductSubmit);

  /* Grid delegation */
  const grid = document.getElementById('product_grid');

  // Remove buttons
  grid.addEventListener('click', (e) => {
    const btn = e.target.closest('.btn-remove');
    if (!btn) return;
    const idx = +btn.dataset.idx;
    if (Number.isInteger(idx)) {
      selectedProducts.splice(idx, 1);
      renderGrid();
    }
  });

  // Qty inline edit on dblclick
  grid.addEventListener('dblclick', (e) => {
    const cell = e.target.closest('td.qty-cell');
    if (!cell) return;
    const idx = +cell.dataset.idx;
    if (!Number.isInteger(idx)) return;

    const old = selectedProducts[idx].qty;
    cell.innerHTML = `<input type="number" min="1" value="${old}" style="width:70px;">`;
    const input = cell.querySelector('input');
    input.focus(); input.select();

    const finish = () => {
      const val = parseInt(input.value, 10);
      if (!isNaN(val) && val > 0) selectedProducts[idx].qty = val;
      renderGrid();
    };
    input.addEventListener('blur', finish);
    input.addEventListener('keydown', (ev) => {
      if (ev.key === 'Enter') finish();
      if (ev.key === 'Escape') renderGrid();
    });
  });
});

/* =========================
   Public functions
========================= */
window.selectProduct = function(name){
  const pd = productData[name];
  if (!pd) return;
  document.getElementById('product_input').value = name;
  document.getElementById('price').value = pd.rrp_raw ?? '';
  document.getElementById('category').value = pd.category ?? '';
  document.getElementById('suggestions').innerHTML = '';
};

window.addProduct = function(){
  const name = document.getElementById('product_input').value.trim();
  const qty  = parseInt(document.getElementById('qty').value, 10);
  if (!name || !qty || !productData[name]) {
    return showAlert('Please select a product and enter a valid quantity.');
  }
  selectedProducts.push({
    id:       productData[name].id,
    qty,
    name,
    rrp_raw:  productData[name].rrp_raw,
    category: productData[name].category
  });

  // clear inputs
  document.getElementById('product_input').value = '';
  document.getElementById('price').value = '';
  document.getElementById('category').value = '';
  document.getElementById('qty').value = '';

  renderGrid();
};

window.renderGrid = function(){
  const grid = document.getElementById('product_grid');
  grid.innerHTML = '';
  const groups = {};
  selectedProducts.forEach((p,i) => {
    (groups[p.category || 'Uncategorized'] = groups[p.category || 'Uncategorized'] || [])
      .push(Object.assign({_idx:i}, p));
  });

  for (const cat in groups) {
    grid.insertAdjacentHTML('beforeend',
      `<tr class="table-light"><td colspan="5"><strong>${escapeHTML(cat)}</strong></td></tr>`
    );
    grid.insertAdjacentHTML('beforeend',
      `<tr class="table-secondary text-dark">
         <th>Product</th><th>Price</th><th>Category</th><th>Qty</th><th></th>
       </tr>`
    );
    groups[cat].forEach(p => {
      grid.insertAdjacentHTML('beforeend',
        `<tr>
           <td>${escapeHTML(p.name)}</td>
           <td>${escapeHTML(String(p.rrp_raw ?? ''))}</td>
           <td>${escapeHTML(String(p.category ?? ''))}</td>
           <td class="qty-cell" data-idx="${p._idx}">${p.qty}</td>
           <td class="text-center">
             <button type="button" class="btn btn-sm btn-outline-danger btn-remove" data-idx="${p._idx}">&times;</button>
           </td>
         </tr>`
      );
    });
  }
};

window.removeProduct = function(i){
  // kept for compatibility
  if (!Number.isInteger(+i)) return;
  selectedProducts.splice(+i,1);
  renderGrid();
};

window.submitForm = function(){
  const storeId = document.getElementById('store_select').value;
  if (!storeId || !selectedProducts.length) {
    return showAlert('Please select a store and add at least one product.');
  }
  const form = document.createElement('form');
  form.method = 'POST';
  form.innerHTML = `<input type="hidden" name="store_id" value="${storeId}">` +
    selectedProducts.map((p,i) =>
      `<input type="hidden" name="products[${i}][id]" value="${p.id}">` +
      `<input type="hidden" name="products[${i}][qty]" value="${p.qty}">`
    ).join('');
  document.body.appendChild(form);
  form.submit();
};

window.newProduct = function(){
  document.getElementById('newProductForm').reset();
  newProductModal.show();
};

function handleNewProductSubmit(e){
  e.preventDefault();
  const form = e.target;
  const payload = {
    product_code: form.product_code.value,
    product_name: form.product_name.value,
    rrp:          form.rrp.value,
    category:     form.category.value
  };

  fetch('add_product.php', {
    method:  'POST',
    headers: { 'Content-Type': 'application/json' },
    body:    JSON.stringify(payload)
  })
  .then(r => {
    if (!r.ok) throw new Error('Server error: ' + r.status);
    return r.json();
  })
  .then(json => {
    if (json.success) {
      // Update productData + searchIndex live
      const name = json.product_name;
      productData[name] = {
        id:       json.product_id,
        rrp_raw:  json.rrp,
        category: json.category
      };
      searchIndex.push({
        name,
        lc: name.toLowerCase(),
        initials: makeInitials(name),
        tokens: tokenize(name)
      });
      newProductModal.hide();
      showAlert('Product added successfully.');
    } else {
      throw new Error(json.error || 'Unknown error');
    }
  })
  .catch(err => showAlert(escapeHTML(err.message)));
}
</script>


</body>
</html>
